// Handle.h - a reference-counting pointer interface.

//	Vamos Automotive Simulator
//  Copyright (C) 2002 Sam Varner
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef _HANDLE_H_
#define _HANDLE_H_

namespace Vamos_Geometry
{
  template <class T> class Handle
  {
  private:
	// The actual pointer.
	T* mp_rep;
	
	// A pointer to the reference count.  mp_count is a pointer so
	// that all references can modify the count.
	int* mp_count;

  public:
	Handle (T* rep = 0);
	Handle (const Handle& handle);
	~Handle ();
	
	Handle& operator = (const Handle& handle);
	T* operator -> () const { return mp_rep; }
	bool null () const { return mp_rep == 0; }
  };
}

template <class T> Vamos_Geometry::Handle <T>::
Handle (T* rep) : mp_rep (rep), mp_count (new int (1))
{
}

template <class T> Vamos_Geometry::Handle <T>::
Handle (const Handle& handle)
{
  // Copy the pointers and increment the reference count.
  mp_rep = handle.mp_rep;
  mp_count = handle.mp_count;
  (*mp_count)++;
}

template <class T> Vamos_Geometry::Handle <T>::
~Handle ()
{
  --(*mp_count);
  if (*mp_count == 0)
	{
	  delete mp_rep;
	  delete mp_count;
	}
}

template <class T> Vamos_Geometry::Handle <T>& Vamos_Geometry::Handle <T>::
operator = ( const Handle& handle)
{
  if (handle.mp_rep != mp_rep)
	{
	  // Assigning removes the current reference.
	  if (--(*mp_count) == 0)
		{
		  delete mp_rep;
		  delete mp_count;
		}
	  
	  // Copy the pointers and increment the reference count.
	  mp_rep = handle.mp_rep;
	  mp_count = handle.mp_count;
	  (*mp_count)++;
	}
  return *this;
}

#endif // not _HANDLE_H_
