/*  $Id: french-deconjugator.cpp,v 1.16 2010/04/18 20:10:35 sarrazip Exp $
    french-deconjugator.cpp - Analyzer of conjugated French verbs

    verbiste - French conjugation system
    Copyright (C) 2003-2009 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/


#include "Command.h"

#ifdef ENABLE_NLS
#ifdef HAVE_GETOPT_LONG
#include <unistd.h>
#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif
#include <getopt.h>
#endif
#endif  /* ENABLE_NLS */

#include <langinfo.h>

#include <iostream>
#include <vector>
#include <string.h>

using namespace std;
using namespace verbiste;


static const char *commandName = "french-deconjugator";


#if defined(ENABLE_NLS) && defined(HAVE_GETOPT_LONG)
static struct option knownOptions[] =
{
    { "help",          no_argument,       NULL, 'h' },
    { "version",       no_argument,       NULL, 'v' },
    { "utf8",          no_argument,       NULL, 'u' },
    { "all-infinitives",no_argument,      NULL, 'i' },

    { NULL, 0, NULL, 0 }  // marks the end
};
#endif


class DeconjugatorCommand : public Command
{
public:

    DeconjugatorCommand(bool utf8Mode) throw(logic_error)
      : Command(utf8Mode)
    {
    }

    virtual ~DeconjugatorCommand()
    {
    }

protected:

    virtual void processInputWord(const std::string &inputWord);

};


/*virtual*/
void
DeconjugatorCommand::processInputWord(const string &inputWord)
{
    // Analyze the word and get the results in a vector:
    vector<InflectionDesc> v;
    fvd->deconjugate(inputWord, v);


    // Display the results in Latin-1, with a terminating empty line:
    for (vector<InflectionDesc>::const_iterator it = v.begin();
					    it != v.end(); it++)
    {
	const InflectionDesc &d = *it;
	string inf = d.infinitive;

	if (utf8Mode)
	{
	    try
	    {
		inf = fvd->latin1ToUTF8(inf);
	    }
	    catch (int)
	    {
		// Conversion failed. Leave 'inf' as is.
	    }
	}

	cout 
	    << inf
	    << ", " << FrenchVerbDictionary::getModeName(d.mtpn.mode)
	    << ", " << FrenchVerbDictionary::getTenseName(d.mtpn.tense)
	    << ", " << int(d.mtpn.person)
	    << ", " << (d.mtpn.plural ? "plural" : "singular")
	    << "\n";
    }
    cout << endl;
}


static
void
displayVersionNo()
{
    cout << commandName << ' ' << VERSION << '\n';
}


static
void
displayHelp()
{
    cout << '\n';

    displayVersionNo();

    cout << "Part of " << PACKAGE << " " << VERSION << "\n";

    cout <<
"\n"
"Copyright (C) 2003-2009 Pierre Sarrazin <http://sarrazip.com/>\n"
"This program is free software; you may redistribute it under the terms of\n"
"the GNU General Public License.  This program has absolutely no warranty.\n"
    ;

    cout <<
"\n"
"Known options:\n"
"--help             Display this help page and exit\n"
"--version          Display this program's version number and exit\n"
"--utf8             Assume that the terminal uses UTF-8 instead of Latin-1\n"
"--all-infinitives  Print the names of all known verbs, one per line (unsorted)\n"
"\n"
"See the " << commandName << "(1) manual page for details.\n"
"\n"
    ;
}


int
main(int argc, char *argv[])
{
    setlocale(LC_CTYPE, "");
    bool utf8Mode = (strcmp(nl_langinfo(CODESET), "UTF-8") == 0);
    bool listAllInfinitives = false;

    #if defined(ENABLE_NLS) && defined(HAVE_GETOPT_LONG)

    /*  Interpret the command-line options:
    */
    int c;
    do
    {
	c = getopt_long(argc, argv, "hv", knownOptions, NULL);

	switch (c)
	{
	    case EOF:
		break;  // nothing to do

	    case 'v':
		displayVersionNo();
		return EXIT_SUCCESS;

	    case 'h':
		displayHelp();
		return EXIT_SUCCESS;

	    case 'u':
	    	utf8Mode = true;
		break;

	    case 'i':
		listAllInfinitives = true;
		break;

	    default:
		displayHelp();
		return EXIT_FAILURE;
	}
    } while (c != EOF && c != '?');

    #else

    int optind = 1;

    #endif

    DeconjugatorCommand cmd(utf8Mode);

    if (listAllInfinitives)
	return cmd.listAllInfinitives(cout);

    return cmd.run(argc - optind, argv + optind);
}
