#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include <unistd.h>
#include <sys/types.h>

#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/disk.h"
#include "../include/prochandle.h"

#include "guiutils.h"
#include "cdialog.h"
#include "fb.h"

#include "vma.h"
#include "vmahelp.h"
#include "config.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


int VMAHelpFile(
	vma_core_struct *core_ptr, const char *help_file, gbool verbose
);


/* Help program argument types, specifies the type of argument it takes
 * (usually a file path or url).
 */
#define HELP_PROG_INPUT_TYPE_PATH	0
#define HELP_PROG_INPUT_TYPE_URL	1


/* Location of help programs, their respective arguments, and argument 
 * type in triplets.  The last three in the list should be NULL.
 */
#define HELP_PROG_LOCATIONS	\
{ \
 "/usr/bin/gnome-help-browser", "", (char *)HELP_PROG_INPUT_TYPE_URL, \
 "/usr/local/bin/gnome-help-browser", "", (char *)HELP_PROG_INPUT_TYPE_URL, \
 "/bin/gnome-help-browser", "", (char *)HELP_PROG_INPUT_TYPE_URL, \
\
 "/usr/bin/netscape", "", (char *)HELP_PROG_INPUT_TYPE_URL, \
 "/usr/local/bin/netscape", "", (char *)HELP_PROG_INPUT_TYPE_URL, \
 "/bin/netscape", "", (char *)HELP_PROG_INPUT_TYPE_URL, \
\
 "/usr/X11R6/bin/nxterm", "-e lynx", (char *)HELP_PROG_INPUT_TYPE_URL, \
 "/usr/X11R6/bin/xterm", "-e lynx", (char *)HELP_PROG_INPUT_TYPE_URL, \
\
 NULL, NULL, NULL \
}



/*
 *	Runs the help program and instructs it to load the file 
 *	specified by help_file.
 *
 *	The given help_file must be not NULL and an absolute path,
 *	if it is not an absolute path then global dname.data_global will
 *	be prefixed to it.
 *
 *	Returns non-zero on error.
 */
int VMAHelpFile(
        vma_core_struct *core_ptr, const char *help_file, gbool verbose  
)
{
	int i;
	pid_t p;
	char *cmd;
	const char	*help_prog = NULL, *help_args = NULL,
			*help_prog_loc[] = HELP_PROG_LOCATIONS;
	int help_input_type = HELP_PROG_INPUT_TYPE_PATH;
	char new_path[PATH_MAX + NAME_MAX];


	if((core_ptr == NULL) || (help_file == NULL))
	{
	    if(verbose)
	    {
	        CDialogSetTransientFor(NULL);
	        CDialogGetResponse(
"Invalid help resource!",
"Help resource identifier is NULL!",
"The specification to the object containing the help\n\
data is NULL, this is probably an internal error.\n\
Please report this experiance to the vendor.",
		    CDIALOG_ICON_ERROR,
		    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		    CDIALOG_BTNFLAG_OK
		);
	    }
	    return(-1);
	}
	if(ISPATHABSOLUTE(help_file))
	{
	    strncpy(new_path, help_file, PATH_MAX + NAME_MAX);
	}
	else
	{
	    const char *cstrptr = PrefixPaths(
		dname.data_global, help_file
	    );
	    if(cstrptr != NULL)
		strncpy(new_path, cstrptr, PATH_MAX + NAME_MAX);
	    else
		(*new_path) = '\0';
	}
	new_path[PATH_MAX + NAME_MAX - 1] = '\0';
	if((*new_path) == '\0')
	    return(-1);


	/* Check if help viewer exists. */
	for(i = 0; 1; i += 3)
	{
	    help_prog = help_prog_loc[i];
	    help_args = help_prog_loc[i + 1];
	    help_input_type = (int)help_prog_loc[i + 2];
	    if(help_prog == NULL)
		break;

	    /* Help program executeable? */
	    if(!access(help_prog, X_OK))
		break;
	}
	if(help_prog == NULL)
	{
	    char *buf = NULL;

	    buf = strcatalloc(buf,
 "Could not find help viewer in the following locations:\n\n"
	    );
            for(i = 0; 1; i += 3)
            {
                help_prog = help_prog_loc[i];
                if(help_prog == NULL)
                    break;

                buf = strcatalloc(buf, help_prog);
		buf = strcatalloc(buf, "\n");
            }

	    if(verbose)
	    {
                CDialogSetTransientFor(NULL);
                CDialogGetResponse(
"Cannot find help viewer!",
		    buf,
"The program used to view help files could not be\n\
found or was not installed on your system. This\n\
program uses standard UNIX manual pages as documentation,\n\
make sure that you have a manual page viewer installed.",
		    CDIALOG_ICON_ERROR,
		    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		    CDIALOG_BTNFLAG_OK
		);
	    }

	    free(buf);

	    return(-2);
	}

	/* Got existing help_prog and matching help_args. */

	/* Format command string. */
	cmd = NULL;
	cmd = strcatalloc(cmd, help_prog);
	if(help_args != NULL)
	{
	    cmd = strcatalloc(cmd, " ");
	    cmd = strcatalloc(cmd, help_args);
	}
	switch(help_input_type)
	{
	  case HELP_PROG_INPUT_TYPE_URL:
	    cmd = strcatalloc(cmd, " file://");
	    cmd = strcatalloc(cmd, new_path);
	    break;

	  default:	/* HELP_PROG_INPUT_TYPE_PATH */
	    cmd = strcatalloc(cmd, " ");
	    cmd = strcatalloc(cmd, new_path);
	    break;
	}

	/* Execute command. */
	p = Exec(cmd);

	/* Error executing command? */
	if(verbose && (p == 0))
	{
	    char *buf = NULL;

	    buf = strcatalloc(buf, "Error executing command:\n\n");
	    buf = strcatalloc(buf, cmd);

	    CDialogSetTransientFor(NULL);
	    CDialogGetResponse(
"Cannot execute help viewer!",
		buf,
"There was an error running the help viewer to\n\
view the requested help resource. It is possible\n\
that the help viewer program does not exist or\n\
the resource does not exist, verify that the\n\
help viewer program exists.",
                CDIALOG_ICON_ERROR,
                CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
                CDIALOG_BTNFLAG_OK
            );

	    free(buf);

	    /* Free command string. */
            free(cmd);
	    cmd = NULL;

	    return(-1);
	}

	/* Free command string. */
	free(cmd);
	cmd = NULL;

	return(0);
}
