
// created 06.2003 by Stefan Kleine Stegemann
// 
// licensed under GPL


#include "AppController.h"
#include "DocumentWindowController.h"
#include "DocumentWindow.h"
#include "ViewPDFDocument.h"
#include "ExtendedImageView.h"
#include "DocumentPosition.h"
#include "ViewPDF.h"

#include <PDFKit/PDFDocument.h>

#include <Foundation/NSBundle.h>
#include <Foundation/NSDictionary.h>
#include <Foundation/NSException.h>
#include <Foundation/NSNotification.h>
#include <Foundation/NSString.h>
#include <Foundation/NSValue.h>
#include <Foundation/NSUserDefaults.h>
#include <AppKit/NSApplication.h>
#include <AppKit/NSClipView.h>
#include <AppKit/NSImage.h>
#include <AppKit/NSMenu.h>
#include <AppKit/NSMenuItem.h>
#include <AppKit/NSProgressIndicator.h>
#include <AppKit/NSScreen.h>
#include <AppKit/NSScroller.h>
#include <AppKit/NSTextField.h>
#include <AppKit/NSWindow.h>
#include <AppKit/NSPasteboard.h>
#include <AppKit/NSAttributedString.h>

/*
 * Non-Public methods of DocumentWindowController
 */
@interface DocumentWindowController(Private)
- (void) _initWindow;
- (void) _scrollToTopOfPage;
- (void) _scrollToBottomOfPage;
- (void) _scrollToPageBorder: (int)direction;
- (void) _scrollVertical: (id)sender 
                byAmount: (float)scrollAmount
             inDirection: (int)direction
               flipAtEnd: (BOOL)flip;
- (void) _notifySelectionChanged: (id)notification;
- (void) _notifySelectionRemoved: (id)notification;
@end

/*
 * An instance of DocumentController controlls an
 * DocumentWindow that displays a ViewableDocument. 
 */
@implementation DocumentWindowController

- (id) initWithWindow: (NSWindow*)win
{
   if (self = [super initWithWindow: win])
   {
      [self setShouldCascadeWindows: NO];
      [(DocumentWindow*)win setController: self];
      [self setScaleFactor: 1.0];
      documentIsDisplayed = NO;
      findPanel = nil;
      pageLayoutPanel = nil;
      outlinePanel = nil;
      bookmarks = [[NSMutableArray alloc] initWithCapacity: 0];
      searchText = nil;
      matchesTable = nil;
      matches = nil;
      matchesLock = [[NSLock alloc] init];
      findInProgressLock = [[NSLock alloc] init];
      findInProgress = NO;
      currentMatch = -1;
      currentPage = 0;
      contentSelection = nil;
      lastSearchText = nil;
      outlineDataSource = nil;
   }

   return self;
}


- (void) dealloc
{
   NSLog(@"dealloc DocumentWindowController, retain count is %d", [self retainCount]);
   
   [[NSNotificationCenter defaultCenter] removeObserver: self];
   [self setContentSelection: nil];
   [lastSearchText release];
   [findPanel release];
   [pageLayoutPanel release];
   [outlinePanel release];
   [matches release];
   [matchesLock release];

   [super dealloc];
}


/*
 * Convinience method that returns the PDFDocument
 * of the controllers document.
 */
 - (PDFDocument*) pdfDocument
{
   return [(ViewPDFDocument*)[self document] pdfDocument];
}


/*
 * Convinience method to obtain the DocumentView of
 * the associated window that displays the PDFDocument.
 */
- (PDFContentView*) pdfContentView
{
   return [(DocumentWindow*)[self window] documentView];
}


/*
 * Convinience method to set the current content selection.
 */
- (void) setContentSelection: (DocumentPosition*)newSelection
{
   [contentSelection release];
   contentSelection = [newSelection retain];
}


/*
 * Returns the current content selection, nil if nothing
 * is selected.
 */
- (DocumentPosition*) contentSelection
{
   return contentSelection;
}


/*
 * Set the number of the page to be displayed.
 */
- (void) setCurrentPage: (int)pageNum
{
   id        win;
   NSString* pageFieldText;

   NSAssert(pageNum > 0, @"new current Page <= 0");
   NSAssert([self window] != nil, @"no window");
   
   currentPage = pageNum;
   [self updatePageImage];

   pageFieldText = [[NSString alloc] initWithFormat: @"%d of %d",
                                     [self currentPage],
                                     [[self pdfDocument] countPages]];
   [pageFieldText autorelease];

   [[[[self pdfContentView] scrollView] pageField] setStringValue: pageFieldText];
}


/*
 * Get the numer of the page that is currently displayed.
 */
- (int) currentPage
{
   return currentPage;
}


/*
 * Set the scale factor (zoom).
 */
- (void) setScaleFactor: (double)factor
{
   scaleFactor = factor;
   NSLog(@"new scale factor is %f", scaleFactor);
}


/*
 * Get the current scale factor.
 */
- (double) scaleFactor
{
   return scaleFactor;
}


/*
 * Refreshes the currently displayed image from the
 * the current page. This is usefull for example when 
 * the image has been scaled and the view should 
 * display the new, scaled image.
 */
- (void) updatePageImage
{
   NSAssert([self window] != nil, @"no window");
   NSAssert([self currentPage] > 0, @"invalid page (<= 0)");

   [[self pdfContentView] setPage: [self currentPage]
                       scaledBy: [self scaleFactor]];

   // show selection?
   if ([self contentSelection])
   {
      if ([self currentPage] == [[self contentSelection] page])
      {
         NSRect selectionRect = [[self contentSelection] boundingRect];
         NSRect scaledSelection =
            NSMakeRect(selectionRect.origin.x * [self scaleFactor],
                       selectionRect.origin.y * [self scaleFactor],
                       selectionRect.size.width * [self scaleFactor],
                       selectionRect.size.height * [self scaleFactor]);
         
         [[[self pdfContentView] imageView] setSelection: scaledSelection];
      }
      else
      {
         [[[self pdfContentView] imageView] removeSelection];
      }
   }

   [[self pdfContentView] updateView];
}


/*
 * Resize window to show as much of the image as
 * possible. This is the natural size of the image 
 * but limited to the screen frame. After the window
 * has been resized, the image is scaled to fit in
 * the window.
 */
- (void) sizePageToMax
{
   NSSize  pageSize;
   NSSize  winSize;
   NSRect  screenFrame;
   id      win;
   float   maxAllowedHeight;
   float   maxAllowedWidth;

   NSAssert([self window] != nil, @"no window");
   NSAssert([self currentPage] > 0, @"invalid page (<= 0)");

   win = [self window];
   
   pageSize = [[self pdfDocument] pageSize: [self currentPage] considerRotation: YES];
   screenFrame = [[NSScreen mainScreen] visibleFrame];

   maxAllowedHeight = 
        screenFrame.size.height
      - 50
      - [NSScroller scrollerWidthForControlSize: NSRegularControlSize];
   
   if ([(DocumentWindow*)[self window] toolbarIsVisible])
   {
      maxAllowedHeight -= TOOLBAR_HEIGHT;
   }

   maxAllowedWidth =
        screenFrame.size.width
      - 50
      - [NSScroller scrollerWidthForControlSize: NSRegularControlSize];
      
   if (pageSize.height > maxAllowedHeight)
   {
      pageSize.height = maxAllowedHeight;
   }

   if (pageSize.width > maxAllowedWidth)
   {
      pageSize.width = maxAllowedWidth;
   }

   winSize = [self scalePageToFitIn: pageSize];

   winSize.height =
        winSize.height
      + [NSScroller scrollerWidthForControlSize: NSRegularControlSize]
      + 2;

   if ([(DocumentWindow*)[self window] toolbarIsVisible])
   {
      winSize.height += TOOLBAR_HEIGHT;
   }
      
   winSize.width =
        winSize.width
      + [NSScroller scrollerWidthForControlSize: NSRegularControlSize]
      + 2;

   [self updatePageImage];
   [win setContentSize: winSize];
}


/*
 * Copy the selected content to the pasteboard (currently only
 * text content).
 */
- (void) copy: (id)sender
{
   NSString* text;

   NSAssert([self contentSelection], @"no selection");

   text = [[self pdfDocument] getTextAtPage: [[self contentSelection] page]
                              inRect: [[self contentSelection] boundingRect]];
   if (text)
   {
      NSPasteboard* pboard = [NSPasteboard generalPasteboard];

      [pboard declareTypes: [NSArray arrayWithObject: NSStringPboardType] owner: nil];
      [pboard setString: text forType: NSStringPboardType];
   }
   else
   {
      NSRunAlertPanel(@"Content selection error",
                      @"The selected content could not be accessed.",
                      @"Continue",
                      nil,
                      nil);
   }
}


/*
 * Go the first page of the document.
 */
- (void) firstPage: (id)sender
{
   NSAssert([self document] != nil, @"no document");

   [self setCurrentPage: 1];
   [self _scrollToTopOfPage];
}


/*
 * Go to the previous page, if available.
 */
- (void) previousPage: (id)sender
{
   int newPageNum;

   NSAssert([self document] != nil, @"no document");
   NSAssert([self currentPage] > 0, @"invalid page (<= 0)");

   newPageNum = [self currentPage] - 1;
   if (newPageNum > 0)
   {
      NSLog(@"current page is %d, new page is %d",
            [self currentPage], newPageNum);

      [self setCurrentPage: newPageNum];
      [self _scrollToTopOfPage];
   }
}


/*
 * Go to the next page, if available.
 */
- (void) nextPage: (id)sender
{
   int newPageNum;

   NSAssert([self document] != nil, @"no document");
   NSAssert([self currentPage] > 0, @"no page (<= 0)");

   newPageNum = [self currentPage] + 1;
   if (newPageNum <= [[self pdfDocument] countPages])
   {
      NSLog(@"current page is %d, new page is %d",
            [self currentPage], newPageNum);

      [self setCurrentPage: newPageNum];
      [self _scrollToTopOfPage];
   }
}


/*
 * Scroll up or down by a page. The direction is determined by
 * the direction parameter. Use either SCROLL_UP or SCROLL_DOWN.
 * If the beginning/end of the page has already been reached, the 
 * previous/next page will be displayed.
 */
- (void) scrollPage: (id)sender direction: (int)direction
{
   id     scrollView = [[self pdfContentView] scrollView];
   NSRect vRect      = [[scrollView documentView] visibleRect];

   [self _scrollVertical: sender
         byAmount: vRect.size.height - [scrollView verticalPageScroll]
         inDirection: direction
         flipAtEnd: YES];
}


/*
 * Scroll vertical by 1 line. The direction is determined by the
 * direction parameter (use SCROLL_UP or SCROLL_DOWN). If the end
 * of the page is reached, nothing is done.
 */
- (void) scrollVertical: (id)sender direction: (int)direction
{
   id scrollView = [[self pdfContentView] scrollView];

   [self _scrollVertical: sender
         byAmount: [scrollView verticalLineScroll]
         inDirection: direction
         flipAtEnd: NO];
}


/*
 * Scroll horizontal by 1 line. The direction is determined by the
 * direction parameter (use SCROLL_LEFT or SCROLL_RIGHT).
 */
- (void) scrollHorizontal: (id)sender direction: (int)direction
{
   NSPoint newOrigin;
   NSRect  vRect;
   float   scrollAmount;
   id      scrollView = [[self pdfContentView] scrollView];

   NSAssert([self currentPage] > 0, @"invalid page (<= 0)");
   NSAssert([self window] != nil, @"no window");

   vRect = [[scrollView documentView] visibleRect];
   scrollAmount = [scrollView verticalLineScroll];

   newOrigin = NSMakePoint(vRect.origin.x + (scrollAmount * direction),
                           vRect.origin.y);

   [[scrollView contentView] scrollToPoint: newOrigin];
}


/*
 * Go the first page of the document.
 */
- (void) lastPage: (id)sender
{
   NSAssert([self document] != nil, @"no document");

   [self setCurrentPage: [[self pdfDocument] countPages]];
   [self _scrollToTopOfPage];
}


/*
 * Display the page in bigger size.
 */
- (void) zoomIn: (id)sender
{
   [self setScaleFactor: [self scaleFactor] + ZOOM_STEP];
   [self updatePageImage];
}


/*
 * Display the page in smaller size.
 */
- (void) zoomOut: (id)sender
{
   float newScaleFactor;

   newScaleFactor = [self scaleFactor] - ZOOM_STEP;

   if (newScaleFactor > 0.1)
   {
      [self setScaleFactor: newScaleFactor];
      [self updatePageImage];
   }
}


/*
 * Display the page in it's original size.
 */
- (void) zoomToRealSize: (id)sender
{
   [self setScaleFactor: 1.0];
   [self updatePageImage];
}


/*
 * Rescale the page so that it fits completly
 * in the window.
 */
- (void) sizePageToFit: (id)sender
{
   id     win;
   NSSize size;

   NSAssert([self window] != nil, @"no window");

   //   size = NSMakeSize([[[win imageView] scrollView] contentSize].width,
   //                  [[[win imageView] scrollView] contentSize].height);
   size = [[[self pdfContentView] scrollView] contentSize];

   [self scalePageToFitIn: size];
   
   [self updatePageImage];
}


/*
 * Rescale the page so that it fits in the
 * window horizontally.
 */
- (void) sizePageToFitWidth: (id)sender
{
   id      win;
   NSSize  size;
   NSSize  newSize;
   double  oldScale;

   NSAssert([self window] != nil, @"no window");

   win = [self window];
   size = NSMakeSize([[[self pdfContentView] scrollView] contentSize].width, 0);

   oldScale = [self scaleFactor];
   newSize = [self scalePageToFitIn: size];
   
   [self updatePageImage];
}


/*
 * Create a bookmark at the current document position
 * (page + scroll position) and add this bookmark to the
 * top of the bookmark-stack.
 */
- (void) setBookmark: (id)sender
{
   NSNumber* pageNumber =
      [NSNumber numberWithInt: [self currentPage]];

   NSRect vRect = 
      [[[[self pdfContentView] scrollView] contentView] visibleRect];

   NSNumber* x = [NSNumber numberWithFloat: (vRect.origin.x / [self scaleFactor])];
   NSNumber* y = [NSNumber numberWithFloat: (vRect.origin.y / [self scaleFactor])];

   NSArray* keys = [NSArray arrayWithObjects: @"PageNumber", @"X", @"Y", nil];
   NSArray* values = [NSArray arrayWithObjects: pageNumber, x, y, nil];
   
   NSDictionary* bookmark =
      [NSDictionary dictionaryWithObjects: values forKeys: keys];

   [bookmarks addObject: bookmark];
}


/*
 * Jump to the topmost bookmark on the bookmark stack.
 * The bookmark is then removed from the stack. If
 * the stack is empty, nothing is done.
 */
- (void) gotoLastBookmark: (id)sender
{
   NSDictionary* bookmark = [bookmarks lastObject];

   if (bookmark)
   {
      NSPoint point;

      int   newPage = [[bookmark objectForKey: @"PageNumber"] intValue];
      float x       = [[bookmark objectForKey: @"X"] floatValue];
      float y       = [[bookmark objectForKey: @"Y"] floatValue];

      [self setCurrentPage: newPage];

      point  = NSMakePoint(x * [self scaleFactor],
                           y * [self scaleFactor]);

      [[[[self pdfContentView] scrollView] contentView] scrollToPoint: point];

      [bookmarks removeLastObject];
   }
}


/*
 * Toggle the visibility of the document windows toolbar.
 */
- (void) toggleToolbarVisible: (id)sender
{
   DocumentWindow* win = (DocumentWindow*)[self window];
   NSUserDefaults* defs = [NSUserDefaults standardUserDefaults];

   [win setToolbarVisible: ![win toolbarIsVisible]];
   [[NSUserDefaults standardUserDefaults] setBool: [win toolbarIsVisible]
                                          forKey: PREFS_TOOLBAR_VISIBLE];
}


/*
 * Get notified when the user changed the value
 * in the page field. When the user entered a valid
 * page number, this page is set as current page.
 */
- (void) controlTextDidEndEditing: (NSNotification*)aNotification
{
   int requestedPage;

   requestedPage = [[[[self pdfContentView] scrollView] pageField] intValue];
   if ((requestedPage > 0) && (requestedPage <= [[self pdfDocument] countPages]))
   {
      [self setCurrentPage: requestedPage];
   }
}


/*
 * The Items of the "Papersizes" menu are only enabled if the
 * papersize of the document can be set.
 */
- (BOOL) validateMenuItem: (id)menuItem
{
   BOOL itemEnabled = YES;

   if ([[menuItem title] isEqualToString: @"go to last Bookmark"])
   {
      itemEnabled = [bookmarks count] > 0;
   }
   else if ([[menuItem title] isEqualToString: @"Document Outline"])
   {
      itemEnabled = [[self pdfDocument] hasOutline];
   }
   else if ([[menuItem title] isEqualToString: @"Copy"])
   {
      itemEnabled = [self contentSelection] != nil;
   }

   return itemEnabled;
}


/*
 * Update the scale factor so that the image will fit into
 * a rectangle of the specified size. The new (scaled) size
 * is returned. Note that scaling is not performed until
 * the image is updated (updatePageImage).
 */
- (NSSize) scalePageToFitIn: (NSSize)size
{
   NSSize pageSize;
   NSSize newSize;
   double scaleFactorX;
   double scaleFactorY;
   double newScaleFactor;

   NSAssert([self currentPage] > 0, @"invalid page (<= 0)");

   // this is the the real (unscaled) page size 
   pageSize = [[self pdfDocument] pageSize: [self currentPage] considerRotation: YES];

   scaleFactorX = (size.width > 0 ? (size.width / pageSize.width) : 1000.0);
   scaleFactorY = (size.height > 0 ? (size.height / pageSize.height) : 1000.0);
   
   newScaleFactor = (scaleFactorX < scaleFactorY ? scaleFactorX : scaleFactorY);

   newSize = NSMakeSize(pageSize.width * newScaleFactor,
                        pageSize.height * newScaleFactor);

   [self setScaleFactor: newScaleFactor];

   return newSize;
}


- (void) windowDidBecomeMain: (NSNotification*)aNotification
{
   if (([self document] != nil) && (!documentIsDisplayed))
   {
      documentIsDisplayed = YES;
      [self _initWindow];
      [[self window] center];
   }
}


- (void) windowWillClose: (NSNotification*)aNotification
{
   NSLog(@"DocumentWindow closed");
   NSLog(@"Retain count of DocumentWindowController is %d", [self retainCount]);
   NSLog(@"Retain count of owned Document is %d", [[self document] retainCount]);

   if (pageLayoutPanel)
   {
      [pageLayoutPanel close];
   }

   [self cancelSearch: nil]; /* abort a possibly running serach */
   if (findPanel)
   {
      [findPanel close];
   }

   if (outlinePanel)
   {
      [outlinePanel close];
      [outlineDataSource release];
   }

   // due to a GNUstep bug we have to autorlease the controller here
   // (see NSWindowController.m)
   [[self window] autorelease];
}

@end


/* -------------------------------------------------------------------------- */

/*
 * Private aspects of DocumentWindowController.
 */
@implementation DocumentWindowController (Private)

/*
 * Updates the managed DocumentWindow to display the contents
 * of the controllers document. When invoked, a document
 * and a window must have been associated with this controller.
 */
- (void) _initWindow
{
   DocumentWindow* win = (DocumentWindow*)[self window];

   NSAssert([self window] != nil, @"no window");
   NSAssert([self document] != nil, @"no document");
   NSAssert([[self pdfDocument] countPages] > 0, @"no pages");

   // tell the content view what to display
   [[self pdfContentView] setDocument: [self pdfDocument]];

   currentPaperSizeName   = DEFAULT_PAPERSIZE;
   currentPaperSize       = NSMakeSize(0, 0);
   currentPageOrientation = TAG_PORTRAIT;
   
   // display the document
   [self setCurrentPage: 1];

   if ([[self pdfDocument] countPages] > 1)
   {
      [win setDisplayPageNavigation: YES];
   }
   else
   {
      [win setDisplayPageNavigation: NO];
   }
   
   [self sizePageToMax];

   // get informed about selection changes
   [[NSNotificationCenter defaultCenter] addObserver: self
                                         selector: @selector(_notifySelectionChanged:)
                                         name: N_SelectionChanged
                                         object: nil];
   
   [[NSNotificationCenter defaultCenter] addObserver: self
                                         selector: @selector(_notifySelectionRemoved:)
                                         name: N_SelectionRemoved
                                         object: nil];

   // tell the window that it the controller is now ready
   // to work with the window.
   [(DocumentWindow*)[self window] windowInitializedWithController];
}


/*
 * Used by the various vertical scrolling methods. Scrolls 
 */
- (void) _scrollVertical: (id)sender 
                byAmount: (float)scrollAmount
             inDirection: (int)direction
               flipAtEnd: (BOOL)flip
{
   NSPoint newOrigin;
   NSRect  vRect;
   id      scrollView = [[self pdfContentView] scrollView];

   NSAssert([self currentPage] > 0, @"invalid page (<= 0)");
   NSAssert([self window] != nil, @"no window");

   vRect = [[scrollView documentView] visibleRect];

   newOrigin = NSMakePoint(vRect.origin.x,
                           vRect.origin.y + (scrollAmount * direction));

   [[scrollView contentView] scrollToPoint: newOrigin];
   
   if (([[scrollView contentView] visibleRect].origin.y == vRect.origin.y) && flip)
   {
      if (direction == SCROLL_UP)
      {
         if ([self currentPage] > 1)
         {
            [self previousPage: sender];
            [self _scrollToBottomOfPage];
         }
      }
      else
      {
         [self nextPage: sender];
      }
   }
}


/*
 * Scroll to the top of the current page.
 */
- (void) _scrollToTopOfPage
{
   [self _scrollToPageBorder: SCROLL_UP];
}


/*
 * Scroll to the bottom of the current page.
 */
- (void) _scrollToBottomOfPage
{
   [self _scrollToPageBorder: SCROLL_DOWN];
}


/*
 * Scrolls to the top or bottom of the current page. This method
 * is used by _scrollToTopOfPage/_scrollToBottomOfPage.
 */
- (void) _scrollToPageBorder: (int)direction;
{
   id      win = [self window];
   NSPoint topOfPage;
   NSRect  vRect;
   float   newY;

   NSAssert([self currentPage] > 0, @"invalid page (<= 0)");
   NSAssert([self window] != nil, @"no window");

   vRect = [[[[self pdfContentView] scrollView] documentView] visibleRect];

   if (direction == SCROLL_UP)
   {
      newY = [[[[self pdfContentView] scrollView] documentView] frame].size.height;
   }
   else
   {
      newY = 0;
   }

   topOfPage = NSMakePoint(vRect.origin.x, newY);

   [[[[self pdfContentView] scrollView] contentView] scrollToPoint: topOfPage];
}


/*
 * Invoked whenever the content selection changed.
 */
- (void) _notifySelectionChanged: (id)notification
{
   NSRect selRect = 
      [[[notification userInfo] objectForKey: UserInfoKeySelection] rectValue];

   // we need the unscaled selection
   selRect = NSMakeRect(selRect.origin.x / [self scaleFactor],
                        selRect.origin.y / [self scaleFactor],
                        selRect.size.width / [self scaleFactor],
                        selRect.size.height / [self scaleFactor]);

   DocumentPosition* newSelection =
      [DocumentPosition positionAtPage: [self currentPage]
                        boundingRect: selRect];

   [self setContentSelection: newSelection];
}


/*
 * Invoked whenever the content selection is cleared.
 */
- (void) _notifySelectionRemoved: (id)notification
{
   [self setContentSelection: nil];
}


@end
