/* Convert 1 or 3-band 8-bit VIPS images to/from JPEG.
 *
 * 28/11/03 JC
 *	- better no-overshoot on tile loop
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/
#include <vips/intl.h>

#ifndef HAVE_JPEG

#include <vips/vips.h>

int
im_vips2jpeg( IMAGE *in, const char *filename )
{
	im_errormsg( "im_vips2jpeg: JPEG support disabled" );

	return( -1 );
}

int
im_vips2bufjpeg( IMAGE *in, IMAGE *out, int qfac, char **obuf, int *olen )
{
	im_errormsg( "im_vips2bufjpeg: JPEG support disabled" );

	return( -1 );
}

int
im_vips2mimejpeg( IMAGE *in, int qfac )
{
	im_errormsg( "im_vips2mimejpeg: JPEG support disabled" );

	return( -1 );
}

int
im_jpeg2vips_header( const char *name, IMAGE *out )
{
	im_errormsg( "im_jpeg2vips_header: JPEG support disabled" );
	return( -1 );
}

int
im_jpeg2vips( const char *name, IMAGE *out )
{
	im_errormsg( "im_jpeg2vips: JPEG support disabled" );

	return( -1 );
}

#else /*HAVE_JPEG*/

/*
#define DEBUG
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <setjmp.h>

#include <vips/vips.h>

/* jpeglib includes jconfig.h, which can define HAVE_STDLIB_H ... which we
 * also define. Make sure it's turned off.
 */
#ifdef HAVE_STDLIB_H
#undef HAVE_STDLIB_H
#endif /*HAVE_STDLIB_H*/

#include <jpeglib.h>
#include <jerror.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* Define a new error handler for when we bomb out.
 */
typedef struct {
	/* Public fields.
	 */
	struct jpeg_error_mgr pub;

	/* Private stuff for us.
	 */
	jmp_buf jmp;		/* longjmp() here to get back to VIPS */
	FILE *fp;		/* fclose() if non-NULL */
} ErrorManager;

/* New output message method - send to VIPS.
 */
METHODDEF(void)
new_output_message( j_common_ptr cinfo )
{
	char buffer[JMSG_LENGTH_MAX];

	(*cinfo->err->format_message)( cinfo, buffer );
	im_errormsg( "JPEG error: %s", buffer );

#ifdef DEBUG
	printf( "vips_jpeg.c: new_output_message: \"%s\"\n", buffer );
#endif /*DEBUG*/
}

/* New error_exit handler.
 */
METHODDEF(void)
new_error_exit( j_common_ptr cinfo )
{
	ErrorManager *eman = (ErrorManager *) cinfo->err;

#ifdef DEBUG
	printf( "vips_jpeg.c: new_error_exit\n" );
#endif /*DEBUG*/

	/* Close the fp if necessary.
	 */
	if( eman->fp ) {
		(void) fclose( eman->fp );
		eman->fp = NULL;
	}

	/* Send the error message to VIPS. This method is overridden above.
	 */
	(*cinfo->err->output_message)( cinfo );

	/* Jump back.
	 */
	longjmp( eman->jmp, 1 );
}

/* What we track during a JPEG write.
 */
typedef struct {
	IMAGE *in;
	int tile_width, tile_height;
	struct jpeg_compress_struct cinfo;
        ErrorManager eman;
	REGION *reg;
	im_threadgroup_t *tg;
	JSAMPROW *row_pointer;
	char *profile_bytes;
	int profile_length;
} Write;

static void
write_destroy( Write *write )
{
	jpeg_destroy_compress( &write->cinfo );
	IM_FREEF( im_threadgroup_free, write->tg );
	IM_FREEF( im_region_free, write->reg );
	IM_FREEF( fclose, write->eman.fp );
	IM_FREE( write->row_pointer );
	IM_FREE( write->profile_bytes );
	im_free( write );
}

static Write *
write_new( IMAGE *in )
{
	Write *write;

	if( !(write = IM_NEW( NULL, Write )) )
		return( NULL );

	im__find_demand_size( in, &write->tile_width, &write->tile_height );

	write->in = in;
	write->reg = im_region_create( write->in );
	write->tg = im_threadgroup_create( write->in );
	write->row_pointer = IM_ARRAY( NULL, write->tile_height, JSAMPROW );

        write->cinfo.err = jpeg_std_error( &write->eman.pub );
	write->eman.pub.error_exit = new_error_exit;
	write->eman.pub.output_message = new_output_message;
	write->eman.fp = NULL;
	write->profile_bytes = NULL;
	write->profile_length = 0;

	if( !write->reg || !write->tg || !write->row_pointer ) {
		write_destroy( write );
		return( NULL );
	}
	
        return( write );
}

/* ICC writer from lcms.
 */

/*
 * Since an ICC profile can be larger than the maximum size of a JPEG marker
 * (64K), we need provisions to split it into multiple markers.  The format
 * defined by the ICC specifies one or more APP2 markers containing the
 * following data:
 *      Identifying string      ASCII "ICC_PROFILE\0"  (12 bytes)
 *      Marker sequence number  1 for first APP2, 2 for next, etc (1 byte)
 *      Number of markers       Total number of APP2's used (1 byte)
 *      Profile data            (remainder of APP2 data)
 * Decoders should use the marker sequence numbers to reassemble the profile,
 * rather than assuming that the APP2 markers appear in the correct sequence.
 */

#define ICC_MARKER  (JPEG_APP0 + 2)     /* JPEG marker code for ICC */
#define ICC_OVERHEAD_LEN  14            /* size of non-profile data in APP2 */
#define MAX_BYTES_IN_MARKER  65533      /* maximum data len of a JPEG marker */
#define MAX_DATA_BYTES_IN_MARKER  (MAX_BYTES_IN_MARKER - ICC_OVERHEAD_LEN)


/*
 * This routine writes the given ICC profile data into a JPEG file.
 * It *must* be called AFTER calling jpeg_start_compress() and BEFORE
 * the first call to jpeg_write_scanlines().
 * (This ordering ensures that the APP2 marker(s) will appear after the
 * SOI and JFIF or Adobe markers, but before all else.)
 */

static void
write_icc_profile (j_compress_ptr cinfo,
                   const JOCTET *icc_data_ptr,
                   unsigned int icc_data_len)
{
  unsigned int num_markers;     /* total number of markers we'll write */
  int cur_marker = 1;           /* per spec, counting starts at 1 */
  unsigned int length;          /* number of bytes to write in this marker */

  /* Calculate the number of markers we'll need, rounding up of course */
  num_markers = icc_data_len / MAX_DATA_BYTES_IN_MARKER;
  if (num_markers * MAX_DATA_BYTES_IN_MARKER != icc_data_len)
    num_markers++;

  while (icc_data_len > 0) {
    /* length of profile to put in this marker */
    length = icc_data_len;
    if (length > MAX_DATA_BYTES_IN_MARKER)
      length = MAX_DATA_BYTES_IN_MARKER;
    icc_data_len -= length;

    /* Write the JPEG marker header (APP2 code and marker length) */
    jpeg_write_m_header(cinfo, ICC_MARKER,
                        (unsigned int) (length + ICC_OVERHEAD_LEN));

    /* Write the marker identifying string "ICC_PROFILE" (null-terminated).
     * We code it in this less-than-transparent way so that the code works
     * even if the local character set is not ASCII.
     */
    jpeg_write_m_byte(cinfo, 0x49);
    jpeg_write_m_byte(cinfo, 0x43);
    jpeg_write_m_byte(cinfo, 0x43);
    jpeg_write_m_byte(cinfo, 0x5F);
    jpeg_write_m_byte(cinfo, 0x50);
    jpeg_write_m_byte(cinfo, 0x52);
    jpeg_write_m_byte(cinfo, 0x4F);
    jpeg_write_m_byte(cinfo, 0x46);
    jpeg_write_m_byte(cinfo, 0x49);
    jpeg_write_m_byte(cinfo, 0x4C);
    jpeg_write_m_byte(cinfo, 0x45);
    jpeg_write_m_byte(cinfo, 0x0);

    /* Add the sequencing info */
    jpeg_write_m_byte(cinfo, cur_marker);
    jpeg_write_m_byte(cinfo, (int) num_markers);

    /* Add the profile data */
    while (length--) {
      jpeg_write_m_byte(cinfo, *icc_data_ptr);
      icc_data_ptr++;
    }
    cur_marker++;
  }
}

/* Write an ICC Profile from a file into the JPEG stream.
 */
static int
write_profile( Write *write, const char *profile )
{
	extern char *im__file_read( FILE *fp, 
		const char *name, int *length_out );

	FILE *fp;

#ifdef BINARY_OPEN
        if( !(fp = fopen( profile, "rb" )) ) {
#else /*BINARY_OPEN*/
        if( !(fp = fopen( profile, "r" )) ) {
#endif /*BINARY_OPEN*/
		im_errormsg( "im_vips2jpeg: unable to read profile \"%s\"",
			profile );
		return( -1 );
	}
	if( !(write->profile_bytes = 
		im__file_read( fp, profile, &write->profile_length )) ) {
		fclose( fp );
		return( -1 );
	}
	fclose( fp );

	write_icc_profile( &write->cinfo, 
		write->profile_bytes, write->profile_length );

#ifdef DEBUG
	printf( "im_vips2jpeg: attached profile \"%s\"\n", profile );
#endif /*DEBUG*/

	return( 0 );
}

/* Write a VIPS image to a JPEG compress struct.
 */
static int
write_vips( Write *write, int qfac, char *profile )
{
	IMAGE *in = write->in;

	Rect area;
	int y, i;

        /* Check input image.
         */
	if( im_pincheck( in ) )
		return( -1 );
        if( in->BandFmt != IM_BANDFMT_UCHAR || in->Coding != IM_CODING_NONE ) {
                im_errormsg( "im_vips2jpeg: uncoded uchar only" );
                return( -1 );
        }
        if( in->Bands != 1 && in->Bands != 3 ) {
                im_errormsg( "im_vips2jpeg: 1 or 3-band only" );
                return( -1 );
        }
        if( qfac < 0 || qfac > 100 ) {
                im_errormsg( "im_vips2jpeg: qfac should be in 0-100 range" );
                return( -1 );
        }

	/* Set compression parameters.
	 */
        write->cinfo.image_width = in->Xsize;
        write->cinfo.image_height = in->Ysize;
	if( in->Bands == 3 ) {
		write->cinfo.input_components = 3;
		write->cinfo.in_color_space = JCS_RGB; 
	}
	else if( in->Bands == 1 ) {
		write->cinfo.input_components = 1;
		write->cinfo.in_color_space = JCS_GRAYSCALE; 
	}

	/* Rest to default. 
	 */
        jpeg_set_defaults( &write->cinfo );
        jpeg_set_quality( &write->cinfo, qfac, TRUE );

	/* Build compress tables.
	 */
	jpeg_start_compress( &write->cinfo, TRUE );

	/* Attach profile.
	 */
	if( profile && write_profile( write, profile ) )
		return( -1 );

	/* Write data.
	 */
	for( y = 0; y < in->Ysize; y += write->tile_height ) {
		area.left = 0;
		area.top = y;
		area.width = in->Xsize;
		area.height = IM_MIN( write->tile_height, in->Ysize - y );

                if( im_prepare_thread( write->tg, write->reg, &area ) )
                        return( -1 );

		for( i = 0; i < area.height; i++ )
			write->row_pointer[i] = (JSAMPROW) 
				IM_REGION_ADDR( write->reg, 0, y + i );

		jpeg_write_scanlines( &write->cinfo, 
			write->row_pointer, area.height );
	}

	jpeg_finish_compress( &write->cinfo );

	return( 0 );
}

/* Write a VIPS image to a file as JPEG.
 */
int
im_vips2jpeg( IMAGE *in, const char *filename )
{
	Write *write;
	int qfac = 75; 
	char *profile = NULL;

	char *p, *q;

	char name[FILENAME_MAX];
	char mode[FILENAME_MAX];
	char buf[FILENAME_MAX];

	/* Parse mode from filename.
	 */
	im_filename_split( filename, name, mode );
	strcpy( buf, mode ); 
	p = &buf[0];
	if( (q = im_getnextoption( &p )) ) {
		if( strcmp( q, "" ) != 0 )
			qfac = atoi( mode );
	}
	if( (q = im_getnextoption( &p )) ) {
		if( strcmp( q, "" ) != 0 ) 
			profile = q;
	}
	if( (q = im_getnextoption( &p )) ) {
		im_errormsg( "im_vips2jpeg: unknown extra options \"%s\"", q );
		return( -1 );
	}

	if( !(write = write_new( in )) )
		return( -1 );

	if( setjmp( write->eman.jmp ) ) {
		/* Here for longjmp() from new_error_exit().
		 */
		write_destroy( write );

		return( -1 );
	}

	/* Can't do this in write_new(), has to be after we've made the
	 * setjmp().
	 */
        jpeg_create_compress( &write->cinfo );

	/* Make output.
	 */
#ifdef BINARY_OPEN
        if( !(write->eman.fp = fopen( name, "wb" )) ) {
#else /*BINARY_OPEN*/
        if( !(write->eman.fp = fopen( name, "w" )) ) {
#endif /*BINARY_OPEN*/
		write_destroy( write );
                im_errormsg( "im_vips2jpeg: unable to open \"%s\"", name );

                return( -1 );
        }
        jpeg_stdio_dest( &write->cinfo, write->eman.fp );

	/* Convert!
	 */
	if( write_vips( write, qfac, profile ) ) {
		write_destroy( write );
		return( -1 );
	}
	write_destroy( write );

	return( 0 );
}

/* Just like the above, but we write to a memory buffer.
 *
 * A memory buffer for the compressed image.
 */
typedef struct {
	/* Public jpeg fields.
	 */
	struct jpeg_destination_mgr pub;

	/* Private stuff during write.
	 */
	JOCTET *data;		/* Allocated area */
	int used;		/* Number of bytes written so far */
	int size;		/* Max size */

	/* Copy the compressed area here.
	 */
	IMAGE *out;		/* Allocate relative to this */
	char **obuf;		/* Allocated buffer, and size */
	int *olen;
} OutputBuffer;

/* Init dest method.
 */
METHODDEF(void)
init_destination( j_compress_ptr cinfo )
{
	OutputBuffer *buf = (OutputBuffer *) cinfo->dest;
	int mx = cinfo->image_width * cinfo->image_height * 
		cinfo->input_components * sizeof( JOCTET );

	/* Allocate relative to the image we are writing .. freed when we junk
	 * this output.
	 */
	buf->data = (JOCTET *) (*cinfo->mem->alloc_large) 
			( (j_common_ptr) cinfo, JPOOL_IMAGE, mx );
	buf->used = 0;
	buf->size = mx;

	/* Set buf pointers for library.
	 */
	buf->pub.next_output_byte = buf->data;
	buf->pub.free_in_buffer = mx;
}

/* Buffer full method ... should never get this.
 */
METHODDEF(boolean)
empty_output_buffer( j_compress_ptr cinfo )
{
	/* Not really a file write error, but why not. Should never happen.
	 */
	ERREXIT( cinfo, JERR_FILE_WRITE );

	return( 0 );
}

/* Cleanup. Write entire buffer as a MIME type.
 */
METHODDEF(void)
term_destination( j_compress_ptr cinfo )
{
        OutputBuffer *buf = (OutputBuffer *) cinfo->dest;
	int len = buf->size - buf->pub.free_in_buffer;
	void *obuf;

	/* Allocate and copy to the VIPS output area.
	 */
	if( !(obuf = im_malloc( buf->out, len )) )
		ERREXIT( cinfo, JERR_FILE_WRITE );
	memcpy( obuf, buf->data, len );
	*(buf->obuf) = obuf;
	*(buf->olen) = len;
}

/* Set dest to one of our objects.
 */
static void
buf_dest( j_compress_ptr cinfo, IMAGE *out, char **obuf, int *olen )
{
	OutputBuffer *buf;

	/* The destination object is made permanent so that multiple JPEG 
	 * images can be written to the same file without re-executing 
	 * jpeg_stdio_dest. This makes it dangerous to use this manager and 
	 * a different destination manager serially with the same JPEG object,
	 * because their private object sizes may be different.  
	 *
	 * Caveat programmer.
	 */
	if( !cinfo->dest ) {    /* first time for this JPEG object? */
		cinfo->dest = (struct jpeg_destination_mgr *)
			(*cinfo->mem->alloc_small) 
				( (j_common_ptr) cinfo, JPOOL_PERMANENT,
				  sizeof( OutputBuffer ) );
	}

	buf = (OutputBuffer *) cinfo->dest;
	buf->pub.init_destination = init_destination;
	buf->pub.empty_output_buffer = empty_output_buffer;
	buf->pub.term_destination = term_destination;

	/* Save output parameters.
	 */
	buf->out = out;
	buf->obuf = obuf;
	buf->olen = olen;
}

/* As above, but save to a buffer. The buffer is allocated relative to out.
 * On success, buf is set to the output buffer and len to the size of the
 * compressed image.
 */
int
im_vips2bufjpeg( IMAGE *in, IMAGE *out, int qfac, char **obuf, int *olen )
{
	Write *write;

	if( !(write = write_new( in )) )
		return( -1 );

	/* Clear output parameters.
	 */
	*obuf = NULL;
	*olen = 0;

	/* Make jpeg compression object.
 	 */
	if( setjmp( write->eman.jmp ) ) {
		/* Here for longjmp() from new_error_exit().
		 */
		write_destroy( write );

		return( -1 );
	}
        jpeg_create_compress( &write->cinfo );

	/* Attach output.
	 */
        buf_dest( &write->cinfo, out, obuf, olen );

	/* Convert!
	 */
	if( write_vips( write, qfac, NULL ) ) {
		write_destroy( write );

		return( -1 );
	}
	write_destroy( write );

	return( 0 );
}

/* As above, but save as a mime jpeg on stdout.
 */
int
im_vips2mimejpeg( IMAGE *in, int qfac )
{
	IMAGE *base;
	int len;
	char *buf;

	if( !(base = im_open( "im_vips2mimejpeg:1", "p" )) )
		return( -1 );
	if( im_vips2bufjpeg( in, base, qfac, &buf, &len ) ) {
		im_close( base );
		return( -1 );
	}

	/* Write as a MIME type.
	 */
	printf( "Content-length: %d\r\n", len );
	printf( "Content-type: image/jpeg\r\n" );
	printf( "\r\n" );
	fwrite( buf, sizeof( char ), len, stdout );
	fflush( stdout );

	im_close( base );

	if( ferror( stdout ) ) {
		im_errormsg( "im_vips2mimejpeg: error writing output" );
		return( -1 );
	}

	return( 0 );
}

/* Read a cinfo to a VIPS image.
 */
static int
read_vips_header( struct jpeg_decompress_struct *cinfo, IMAGE *out )
{
	/* Read JPEG header.
	 */
	jpeg_read_header( cinfo, TRUE );
	jpeg_calc_output_dimensions( cinfo );

	/* Set VIPS header.
	 */
	im_initdesc( out,
		 cinfo->output_width, cinfo->output_height,
		 cinfo->output_components,
		 IM_BBITS_BYTE, IM_BANDFMT_UCHAR, IM_CODING_NONE, IM_TYPE_sRGB,
		 1.0, 1.0, 0, 0 );

	return( 0 );
}

/* Read a cinfo to a VIPS image.
 */
static int
read_vips_image( struct jpeg_decompress_struct *cinfo, IMAGE *out )
{
	int y, sz;
	JSAMPROW row_pointer[1];

	/* Check VIPS.
	 */
	if( im_outcheck( out ) || im_setupout( out ) )
		return( -1 );

	/* Get size of output line and make a buffer.
	 */
	sz = cinfo->output_width * cinfo->output_components;
	row_pointer[0] = (JSAMPLE *) (*cinfo->mem->alloc_large) 
			( (j_common_ptr) cinfo, JPOOL_IMAGE, sz );

	/* Start up decompressor.
	 */
	jpeg_start_decompress( cinfo );

	/* Process image.
	 */
	for( y = 0; y < out->Ysize; y++ ) {
		if( jpeg_read_scanlines( cinfo, &row_pointer[0], 1 ) != 1 ) {
			im_errormsg( "im_jpeg2vips: truncated JPEG file" );
			return( -1 );
		}
		if( im_writeline( y, out, row_pointer[0] ) )
			return( -1 );
	}

	/* Stop decompressor.
	 */
	jpeg_finish_decompress( cinfo );

	return( 0 );
}

/* Read a JPEG file header into a VIPS header.
 */
int
im_jpeg2vips_header( const char *name, IMAGE *out )
{
	struct jpeg_decompress_struct cinfo;
        ErrorManager eman;
	FILE *fp;

	/* Make jpeg decompress object.
 	 */
        cinfo.err = jpeg_std_error( &eman.pub );
	eman.pub.error_exit = new_error_exit;
	eman.pub.output_message = new_output_message;
	eman.fp = NULL;
	if( setjmp( eman.jmp ) ) {
		/* Here for longjmp() from new_error_exit().
		 */
		jpeg_destroy_decompress( &cinfo );

		return( -1 );
	}
        jpeg_create_decompress( &cinfo );

	/* Make input.
	 */
#ifdef BINARY_OPEN
        if( !(fp = fopen( name, "rb" )) ) {
#else /*BINARY_OPEN*/
        if( !(fp = fopen( name, "r" )) ) {
#endif /*BINARY_OPEN*/
		jpeg_destroy_decompress( &cinfo );
                im_errormsg( "im_jpeg2vips_header: unable to open \"%s\"", 
			name );

                return( -1 );
        }
	eman.fp = fp;
        jpeg_stdio_src( &cinfo, fp );

	/* Read!
	 */
	if( read_vips_header( &cinfo, out ) ) {
		jpeg_destroy_decompress( &cinfo );
		fclose( fp );
		return( -1 );
	}

	/* Close and tidy.
	 */
	fclose( fp );
	eman.fp = NULL;
	jpeg_destroy_decompress( &cinfo );

	if( eman.pub.num_warnings != 0 ) {
		im_warning( "im_jpeg2vips_header: read header gave "
			"%d warnings", eman.pub.num_warnings );
		im_warning( "error log:\n%s", im_errorstring() );
	}

	return( 0 );
}

/* Read a JPEG file into a VIPS image.
 */
int
im_jpeg2vips( const char *name, IMAGE *out )
{
	struct jpeg_decompress_struct cinfo;
        ErrorManager eman;
	FILE *fp;

	/* Make jpeg compression object.
 	 */
        cinfo.err = jpeg_std_error( &eman.pub );
	eman.pub.error_exit = new_error_exit;
	eman.pub.output_message = new_output_message;
	eman.fp = NULL;
	if( setjmp( eman.jmp ) ) {
		/* Here for longjmp() from new_error_exit().
		 */
		jpeg_destroy_decompress( &cinfo );

		return( -1 );
	}
        jpeg_create_decompress( &cinfo );

	/* Make input.
	 */
#ifdef BINARY_OPEN
        if( !(fp = fopen( name, "rb" )) ) {
#else /*BINARY_OPEN*/
        if( !(fp = fopen( name, "r" )) ) {
#endif /*BINARY_OPEN*/
		jpeg_destroy_decompress( &cinfo );
                im_errormsg( "im_jpeg2vips: unable to open \"%s\"", name );

                return( -1 );
        }
	eman.fp = fp;
        jpeg_stdio_src( &cinfo, fp );

	/* Convert!
	 */
	if( read_vips_header( &cinfo, out ) ||
		read_vips_image( &cinfo, out ) ) {
		jpeg_destroy_decompress( &cinfo );
		fclose( fp );
		return( -1 );
	}

	/* Close and tidy.
	 */
	fclose( fp );
	eman.fp = NULL;
	jpeg_destroy_decompress( &cinfo );

	if( eman.pub.num_warnings != 0 ) {
		im_warning( "im_jpeg2vips: read gave %d warnings", 
			eman.pub.num_warnings );
		im_warning( "error log:\n%s", im_errorstring() );
	}

	return( 0 );
}

#endif /*HAVE_JPEG*/
