/* @(#) Program which inverts a matrix of any size
 * @(#) page 43-45 of numerical recipes in C 1998
 * @(#)
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/
#include <vips/intl.h>

#include <stdio.h>
#include <math.h>

#include <vips/vips.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

#define TINY	1.0e-200

static int ludcmp(a, n, indx, d)
int n, *indx;
double **a, *d;
{
	int i, j, k;
	int imax = 0;
	double big, dum, sum, temp;
	double *vv, *im_dvector();
	void im_free_dvector();

	if ( (vv = im_dvector(1,n)) == NULL)
		return(-1);
	*d = 1.0;
	for(i=1; i<=n; i++)
		{
		big = 0.0;
		for (j=1; j<=n; j++)
			if((temp=fabs(a[i][j])) > big)
				big = temp;
		if (big == 0.0)
			{
			im_errormsg("im_invmat: singular matrix in ludcmp");
			return(-1);
			}
		vv[i] = 1.0/big;
		}
	
	for (j=1; j<=n; j++)
		{
		for (i=1; i<j; i++)
			{
			sum = a[i][j];
			for (k=1; k<i; k++)
				sum -= a[i][k] * a[k][j];
			a[i][j] = sum;
			}
		big = 0.0;
		for (i=j; i<=n; i++)
			{
			sum = a[i][j];
			for (k=1; k<j; k++)
				sum -= a[i][k] * a[k][j];
			a[i][j] = sum;
			if ( (dum=vv[i] * fabs(sum)) >= big)
				{
				big = dum;
				imax = i;
				}
			}
		if (j!= imax)
			{
			for (k=1; k<=n; k++)
				{
				dum = a[imax][k];
				a[imax][k] = a[j][k];
				a[j][k] = dum;
				}
			*d = -(*d);
			vv[imax] = vv[j];
			}
		indx[j] = imax;
		if (a[j][j] == 0.0)
			a[j][j] = TINY;
		if (j != n)
			{
			dum = 1.0/(a[j][j]);
			for (i=j+1; i<=n; i++)
				a[i][j] *= dum;
			}
		}
	im_free_dvector(vv, 1, n);
	return(0);
}

static void lubksb(a, n, indx, b)
double **a, b[];
int n, *indx;
{
	int i, ii=0, ip, j;
	double sum;

	for (i=1; i<=n; i++)
		{
		ip = indx[i];
		sum = b[ip];
		b[ip] = b[i];
		if (ii)
			for (j=ii; j<=i-1; j++)
				sum -= a[i][j] * b[j];
		else if (sum)
			ii = i;
		b[i] = sum;
		}
	
	for (i=n; i>=1; i--)
		{
		sum = b[i];
		for (j=i+1; j<=n; j++)
			sum -= a[i][j] * b[j];
			b[i] = sum/a[i][i];
		}
}

/* Invert a square  size x size matrix stored in matrix[][]
 * result is returned in the same matrix
 */
int 
im_invmat(matrix, size)
double **matrix;
int size;
{
	double **a, **y, *col, d;
	int N, *indx;
	int i,j;
#ifdef DEBUG
	int p, q;
#endif

	N = size;
	/* Malloc a matrix a[1,N][1,N] */
	a = im_dmat_alloc(1, N, 1, N);
	if (a == NULL)
		{
		im_errormsg("im_invmat: dmat_alloc failed");
		return(-1);
		}
	for (j=1; j<=N; j++)
		for (i=1; i<=N; i++)
			a[i][j] = matrix[i-1][j-1];

#ifdef DEBUG
		fprintf(stderr, "i=%d j=%d\n", i, j);
		for (q=1; q<=N; q++)
			{
			for (p=1; p<=N; p++)
				fprintf(stderr, "%f\t", a[p][q]);
			fprintf(stderr, "\n");
			}
#endif

	y = im_dmat_alloc(1, N, 1, N);
	if (y == NULL)
		{
		im_errormsg("im_invmat: dmat_alloc failed (2)");
		return(-1);
		}
	if ( ((indx = im_ivector(1,N)) == NULL) || ((col = im_dvector(1,N)) == NULL) )
		{
		im_errormsg("im_invmat: ivector or dvector failed");
		return(-1);
		}
	if (ludcmp(a, N, indx, &d) == -1)
		{
		im_errormsg("im_invmat: ludcmp failed");
		return(-1);
		}

#ifdef DEBUG
		for (q=1; q<=N; q++)
			{
			for (p=1; p<=N; p++)
				fprintf(stderr, "%f\t", a[p][q]);
			fprintf(stderr, "\n");
			}
#endif

	for(j=1; j<=N; j++)
		{
		for(i=1; i<=N; i++)
			col[i] = 0.0;
		col[j] = 1.0;
		lubksb(a, N, indx, col);
#ifdef DEBUG
		fprintf(stderr, "j=%d\n", j);
		for (q=1; q<=N; q++)
			{
			for (p=1; p<=N; p++)
				fprintf(stderr, "%f\t", a[p][q]);
			fprintf(stderr, "\n");
			}
#endif
		for(i=1; i<=N; i++)
			y[i][j] = col[i];
		}
	im_free_ivector(indx, 1, N);
	im_free_dvector(col, 1, N);
	im_free_dmat(a, 1, N, 1, N);

	for (j=1; j<=N; j++)
		for (i=1; i<=N; i++)
			matrix[i-1][j-1] = y[i][j];

	im_free_dmat(y, 1, N, 1, N);
	return(0);
}
