/** @file
 *
 * Seamless mode:
 * Linux guest.
 */

/*
 * Copyright (C) 2006-2007 innotek GmbH
 *
 * This file is part of VirtualBox Open Source Edition (OSE), as
 * available from http://www.virtualbox.org. This file is free software;
 * you can redistribute it and/or modify it under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * in version 2 as it comes in the "COPYING" file of the VirtualBox OSE
 * distribution. VirtualBox OSE is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY of any kind.
 */

/*****************************************************************************
*   Header files                                                             *
*****************************************************************************/

#include <iprt/thread.h>
#include <iprt/log.h>
#include <iprt/err.h>
#include <VBox/VBoxGuest.h>

#include "seamless.h"

/*****************************************************************************
*   Defined Constants and Macros                                             *
*****************************************************************************/

/** List of names known to be used for desktop windows by X11 window managers. */
static const char *gDesktopWMNames[] =
{
    "Desktop",
    "KDE Desktop"
};

/**
 * Perform the actual class initialisation.  This involves connecting to the X server, checking
 * for the presence of any needed X extensions, signalling that we can do seamless mode and
 * starting a thread to listen for switches to and from seamless mode in the host.
 *
 * @returns IPRT status code
 *
 * @note As always, anyone who feels like replacing the error codes with better ones is
 *       encouraged to do so.
 */
int VBoxSeamlessLinux::init(void)
{
    int event, error, rc;

    if (!XInitThreads())
        return VERR_NOT_SUPPORTED;
    mDisplay = XOpenDisplay(0);
    if (mDisplay == 0)
        return VERR_ACCESS_DENIED;
    if (!XShapeQueryExtension(mDisplay, &event, &error))
    {
        XCloseDisplay(mDisplay);
        mDisplay = 0;
        return VERR_NOT_SUPPORTED;
    }
    rc = VbglR3SeamlessSetCap(true);
    if (RT_FAILURE(rc))
    {
        XCloseDisplay(mDisplay);
        mDisplay = 0;
        return rc;
    }
    rc = RTThreadCreate(&mHostEventThread, VBoxSeamlessLinux::hostEventThread,
                        reinterpret_cast<void *>(this), 0, RTTHREADTYPE_INFREQUENT_POLLER,
                        RTTHREADFLAGS_WAITABLE, "Seamless event thread");
    if (RT_FAILURE(rc))
    {
        VbglR3SeamlessSetCap(false);
        XCloseDisplay(mDisplay);
        mDisplay = 0;
        return rc;
    }
    return VINF_SUCCESS;
}

/**
 * Class destructor.  Terminate the event thread and signal to the host that we no longer
 * support seamless mode.
 */
VBoxSeamlessLinux::~VBoxSeamlessLinux(void)
{
    mExitHostEventThread = true;
    VbglR3InterruptEventWaits();
    RTThreadWait(mX11SeamlessThread, SEAMLESS_WAIT_TIME, 0);
    VbglR3SeamlessSetCap(false);
}

/**
 * This thread waits for seamless state change events from the host and deals with them.
 *
 * @returns        IRPT return code.
 * @param   pvUser A pointer to the instance of the VBoxSeamlessLinux class in which the
 *                 thread is running.
 */
int VBoxSeamlessLinux::hostEventThread(RTTHREAD, void *pvUser)
{
    VBoxSeamlessLinux *pSelf = reinterpret_cast<VBoxSeamlessLinux *>(pvUser);

    while (!pSelf->mExitHostEventThread)
    {
        VMMDevSeamlessMode newMode;

        int rc = VbglR3SeamlessWaitEvent(&newMode);
        if (RT_FAILURE(rc))
        {
            /* Are we supposed to stop? */
            if (pSelf->mExitHostEventThread)
            {
                pSelf->stopX11SeamlessThread();
                return VINF_SUCCESS;
            }
            /* If not, sleep for a bit to avoid using up too much CPU wile retrying. */
            RTThreadYield();
            continue;
        }
        switch(newMode)
        {
        case VMMDev_Seamless_Visible_Region:
        /* A simplified seamless mode, obtained by making the host VM window borderless and
           making the guest desktop transparent. */
           pSelf->startX11SeamlessThread();
           break;
        case VMMDev_Seamless_Host_Window:
        /* One host window represents one guest window.  Not yet implemented. */
            LogRelFunc(("Warning: VMMDev_Seamless_Host_Window request received.\n"));
            /* fall through to default */
        default:
            LogRelFunc(("Warning: unsupported VMMDev_Seamless request received.\n"));
            /* fall through to case VMMDev_Seamless_Disabled */
        case VMMDev_Seamless_Disabled:
            pSelf->stopX11SeamlessThread();
        }
    }
    pSelf->stopX11SeamlessThread();
    return VINF_SUCCESS;
}

/**
 * Read information about currently visible windows in the guest and start an X11 event
 * loop to get information about updates.  All this information is passed on to the host.
 *
 * @note This function should only be called from within the host event thread.
 * @todo If this function fails (or something fatal happens inside the seamless thread),
 *       we should disable seamless capabilities again and shutdown this class as far as
 *       possible.
 * @todo This function should switch the guest to fullscreen mode.
 */
void VBoxSeamlessLinux::startX11SeamlessThread(void)
{
    int rc;

    if (mX11SeamlessThread != 0)
        return;
    if (!addDesktopWindow(DefaultRootWindow(mDisplay)))
        return;
    rc = RTThreadCreate(&mX11SeamlessThread, VBoxSeamlessLinux::X11SeamlessThread,
                        reinterpret_cast<void *>(this), 0, RTTHREADTYPE_INFREQUENT_POLLER,
                        RTTHREADFLAGS_WAITABLE, "X11 seamless thread");
    if (RT_FAILURE(rc))
    {
        mX11SeamlessThread = 0;
        freeWindowTree();
        return;
    }
}

/**
 * Stop the X11 seamless event thread.
 *
 * @note This function should only be called from within the host event thread.
 * @todo See todos for for VBoxSeamlessLinux::startX11SeamlessThread
 */
void VBoxSeamlessLinux::stopX11SeamlessThread(void)
{
    XClientMessageEvent clientMessage = { ClientMessage };  /* Other members set to zero. */
    if (mX11SeamlessThread == 0)
        return;
    mExitX11SeamlessThread = true;
    XSendEvent(mDisplay, DefaultRootWindow(mDisplay), false, StructureNotifyMask,
               reinterpret_cast<XEvent *>(&clientMessage));
    RTThreadWait(mX11SeamlessThread, SEAMLESS_WAIT_TIME, 0);
    mX11SeamlessThread = 0;
    freeWindowTree();
}

/**
 * Go through the list of this window's children.  Add information about mapped windows
 * to the tree of visible windows.  Recursively repeat the process for any desktop
 * windows found (hopefully there will only be one!)  Register for X11 substructure events
 * (i.e. children have been created/destroyed/moved/whatever) on the desktop window(s).
 *
 * @returns true on success, false on failure.  In the last case, the tree of visible
 *          windows will be freed.
 * @param   hWin     the window concerned - should be a desktop window
 */
bool VBoxSeamlessLinux::addDesktopWindow(Window hWin)
{
    Window hRootWin, hParent, *ahChildren;
    int rc;
    unsigned int cChildren;

    if (XSelectInput(mDisplay, hWin, StructureNotifyMask) == 0)
    {
        freeWindowTree();
        return false;
    }
    mDesktopWindows.push_back(hWin);
    rc = XQueryTree(mDisplay, hWin, &hRootWin, &hParent, &ahChildren, &cChildren);
    if (rc == 0)
    {
        freeWindowTree();
        return false;
    }

    for (unsigned int i = 0; i < cChildren; ++i)
    {
        char *pcWMName;
        bool isDesktopWin = false;

        if ((XFetchName(mDisplay, ahChildren[i], &pcWMName) != 0) && (pcWMName != 0))
        {
            for (unsigned int j = 0; j < sizeof(gDesktopWMNames); ++i)
                if (gDesktopWMNames[j] != 0)
                    isDesktopWin = true;
            XFree(pcWMName);
        }
        if (isDesktopWin)
            if (!addDesktopWindow(ahChildren[i]))
            {
                freeWindowTree();
                return false;
            }
        else
        {
            XWindowAttributes winAttrib;

            if (XGetWindowAttributes(mDisplay, ahChildren[i], &winAttrib) == 0)
            {
                freeWindowTree();
                return false;
            }
            if (winAttrib.map_state == IsViewable)
                if (!addWindowToTree(ahChildren[i], winAttrib.x, winAttrib.y))
                {
                    freeWindowTree();
                    return false;
                }
        }
    }
    return true;
}

/**
 * Adds information about a guest window to the tree.  Only information about mapped
 * windows should be added.
 *
 * @returns true on success, false on failure
 * @param hWin     the ID of the window on the display
 * @param x        the X offset of the window on the guest screen
 * @param y        the Y offset of the window on the guest screen
 */
bool VBoxSeamlessLinux::addWindowToTree(Window hWin, int x, int y)
{
    XRectangle *pRects;
    int cRects, iOrdering;
    typedef std::pair<Window, vboxGuestWinInfo> pairWindow;

    pRects = XShapeGetRectangles(mDisplay, hWin, ShapeClip, &cRects, &iOrdering);
    if (pRects == 0)
        return false;
    XShapeSelectInput(mDisplay, hWin, ShapeNotifyMask);
    mWinInfo.insert(pairWindow(hWin, vboxGuestWinInfo(x, y, cRects, pRects)));
    mcRects += cRects;
    return true;
}

/**
 * Updates the list of visible rectangles for a guest window in the tree.
 *
 * @returns true on success, false on failure
 * @param hWin     the ID of the window on the display
 */
bool VBoxSeamlessLinux::updateRectsInTree(Window hWin)
{
    std::map<Window, vboxGuestWinInfo>::iterator iter;
    int cRects, iOrdering;

    iter = mWinInfo.find(hWin);
    if (iter == mWinInfo.end())
    {
        LogRelFunc(("Warning: ShapeNotify event received for a window not listed as mapped\n"));
        return false;
    }
    mcRects -= iter->second.cRects;
    XFree(iter->second.pRects);
    iter->second.pRects = XShapeGetRectangles(mDisplay, hWin, ShapeClip, &cRects,
                                                 &iOrdering);
    if (iter->second.pRects == 0)
    {
        LogRelFunc(("Warning: failed to get shape rectangles for window (id %d)\n",
                    hWin));
        removeWindowFromTree(hWin);
        return false;
    }
    iter->second.cRects += cRects;
    mcRects += cRects;
    return true;
}

/**
 * Removes information about a guest window from the tree.
 *
 * @returns true on success, false on failure
 * @param hWin     the ID of the window on the display
 * @param x        the X offset of the window on the guest screen
 * @param y        the Y offset of the window on the guest screen
 */
void VBoxSeamlessLinux::removeWindowFromTree(Window hWin)
{
    std::map<Window, vboxGuestWinInfo>::iterator iter;

    iter = mWinInfo.find(hWin);
    if (iter == mWinInfo.end())
    {
        LogRelFunc(("Warning: UnmapNotify event received for a window not listed as mapped\n"));
        return;
    }
    mcRects -= iter->second.cRects;
    XFree(iter->second.pRects);
    mWinInfo.erase(iter);
}

/**
 * Free all information in the tree of visible windows
 */
void VBoxSeamlessLinux::freeWindowTree(void)
{
    std::map<Window, vboxGuestWinInfo>::iterator j;

    mcRects = 0;
    for (unsigned int i = 0; i != mDesktopWindows.size(); ++i)
        XSelectInput(mDisplay, mDesktopWindows[i], 0);
    mDesktopWindows.clear();
    for (j = mWinInfo.begin(); j != mWinInfo.end();
         mWinInfo.erase(j++) /* increment i and erase the dereferenced *old* value */)
        XShapeSelectInput(mDisplay, j->first, 0);
}

/**
 * This thread waits for position and shape-related events from guest windows and for 
 *
 * @returns        IRPT return code.
 * @param   pvUser A pointer to the instance of the VBoxSeamlessLinux class in which the
 *                 thread is running.
 *
 * @todo See todos for VBoxSeamlessLinux::startX11SeamlessThread
 */
int VBoxSeamlessLinux::X11SeamlessThread(RTTHREAD, void *pvUser)
{
    VBoxSeamlessLinux *pSelf = reinterpret_cast<VBoxSeamlessLinux *>(pvUser);

    /* Start by sending information about the current window setup to the host.  We do this
       here because we want to send all such information from a single thread. */
    pSelf->updateHostSeamlessInfo();
    while (!pSelf->mExitX11SeamlessThread)
    {
        XEvent event;

        XNextEvent(pSelf->mDisplay, &event);
        switch (event.type)
        {
        case ConfigureNotify:
            pSelf->doConfigureEvent(&event.xconfigure);
            break;
        case MapNotify:
            pSelf->doMapEvent(&event.xmap);
            break;
        case UnmapNotify:
            pSelf->doUnmapEvent(&event.xunmap);
            break;
        case ShapeNotify:
            pSelf->doShapeEvent(reinterpret_cast<XShapeEvent *>(&event));
            break;
        default:
            break;
        }
    }
    return VINF_SUCCESS;
}

/**
 * Handle a configuration event in the seamless event thread by setting the new position and
 * updating the host's rectangle information.
 *
 * @param event the X11 event structure
 */
void VBoxSeamlessLinux::doConfigureEvent(XConfigureEvent *event)
{
    std::map<Window, vboxGuestWinInfo>::iterator iter;

    iter = mWinInfo.find(event->window);
    if (iter == mWinInfo.end())
    {
        LogRelFunc(("Warning: ConfigureNotify event received for an unknown window\n"));
        return;
    }
    if (   (iter->second.x != event->x)
        || (iter->second.y != event->y))
    {
        iter->second.x = event->x;
        iter->second.y = event->y;
        updateHostSeamlessInfo();
    }
}

/**
 * Handle a map event in the seamless event thread by adding the guest window to the list of
 * visible windows and updating the host's rectangle information.
 *
 * @param event the X11 event structure
 */
void VBoxSeamlessLinux::doMapEvent(XMapEvent *event)
{
    std::map<Window, vboxGuestWinInfo>::iterator iter;
    Window rootReturn;
    int xReturn, yReturn;
    unsigned int wReturn, hReturn, borderReturn, depthReturn;

    /* Make sure that the window is not already present in the tree */
    iter = mWinInfo.find(event->window);
    if (iter != mWinInfo.end())
    {
        LogRelFunc(("Warning: MapNotify event received for a window listed as mapped\n"));
        removeWindowFromTree(event->window);
    }
    /* Get its coordinates */
    if (!XGetGeometry(mDisplay, event->window, &rootReturn, &xReturn, &yReturn,
                              &wReturn, &hReturn, &borderReturn, &depthReturn))
    {
        LogRelFunc(("Warning: failed to get attributes for newly mapped window (id %d)\n",
                    event->window));
        return;
    }
    if (!addWindowToTree(event->window, xReturn, yReturn))
    {
        LogRelFunc(("Warning: failed to add window (id %d) to the tree of guest windows\n",
                    event->window));
        return;
    }
    updateHostSeamlessInfo();
}

/**
 * Handle an unmap event in the seamless event thread by removing the guest window from the
 * list of visible windows and updating the host's rectangle information.
 *
 * @param event the X11 event structure
 */
void VBoxSeamlessLinux::doUnmapEvent(XUnmapEvent *event)
{
    removeWindowFromTree(event->window);
    updateHostSeamlessInfo();
}

/**
 * Handle a window shape change event in the seamless event thread by changing the set of
 * visible rectangles for the window in the list of visible guest windows and updating the
 * host's rectangle information.
 *
 * @param event the X11 event structure
 */
void VBoxSeamlessLinux::doShapeEvent(XShapeEvent *event)
{
    updateRectsInTree(event->window);
    updateHostSeamlessInfo();
}

void VBoxSeamlessLinux::updateHostSeamlessInfo()
{
    std::map<Window, vboxGuestWinInfo>::iterator iter;
    RTRECT *pRects = new RTRECT[mcRects];
    int cRects = 0;

    for (iter = mWinInfo.begin(); iter != mWinInfo.end(); ++iter)
    {
        for (int i = 0; i < iter->second.cRects; ++i, ++cRects)
        {
            pRects[cRects].xLeft = iter->second.pRects[i].x;
            pRects[cRects].yBottom = iter->second.pRects[i].y + iter->second.pRects[i].height;
            pRects[cRects].xRight = iter->second.pRects[i].x + iter->second.pRects[i].width;
            pRects[cRects].yTop = iter->second.pRects[i].y;
        }
    }
    VbglR3SeamlessSendRects(cRects, pRects);
    delete[] pRects;
}
