/** @file
 *
 * VBox frontends: Qt GUI ("VirtualBox"):
 * Declarations of utility classes and functions
 */

/*
 * Copyright (C) 2006-2008 Sun Microsystems, Inc.
 *
 * This file is part of VirtualBox Open Source Edition (OSE), as
 * available from http://www.virtualbox.org. This file is free software;
 * you can redistribute it and/or modify it under the terms of the GNU
 * General Public License (GPL) as published by the Free Software
 * Foundation, in version 2 as it comes in the "COPYING" file of the
 * VirtualBox OSE distribution. VirtualBox OSE is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY of any kind.
 *
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa
 * Clara, CA 95054 USA or visit http://www.sun.com if you need
 * additional information or have any questions.
 */

#ifndef __VBoxUtils_h__
#define __VBoxUtils_h__

/* Qt includes */
#include <QMouseEvent>
#include <QWidget>
#include <QTextEdit>

/**
 *  Simple class that filters out all key presses and releases
 *  got while the Alt key is pressed. For some very strange reason,
 *  QLineEdit accepts those combinations that are not used as accelerators,
 *  and inserts the corresponding characters to the entry field.
 */
class QIAltKeyFilter : protected QObject
{
    Q_OBJECT;

public:

    QIAltKeyFilter (QObject *aParent) :QObject (aParent) {}

    void watchOn (QObject *aObject) { aObject->installEventFilter (this); }

protected:

    bool eventFilter (QObject * /* aObject */, QEvent *aEvent)
    {
        if (aEvent->type() == QEvent::KeyPress || aEvent->type() == QEvent::KeyRelease)
        {
            QKeyEvent *event = static_cast<QKeyEvent *> (aEvent);
            if (event->modifiers() & Qt::AltModifier)
                return true;
        }
        return false;
    }
};

/**
 *  Simple class which simulates focus-proxy rule redirecting widget
 *  assigned shortcut to desired widget.
 */
class QIFocusProxy : protected QObject
{
    Q_OBJECT;

public:

    QIFocusProxy (QWidget *aFrom, QWidget *aTo)
        : QObject (aFrom), mFrom (aFrom), mTo (aTo)
    {
        mFrom->installEventFilter (this);
    }

protected:

    bool eventFilter (QObject *aObject, QEvent *aEvent)
    {
        if (aObject == mFrom && aEvent->type() == QEvent::Shortcut)
        {
            mTo->setFocus();
            return true;
        }
        return QObject::eventFilter (aObject, aEvent);
    }

    QWidget *mFrom;
    QWidget *mTo;
};

/**
 *  QTextEdit reimplementation to feat some extended requirements.
 */
class QRichTextEdit : public QTextEdit
{
    Q_OBJECT;

public:

    QRichTextEdit (QWidget *aParent) : QTextEdit (aParent) {}

    void setViewportMargins (int aLeft, int aTop, int aRight, int aBottom)
    {
        QTextEdit::setViewportMargins (aLeft, aTop, aRight, aBottom);
    }
};

#ifdef Q_WS_MAC
# undef PAGE_SIZE
# undef PAGE_SHIFT
# include <Carbon/Carbon.h>

/* Asserts if a != noErr and prints the error code */
#define AssertCarbonOSStatus(a) AssertMsg ((a) == noErr, ("Carbon OSStatus: %d\n", static_cast<int> (a)))

class QImage;
class QPixmap;
class QToolBar;
class VBoxFrameBuffer;

/* Converting stuff */
CGImageRef darwinToCGImageRef (const QImage *aImage);
CGImageRef darwinToCGImageRef (const QPixmap *aPixmap);
CGImageRef darwinToCGImageRef (const char *aSource);

/**
 * Returns a reference to the HIView of the QWidget.
 *
 * @returns HIViewRef of the QWidget.
 * @param   aWidget   Pointer to the QWidget
 */
inline HIViewRef darwinToHIViewRef (QWidget *aWidget)
{
    return HIViewRef(aWidget->winId());
}

/**
 * Returns a reference to the Window of the HIView.
 *
 * @returns WindowRef of the HIView.
 * @param   aViewRef  Reference to the HIView
 */
inline WindowRef darwinToWindowRef (HIViewRef aViewRef)
{
    return reinterpret_cast<WindowRef> (HIViewGetWindow(aViewRef));
}

/**
 * Returns a reference to the Window of the QWidget.
 *
 * @returns WindowRef of the QWidget.
 * @param   aWidget   Pointer to the QWidget
 */
inline WindowRef darwinToWindowRef (QWidget *aWidget)
{
    return ::darwinToWindowRef (::darwinToHIViewRef (aWidget));
}

/**
 * Returns a reference to the CGContext of the QWidget.
 *
 * @returns CGContextRef of the QWidget.
 * @param   aWidget      Pointer to the QWidget
 */
inline CGContextRef darwinToCGContextRef (QWidget *aWidget)
{
    return static_cast<CGContext *> (aWidget->macCGHandle());
}

/**
 * Converts a QRect to a HIRect.
 *
 * @returns HIRect for the converted QRect.
 * @param   aRect  the QRect to convert
 */
inline HIRect darwinToHIRect (const QRect &aRect)
{
    return CGRectMake (aRect.x(), aRect.y(), aRect.width(), aRect.height());
}

QString darwinSystemLanguage();

bool darwinIsMenuOpen();

void darwinSetShowToolBarButton (QToolBar *aToolBar, bool aShow);

void darwinWindowAnimateResize (QWidget *aWidget, const QRect &aTarget);

/* Proxy icon creation */
QPixmap darwinCreateDragPixmap (const QPixmap& aPixmap, const QString &aText);

/* Special routines for the dock handling */
CGImageRef darwinCreateDockBadge (const char *aSource);
void darwinUpdateDockPreview (CGImageRef aVMImage, CGImageRef aOverlayImage, CGImageRef aStateImage = NULL);
void darwinUpdateDockPreview (VBoxFrameBuffer *aFrameBuffer, CGImageRef aOverlayImage);

/* Icons in the menu of an mac application are unusual. */
void darwinDisableIconsInMenus();

/* Experimental region handler for the seamless mode */
OSStatus darwinRegionHandler (EventHandlerCallRef aInHandlerCallRef, EventRef aInEvent, void *aInUserData);

# ifdef DEBUG
void darwinDebugPrintEvent (const char *aPrefix, EventRef aEvent);
# endif
#endif /* Q_WS_MAC */

#endif // __VBoxUtils_h__

