/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developpers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/
/*!
 *\file interface_graphic.cpp
 *\brief This is graphical interface C source code.
 */


#include "interface_graphic.hpp"

using namespace std;

/***********************************
 *
 *
 *
 * Constructors and destructor.
 *
 *
 *
 **********************************/


Interface_graphic::Interface_graphic(Interface_console* core, QWidget *parent):QMainWindow(parent){ 

    setupUi(this);
    
    _zoom_box_check_value="";
    _core = core;
    _is_rendering_trace = false;
    _no_warning = false;/* display warnings */
     _on_manual_change = false;
    
    _x_factor_virtual_to_real = 0.1;
    _y_factor_virtual_to_real = 0.1;
    
    load_windows();

    setMouseTracking (true);/* to catch mouse events */
    
    _help_widget = new Help();

    _stats_window = new Stats_window(this);

}

Interface_graphic::~Interface_graphic(){

    delete _stats_window;
    delete _help_widget;

    /* Qt desallocates this, _ui_info_window and _render_area automatically */
}



/***********************************
 *
 *
 *
 * Window function.
 *
 *
 *
 **********************************/

void Interface_graphic::load_windows(){

    QUiLoader loader;
    QFile file_info( ":/window/info_window.ui");
    QFile file_selection_export(":/window/option_export_window.ui");
    

    if (!QGLFormat::hasOpenGL())
        qFatal("This system has no OpenGL support");
       
    /* Load the informative window from a .ui file */
    file_info.open(QFile::ReadOnly);
    CKFP(_ui_info_window = loader.load(&file_info, this), "Cannot open the .ui file : " << ":/window/info_window.ui");
    file_info.close();

    /* Load the _option_export_window from a .ui file */
    file_selection_export.open(QFile::ReadOnly);
    CKFP(_ui_time_selection_export = loader.load(&file_selection_export, this), "Cannot open the .ui file : " << "option_export_window.ui");
    file_selection_export.close();

 
    /* Set some windows properties */
    _ui_info_window->setWindowFlags(_ui_info_window->windowFlags() | Qt::WindowStaysOnTopHint);/* Always display info_window on top */
    _ui_time_selection_export->setWindowFlags(_ui_time_selection_export->windowFlags() | Qt::WindowStaysOnTopHint);
   
    /* Load widget from the .ui file */
    CKFP(_ui_render_area_layout = qFindChild<QVBoxLayout*>(this, "render_area_layout"), "Cannot find the render_area layout in the .ui file");   
    CKFP(_ui_fullscreen_menu = qFindChild<QAction*>(this, "fullscreen"), "Cannot find the fullscreen menu in the .ui file");
    CKFP(_ui_info_trace_text = qFindChild<QTextEdit*>(_ui_info_window, "info_trace_text"), "Cannot find the info_trace_text QTextEdit widget in the .ui file");
    CKFP(_ui_info_selection_text = qFindChild<QTextEdit*>(_ui_info_window, "info_selection_text"), "Cannot find the info_selection_text QTextEdit widget in the .ui file");
    CKFP(_ui_toolbar_check = qFindChild<QAction*>(this, "toolbar_menu"), "Cannot find the tool bar checker in the .ui file");   
    CKFP(_ui_toolbar = qFindChild<QToolBar*>(this, "toolBar"), "Cannot find the tool bar in the .ui file");   

    CKFP(_ui_x_scroll = qFindChild<QScrollBar*>(this, "x_scroll"), "Cannot find the horizontal scroll bar in the .ui file");
    CKFP(_ui_y_scroll = qFindChild<QScrollBar*>(this, "y_scroll"), "Cannot find the vertical scroll bar in the .ui file");

    CKFP(_ui_zoom_box = qFindChild<QComboBox*>(this, "zoom_box"), "Cannot find the zoom box in the .ui file");   

    // Export window
    CKFP(_ui_export_ok_button = qFindChild<QPushButton*>(_ui_time_selection_export, "option_export_ok"), "Cannot find the ok push button in the export dialog .ui file");   
    CKFP(_ui_min_time_export = qFindChild<QLineEdit*>(_ui_time_selection_export, "min_time"), "Cannot find the ok push button in the export dialog .ui file");

    CKFP(_ui_max_time_export = qFindChild<QLineEdit*>(_ui_time_selection_export, "max_time"), "Cannot find the ok push button in the export dialog .ui file");

    CKFP(_ui_CheckBox_time_export = qFindChild<QCheckBox*>(_ui_time_selection_export, "export_all"), "Cannot find the Line text \"export_all\" in the export dialog .ui file");   

    CKFP(_ui_wide_export = qFindChild<QLineEdit*>(_ui_time_selection_export, "wide"), "Cannot find the Line text \"wide\" in the export dialog .ui file");   
    CKFP(_ui_grow_export = qFindChild<QLineEdit*>(_ui_time_selection_export, "grow"), "Cannot find the Line text \"grow\" in the export dialog .ui file");   
    CKFP(_ui_accuracy_export = qFindChild<QLineEdit*>(_ui_time_selection_export, "accuracy"), "Cannot find the Line text \"accuracy\" in the export dialog .ui file");   
    CKFP(_ui_CheckBox_wide_export = qFindChild<QCheckBox*>(_ui_time_selection_export, "checkBox_wide"), "Cannot find the Line text \"wide\" in the export dialog .ui file");   
    CKFP(_ui_CheckBox_grow_export = qFindChild<QCheckBox*>(_ui_time_selection_export, "checkBox_grow"), "Cannot find the Line text \"grow\" in the export dialog .ui file");   
    CKFP(_ui_CheckBox_accuracy_export = qFindChild<QCheckBox*>(_ui_time_selection_export, "checkBox_accuracy"), "Cannot find the Line text \"accuracy\" in the export dialog .ui file");   
    CKFP(_ui_more_options_export = qFindChild<QCommandLinkButton*>(_ui_time_selection_export, "more_options"), "Cannot find the button \"more_options\" in the export dialog .ui file");   



    connect(_ui_export_ok_button, SIGNAL(pressed()),
            this, SLOT(option_export_ok_pressed()));
    /* 
       Special function of Qt which allows methods declared as slots and which name are 'on_[widget]_[action]()' to be called when the 'widget' triggered the signal corresponding to 'action'.
       /!\ -> use NULL as argument, else messages will be duplicated! 
    */
    QMetaObject::connectSlotsByName(NULL);
    // QMetaObject::connectSlotsByName(_ui_time_selection_export);
    
    _ui_x_scroll->setMaximum(_REAL_X_SCROLL_LENGTH);
    _ui_y_scroll->setMaximum(_REAL_Y_SCROLL_LENGTH);
    
    /* Display the main window */
    this->show();
}



/***********************************
 *
 *
 *
 * Informative message functions.
 *
 *
 *
 **********************************/

void Interface_graphic::error(const string s) const{
    if ( true==_no_warning ) return;/* do not display error messages */

    QString buf = s.c_str();
    _ui_info_trace_text->moveCursor(QTextCursor::End);/* Insert the new text on the end */
    _ui_info_trace_text->insertHtml("<font color='red'>"+buf+"</font><br /><br />");

    _ui_info_window->show();/* show info_window */
}


void Interface_graphic::warning(const string s) const{
    if ( true==_no_warning ) return;/* do not display warning messages */

    QString buf = s.c_str();
    _ui_info_trace_text->moveCursor(QTextCursor::End);/* Insert the new text on the end */
    _ui_info_trace_text->insertHtml("<font color='orange'>"+buf+"</font><br /><br />");

    _ui_info_window->show();/* show info_window */
}


void Interface_graphic::information(const string s) const{
    QString buf = s.c_str();
    _ui_info_trace_text->moveCursor(QTextCursor::End);/* Insert the new text on the end */
    _ui_info_trace_text->insertHtml("<font color='green'>"+buf+"</font><br /><br />");
}





/***********************************
 *
 *
 *
 * Selected Trace entity informative function.
 *
 *
 *
 **********************************/


void Interface_graphic::selection_information(const string s) const{
    QString buf = s.c_str();
    _ui_info_selection_text->clear();/* Clear the current text (if exists) */
    _ui_info_selection_text->insertHtml("<font color='blue'>"+buf+"</font>");
    _ui_info_selection_text->moveCursor(QTextCursor::Start);/* Insert the new text on the begin */

    _ui_info_window->show();/* show info_window */
}





/***********************************
 *
 *
 *
 * Tool functions.
 *
 *
 *
 **********************************/

void Interface_graphic::opening_file(const string &path){
    
    information(string("File opened: ")+ path);
    
    _trace_path = path;

    this->setWindowTitle(QString("ViTE :: ")+_trace_path.c_str());
    
    QApplication::setOverrideCursor(Qt::WaitCursor);
   
    /* if (_core->draw_trace(path)==false){
       error("Draw trace failed");
       }*/

    //  _ui_render_area->build();

    _is_rendering_trace = true;
        
    //   _ui_render_area->updateGL();  

    QApplication::restoreOverrideCursor();   

}


void Interface_graphic::bind_render_area(QGLWidget* render_area){
    /* Bind the render area to a layout (_ui_render_area_layout) */
    _ui_render_area = render_area;
    _ui_render_area_layout->addWidget(_ui_render_area);
}


void Interface_graphic::set_scroll_bars_length(Element_pos new_x_virtual_length, Element_pos new_y_virtual_length){
    _x_factor_virtual_to_real = _REAL_X_SCROLL_LENGTH / new_x_virtual_length;
    _y_factor_virtual_to_real = _REAL_Y_SCROLL_LENGTH / new_y_virtual_length;
}


void Interface_graphic::linking_scroll_bars(Element_pos x_bar_new_value, Element_pos y_bar_new_value){
    _on_manual_change = true;
    _ui_x_scroll->setValue(virtual_to_real_scroll_unit(x_bar_new_value, 'x'));
    _ui_y_scroll->setValue(virtual_to_real_scroll_unit(y_bar_new_value, 'y'));
    _on_manual_change = false;
}

Element_pos Interface_graphic::real_to_virtual_scroll_unit(int scroll_position, const char &axe){
    switch (axe){
    case 'x':
	return (scroll_position/_x_factor_virtual_to_real);
	break;
    case 'y':
	return (scroll_position/_y_factor_virtual_to_real);
	break;
    
    default:
	error("Option not recognized.");
	break;
    }
    return scroll_position;
}

int Interface_graphic::virtual_to_real_scroll_unit(Element_pos scroll_position, const char &axe){
    switch (axe){
    case 'x':
      return (int)(scroll_position*_x_factor_virtual_to_real);
	break;
    case 'y':
	return (int)(scroll_position*_y_factor_virtual_to_real);
	break;
    
    default:
	error("Option not recognized.");
	break;
    }
    return (int)scroll_position;
}

void Interface_graphic::change_zoom_box_value(int new_value){
    QString formated_value;
    ostringstream int_to_string;
    int index;

    int max_value = -2147483648L;

    if (max_value == new_value)/* prevent overflow value */
        new_value = 2147483647;

    int_to_string << new_value;
  
    formated_value = QString(int_to_string.str().c_str())+QString("%");
   
    index = _ui_zoom_box->findText( formated_value, Qt::MatchExactly);

    _zoom_box_check_value = formated_value;

    if (-1 == index){/* value does not exist in the list */
        _ui_zoom_box->removeItem(0);
        _ui_zoom_box->insertItem(0,  formated_value);/* add the value */
        _ui_zoom_box->setCurrentIndex(0);
       
    }
    else{
        _ui_zoom_box->setCurrentIndex(index);/* value exists, select it */
    }
}


/***********************************
 *
 *
 *
 * Widget slot functions.
 *
 *
 *
 **********************************/

void Interface_graphic::on_open_triggered(){

    /* 
       Be careful! Here, this is used instead 'this' pointer because 'this' is not a window, 
       it's an interface class (Interface_graphic*). The real window is this. 
       If 'this' is put, the application closes after the getOpenFilename() dialog box closes.
    */
    QString filename = QFileDialog::getOpenFileName(this);
  
    if (!filename.isEmpty()){

        if (_is_rendering_trace == true){ /* open a new process */
            
	    QStringList arguments = (QStringList() << filename);
	    QString* program;
	    const QString* run_env = _core->get_runenv();
	    QDir::setCurrent(run_env[0]);
	    QString run_cmd = run_env[1];
	    
	    if (run_cmd.startsWith("."))
		program = new QString(run_env[0]+(run_cmd.remove(0,1)));
	    else
		program = new QString(run_cmd);
	    
	    QProcess * new_process = new QProcess();
	    new_process->startDetached(*program, arguments);
	}
        else{
            opening_file(filename.toStdString());

            if(false==_core->draw_trace(filename.toStdString(), Interface_console::_DRAW_OPENGL))
                error("Draw trace failed");
        }/* end else of if (_is_rendering_trace == true) */
    
    }/* end if (!filename.isEmpty()) */
    
}



void Interface_graphic::on_export_file_triggered(){
    if(_is_rendering_trace == false)
        return;

    ostringstream temp;
    
    temp.str("");
    
    temp << Info::Render::_x_min_visible; 
    _ui_min_time_export->setText(temp.str().c_str());

    temp.str("");
    temp << Info::Render::_x_max_visible; 
    _ui_max_time_export->setText(temp.str().c_str());
    _ui_time_selection_export->show();
}

void Interface_graphic::option_export_ok_pressed(){
    // We have to save the option from _ui_time_selection_export and hide her if it is not do automatically

    _ui_time_selection_export->hide();
    const QString path_by_default = QString(_trace_path.substr(0, _trace_path.find_last_of('.')).c_str()) + ".svg";
    QString filename = QFileDialog::getSaveFileName(this, tr("Export File"),
                                                    path_by_default,
                                                    tr("Images (*.svg)"));
    
    if (!filename.isEmpty()) {
        // Adding .svg to the end
        if(!filename.endsWith(".svg")) {
            filename += ".svg";
        }

        information(string("Exporting trace to ")+filename.toStdString());
                     
        _core->set_path_to_export(filename.toStdString());
        // _core->draw_trace(_trace_path, Interface_console::_DRAW_SVG);

        if(_ui_CheckBox_time_export->isChecked()){
            _core->set_min_value_for_export(convert_to_double(_ui_min_time_export->text().toStdString()));
            _core->set_max_value_for_export(convert_to_double(_ui_max_time_export->text().toStdString()));
        }
        else{
            _core->set_min_value_for_export(0);
            _core->set_max_value_for_export(Info::Entity::x_max);//TO DO TODO max_size de la trace?
        }
        // We configure the svg exporter
        double accuracy = convert_to_double(_ui_accuracy_export->text().toStdString());
        double height = convert_to_double(_ui_grow_export->text().toStdString());
        double wide = convert_to_double(_ui_wide_export->text().toStdString());
        
        if(_ui_more_options_export->isChecked()) {
            if(accuracy >= 0. && _ui_CheckBox_accuracy_export->isChecked())
                Svg::set_accuracy(accuracy);
            else { // Default value
                Svg::set_accuracy(0.);
            }
            
            if(height >= 0. && _ui_CheckBox_grow_export->isChecked())
                Svg::set_height_factor(height);
            else { // Default value
                Svg::set_height_factor(20.);
            }
            
            if(wide >= 0. && _ui_CheckBox_wide_export->isChecked())
                Svg::set_wide_factor(wide);
            else { // Default value
                Svg::set_wide_factor(100.);
            }
        }
        else{ // Default values
            Svg::set_accuracy(0.);
            Svg::set_height_factor(20.);
            Svg::set_wide_factor(100.);
        }

        _core->launch_action(Interface_console::_STATE_EXPORT_FILE_IN_INTERVAL);
    }
    else {
        error("No file specified for exportation");
    }
    
}


void Interface_graphic::on_close_triggered(){

    if(_is_rendering_trace == false)
        return;

    /*
     * Clear the informative window texts and hide it. 
     */
    _ui_info_trace_text->clear();/* Clear the current text (if exists) */
    _ui_info_selection_text->clear();/* Clear the current text (if exists) */
    _ui_info_window->hide();/* Hide the informative window */

    //_stats_window->close_window();

    _core->launch_action(Interface_console::_STATE_RELEASE_RENDER_AREA);

   
    
    _is_rendering_trace = false; 



    information(string("File closed."));

   

    this->setWindowTitle(QString("ViTE"));

}


void Interface_graphic::on_quit_triggered(){
    ((QWidget*)_help_widget)->close();
    ((QWidget*)this)->close();
}


void Interface_graphic::on_fullscreen_triggered(){

    /* 
       Notice that some problems can appears under X systems with
       the window decoration. Please refer to the Qt official documentation.
    */

    /* 
       The menu is checked before function call, so if 'fullscreen menu' is checked, 
       the main window is displayed in fullscreen mode 
    */
    if (_ui_fullscreen_menu->isChecked())
        this->showFullScreen();
    else
        this->showNormal();
}

void Interface_graphic::on_toolbar_menu_triggered(){
    if (_ui_toolbar_check->isChecked()){
        _ui_toolbar_check->setText("Hide toolbar");
        _ui_toolbar->show();
    }
    else{
        _ui_toolbar_check->setText("Show toolbar");
        _ui_toolbar->hide();
    }
}

void Interface_graphic::on_show_info_triggered(){
    if (_ui_info_window->isHidden())
        _ui_info_window->show();
    else
        _ui_info_window->hide();
}


void Interface_graphic::on_help_triggered(){

    if(_help_widget->isVisible()){
        _help_widget->hide();
    }
    else{
        _help_widget->show();
    }
}


void Interface_graphic::on_about_triggered(){

    /* QMessageBox::about(this, tr("About ViTE"),
       tr("<b>ViTE</b> is a trace format visualizer developped "
       "by french engineering school students for INRIA researchers (<a href='http://www.inria.fr'>http://www.inria.fr</a>).<br/><br/>"
       "It is licensed under <i>CECILL-A</i>."));*/

    QMessageBox::about(this, tr("About ViTE"),
                       tr("<h2>ViTE</h2>"
                          "the <b>Vi</b><i>sual </i><b>T</b><i>race</i> <b>E</b><i>xplorer</i> - <i>version 1.0</i> - <i>June 2009</i>.<br /><br />"
                          "Under the CeCILL A licence. The content can be found <a href=\"http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt\">here</a>."
                          "<p><b>Developers:</b><ul><li>Kevin COULOMB</li><li>Mathieu FAVERGE</li><li>Johnny JAZEIX</li><li>Olivier LAGRASSE</li><li>Jule MARCOUEILLE</li><li>Pascal NOISETTE</li><li>Arthur REDONDY</li><li>Cl&eacute;ment VUCHENER</li></ul></p>"
                          "The main page project is: <a href=\"http://vite.gforge.inria.fr/\">http://vite.gforge.inria.fr/</a>.<br /><br />"));
}

void Interface_graphic::on_show_stats_triggered(){
    if(_core->get_trace() != NULL) {
        _stats_window->set_filename(_core->get_filename());
        _stats_window->set_trace(_core->get_trace());
        _stats_window->init_window();
        _stats_window->show(); 
    }
    else {
        error("Must load a trace before viewing stats");
    }
}

void Interface_graphic::on_no_warning_triggered(){
    _no_warning = !_no_warning;
}

void Interface_graphic::on_no_arrows_triggered(){
    Info::Render::_no_arrows = !Info::Render::_no_arrows;
    _core->launch_action(Interface_console::_STATE_RENDER_UPDATE);
}

void Interface_graphic::on_no_events_triggered(){
    Info::Render::_no_events = !Info::Render::_no_events;
    _core->launch_action(Interface_console::_STATE_RENDER_UPDATE);
}



void Interface_graphic::on_zoom_in_triggered(){
    Element_pos t = 1;

    if (true==Info::Render::_key_alt)/* on Y axe */
        _core->launch_action(Interface_console::_STATE_RENDER_AREA_CHANGE_SCALE_Y, &t);
    else/* on X axe */
        _core->launch_action(Interface_console::_STATE_RENDER_AREA_CHANGE_SCALE, &t);
}

void Interface_graphic::on_zoom_out_triggered(){
    Element_pos t = -1;

    if (true==Info::Render::_key_alt)/* on Y axe */
        _core->launch_action(Interface_console::_STATE_RENDER_AREA_CHANGE_SCALE_Y, &t);
    else/* on X axe */
        _core->launch_action(Interface_console::_STATE_RENDER_AREA_CHANGE_SCALE, &t);
}

void Interface_graphic::on_goto_start_triggered(){
    int id;

    if (true==Info::Render::_key_alt)/* on Y axe */
        id = Info::Render::Y_TRACE_BEGINNING;
    else/* on X axe */
        id = Info::Render::X_TRACE_BEGINNING;

    _core->launch_action(Interface_console::_STATE_RENDER_AREA_REGISTERED_TRANSLATE, &id);
}

void Interface_graphic::on_goto_end_triggered(){
    int id;

    if (true==Info::Render::_key_alt)/* on Y axe */
        id = Info::Render::Y_TRACE_ENDING;
    else/* on X axe */
        id = Info::Render::X_TRACE_ENDING;

    _core->launch_action(Interface_console::_STATE_RENDER_AREA_REGISTERED_TRANSLATE, &id);
}

void Interface_graphic::on_show_all_trace_triggered(){
    int id = Info::Render::X_TRACE_ENTIRE;/* on X axe */
    _core->launch_action(Interface_console::_STATE_RENDER_AREA_REGISTERED_TRANSLATE, &id);
    id = Info::Render::Y_TRACE_ENTIRE;/* on Y axe */
    _core->launch_action(Interface_console::_STATE_RENDER_AREA_REGISTERED_TRANSLATE, &id);
}

void Interface_graphic::on_zoom_box_textChanged(QString s){
    QRegExp reg_exp_number("^(\\d+)");
    QString result;
    Element_pos d;

    if (s=="")/* if there is not value, go out */
       return;

    if (_zoom_box_check_value == s)/* We have just change the zoom box value (in  change_zoom_box_value()) what emits the signal.
                                      So ignore this function call */
        return;
    
   
    if (-1 != reg_exp_number.indexIn(s)){/* match a number */
        result = reg_exp_number.cap(1);/* capture number inside parenthesis in the RegExp */
        d = 0.01*result.toDouble();
        if (0 == d)/* if zoom is null ignore it */
            return;
        
        _core->launch_action(Interface_console::_STATE_RENDER_AREA_REPLACE_SCALE, &d);
    }

}

void Interface_graphic::on_x_scroll_valueChanged(int new_value){
    if (!_on_manual_change){
	Element_pos converted_new_value = real_to_virtual_scroll_unit(new_value, 'x');/* change type of new_value */
	_core->launch_action(Interface_console::_STATE_RENDER_AREA_REPLACE_TRANSLATE, &converted_new_value);
    }
}

void Interface_graphic::on_y_scroll_valueChanged(int new_value){
    if (!_on_manual_change){
	Element_pos converted_new_value = real_to_virtual_scroll_unit(new_value, 'y');/* change type of new_value */
	_core->launch_action(Interface_console::_STATE_RENDER_AREA_REPLACE_TRANSLATE_Y, &converted_new_value);
    }
}


void Interface_graphic::on_scale_container_state_valueChanged(int new_value){
    _core->launch_action(Interface_console::_STATE_RENDER_AREA_CHANGE_CONTAINER_SCALE, &new_value);
}


void Interface_graphic::closeEvent(QCloseEvent *event){

    _help_widget->close();

    if(isEnabled()){
        event->accept();/* accept to hide the window for a further destruction */
    }
    else{
        event->ignore();
    }
}

const std::string Interface_graphic::get_filename() const{
    return _trace_path;
}
