/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developpers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/
#include "TokenSource.hpp"

using namespace std;

int TokenSource::_cursor = 0;

#if defined (WIN32) || !defined(_POSIX_MAPPED_FILES)
std::ifstream TokenSource::_file;
int TokenSource::_filesize = 1;
#else
off_t TokenSource::_filesize = 1;
#endif

TokenSource::TokenSource(){
    //no file are opened
    //you must use open later
    _is_eof = true;
    _is_mmap_ok = false;
    _is_file_opened = false;
    _cursor = 0;
}


TokenSource::TokenSource(const char *filename){
    open(filename);
}

TokenSource::~TokenSource(){
#if defined (WIN32) || !defined(_POSIX_MAPPED_FILES)
    if(_is_file_opened) {
        _file.close();
    }
#else

    if (_is_mmap_ok) {
        if(munmap(_buffer, _filesize) == -1){
            Error::set(Error::_MUNMAP, Error::_WARNING);
        }
    }
    _buffer  = NULL;
    if(_is_file_opened){
        close(_fd);
    }
#endif

    if (!_token.empty()) {
        _token.clear();
    }
}

void TokenSource::open(const char *filename){
    
    _is_eof = false;


    /* Getting the file size */

#if defined (WIN32)|| !defined(_POSIX_MAPPED_FILES)

    
    _file.open(filename, ios::in);
    if(!_file.is_open()){
        throw Error::_OPEN;
    }
    _is_file_opened = true;

    _file.seekg (0, ios::end);
    TokenSource::_filesize = _file.tellg();
    _file.seekg (0, ios::beg);
    
#else
    struct stat buf;
    if(stat(filename, &buf) == -1) {
        throw Error::_FSTAT;
    }
    _filesize = buf.st_size;
#endif

    if(TokenSource::_filesize == 0) {
        throw Error::_EMPTY_FILE;
    }

#if defined (WIN32) || !defined(_POSIX_MAPPED_FILES)
        fill_buffer();
#else
    if ((_fd = ::open(filename, O_RDONLY)) == -1) {
        throw Error::_OPEN;
    }
    _is_file_opened = true;

#undef _MMAP
    if ((_buffer = (char *)mmap(0, _filesize, PROT_READ, MAP_PRIVATE, _fd, 0)) == MAP_FAILED) {
        throw Error::_MMAP;
    }
    _is_mmap_ok = true;
#endif
}

#if defined (WIN32) || !defined(_POSIX_MAPPED_FILES)
bool TokenSource::fill_buffer() {
    if (_file.eof()) {
        return false;
    }
    else {
        _file.read(_buffer, BUFFER_SIZE);
        _buffer_size = _file.gcount();
        return true;
    }
}
#endif

bool TokenSource::go_to_next_token() {
    while(_buffer[_cursor] == ' ' || _buffer[_cursor] == '\t') {
        _cursor ++;
#if defined (WIN32) || !defined(_POSIX_MAPPED_FILES)
        if (_cursor >= _buffer_size) {
            if (!fill_buffer()) {
                return true;
            }
            _cursor = 0;
        }
#else
        if(ensure_capacity())
            break;
#endif
    }
    return false;
}


void TokenSource::build_composite_token() {
    _cursor ++;
#if defined (WIN32) || !defined(_POSIX_MAPPED_FILES)
    if (_cursor == _buffer_size) {
        if (!fill_buffer()) {
	    Error::set(Error::_FSTAT, Error::_WARNING);
	    return;
        }
        _cursor = 0;
    }
#else
    if (ensure_capacity()) {
        Error::set(Error::_FSTAT, Error::_WARNING);
	return;
    }
#endif
    do {
        _token.push_back(_buffer[_cursor]);
        _cursor ++;
        
        if (ensure_capacity()) {
            break;
        }
    }
    while((_buffer[_cursor] != '"') && (_buffer[_cursor] != '\''));
    _cursor ++;
    
    ensure_capacity();
}

void TokenSource::build_simple_token() {
    while(_buffer[_cursor] != ' ' && _buffer[_cursor] != '\n' && _buffer[_cursor] != '\t'){
        _token.push_back(_buffer[_cursor]);
        _cursor ++;
        
        if (ensure_capacity()){
            break;
        }
    }
}

bool TokenSource::ensure_capacity() {
#if defined (WIN32) || !defined(_POSIX_MAPPED_FILES)
    if (_cursor >= _buffer_size) {
        if (!fill_buffer()) {
            _is_eof = true;
            return true;
        }
        _cursor = 0;
    }
    return false;
#else
    if (_cursor < _filesize)
        return false;
    _is_eof = true;
    return true;
#endif
}

void TokenSource::build_line_breaks() {
    while(_buffer[_cursor] == '\n') {
        _cursor ++;
        if (ensure_capacity()) {
            break;
        }
    }
    _token.push_back('\n');
}

void TokenSource::build_definition_starter() {
    _cursor ++;
    ensure_capacity();
    _token.push_back('%');
}


std::string *TokenSource::read_token() {
    //regexp recognition
    //a token is
    //(1) a line break
    //(2) composite : a char buffer delimited by \"
    //(3) a char buffer delimited by space

    if (!_token.empty()) {
        _token.clear();
    }

    if (_is_eof || go_to_next_token()) {
        return NULL;
    }

    if (_buffer[_cursor] == '\n') {
        build_line_breaks();
    }
    else if (_buffer[_cursor] == '%') {
        build_definition_starter();
    }
    else if ((_buffer[_cursor] != '"') &&
	     (_buffer[_cursor] != '\'')){
        build_simple_token();
    }
    else {
        build_composite_token();
    }
    
    return &_token;
}

float TokenSource::get_size_loaded() {

#if defined (WIN32) || !defined(_POSIX_MAPPED_FILES)
    if (_file == NULL)
        return 1.0f;

    if (!_file.is_open())
        return 2.0f;

    if (_file.bad())
        return 3.0f;
  
    return ((float)_file.tellg())/(float)TokenSource::_filesize;
#else
    if(_filesize == 0) {
        return 1.0f;
    }
    return ((float)_cursor)/(float)_filesize;
#endif
}

void TokenSource::reinit_cursor() {
    _cursor = 0;
}
