/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developpers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/
#ifndef CONTAINER_HPP
#define CONTAINER_HPP

/*!
 * \file Container.hpp
 */

#include <list>
using std::list;
#include <map>
using std::map;
#include <stack>
using std::stack;
#include "tree/BinaryTree.hpp"

#include "values/Values.hpp"

#include "ContainerType.hpp"

class Container;

#include "State.hpp"
#include "Event.hpp"
#include "Link.hpp"
#include "StateChange.hpp"

#include "VariableType.hpp"
#include "Variable.hpp"
#include "../statistics/Statistic.hpp"

/*!
 * \class Container
 * \brief Contains others containers or entities
 */
class Container {
friend class State;
private:
    Name _name;
    Date _creation_time, _destruction_time;
    ContainerType *_type;
    Container *_parent;
    list<Container *> _children;
    unsigned int _n_states;
    list<StateChange *> _states;
    BinaryTree<StateChange> *_state_tree;
    unsigned int _n_events;
    list<Event *> _events;
    BinaryTree<Event> *_event_tree;
    list<Link *> _links;
    map<VariableType *, Variable *> _variables;
    unsigned int _n_variables;
    
    /*
     * Temporary stores states before complete definition
     */
    struct current_state_t {
        Date start;
        StateType *type;
        EntityValue *value;
        map<std::string, Value *> opt;
    };
    stack<current_state_t> _current_states;
        
    /*
     * Temporary stores links before complete definition
     */
    struct current_link_t {
        Date start;
        LinkType *type;
        Container *source;
        EntityValue *value;
        map<std::string, Value *> opt;
    };
    map<String, current_link_t, String::less_than> _current_links;

    void add_current_state(Date end);

public: 
    /*!
     * \fn Container(Name name, Date creation_time, ContainerType *type, Container *parent)
     * \brief Constructor of Container
     * \param name Name of the container
     * \param creation_time Date when the container was created
     * \param type Type of the container
     * \param parent Parent container (NULL if the container is root)
     */
    Container(Name name, Date creation_time, ContainerType *type, Container *parent);
    
    /*!
     * \fn ~Container()
     * \brief Destructor of Container
     */
    ~Container();
    
    /*!
     * \fn add_child(Container *)
     * \brief Add a child to the container
     * \param child new child container
     */
    void add_child(Container *child);
    
    /*!
     * \fn set_state(Date time, StateType *type, EntityValue *value, map<std::string, Value *> &opt)
     * \brief Set the current state of the container
     * \param time Date of the event
     * \param type Type of the state
     * \param value Value of the state
     * \param opt Optional options of the state
     */
    void set_state(Date time, StateType *type, EntityValue *value, map<std::string, Value *> &opt);
    
    /*!
     * \fn push_state(Date time, StateType *type, EntityValue *value, map<std::string, Value *> &opt)
     * \brief Set the current state of the container and save the previous state
     * \param time Date of the event
     * \param type Type of the state
     * \param value Value of the state
     * \param opt Optional options of the state
     */
    void push_state(Date time, StateType *type, EntityValue *value, map<std::string, Value *> &opt);
    
    /*!
     * \fn pop_state(Date time)
     * \brief Restore a previous state
     * \param time Date of the event
     */
    void pop_state(Date time);
    
    /*!
     * \fn new_event(Date time, EventType *type, EntityValue *value, map<std::string, Value *> &opt)
     * \brief Add a new event
     * \param time Date of the event
     * \param type Type of the event
     * \param value Value of the event
     * \param opt Optional options of the state
     */
    void new_event(Date time, EventType *type, EntityValue *value, map<std::string, Value *> &opt);
    
    /*!
     * \fn start_link(Date time, LinkType *type, Container *source, EntityValue *value, String key, map<std::string, Value *> &opt)
     * \brief Start a new link identified by key
     * \param time Date of the start of the link
     * \param type Type of the link
     * \param source Container from where the link is sent
     * \param value Value of the link
     * \param key String that identifies the link to match its end
     * \param opt Optional options of the state
     */
    void start_link(Date time, LinkType *type, Container *source, EntityValue *value, String key, map<std::string, Value *> &opt);
    
    /*!
     * \fn end_link(Date time, Container *destination, String key, map<std::string, Value *> &opt)
     * \brief End a link identified by key
     * \param time Date of the end of the link
     * \param destination Container to where the link is sent
     * \param key String that identifies the link to match its beginning
     * \param opt Optional options of the state
     */
    void end_link(Date time, Container *destination, String key, map<std::string, Value *> &opt);
    
    /*!
     * \fn set_variable(Date time, VariableType *type, Double value)
     * \brief Set a new value of a variable
     * \param time Date of the event
     * \param type Type of the variable
     * \param value New value of the variable
     */
    void set_variable(Date time, VariableType *type, Double value);
    
    /*!
     * \fn add_variable(Date time, VariableType *type, Double value)
     * \brief Add a value to the current value of a variable
     * \param time Date of the event
     * \param type Type of the variable
     * \param value Value to add
     */
    void add_variable(Date time, VariableType *type, Double value);
    
    /*!
     * \fn sub_variable(Date time, VariableType *type, Double value)
     * \brief Substract a value to the current value of a variable
     * \param time Date of the event
     * \param type Type of the variable
     * \param value Value to substract
     */
    void sub_variable(Date time, VariableType *type, Double value);
    
    /*!
     * \fn get_name() const
     * \brief Get the name and the alias of the container
     */
    Name get_name() const;                    

    /*!
     * \fn get_parent() const
     * \brief Get the parent container
     */                
    const Container *get_parent() const;
    
    /*!
     * \fn get_type() const
     * \brief Get the type of the container
     */                    
    const ContainerType *get_type() const;
    
    /*!
     * \fn get_children() const
     * \brief Get the list of the child containers
     */                  
    const list<Container *> *get_children() const;
    
    /*!
     * \fn get_creation_time() const
     * \brief Get the time when the container was created
     */                    
    Date get_creation_time() const;
    
    /*!
     * \fn get_destruction_time() const
     * \brief Get the time when the container was destroyed
     */                    
    Date get_destruction_time() const;
    
    /*!
     * \fn get_states() const
     * \brief Get the state list
     */                              
    BinaryTree<StateChange> *get_states() const;// MODIF -> enleve * du template/ du const
    
    /*!
     * \fn get_events() const
     * \brief Get the event list
     */
    BinaryTree<Event> *get_events() const;
    
    /*!
     * \fn get_links() const
     * \brief Get the link list
     */
    const list<Link *> *get_links() const; 
    
    /*!
     * \fn get_variables() const
     * \brief Get the variables
     */
    const map<VariableType *, Variable *> *get_variables() const;
    
    /*!
     * \fn get_variable_number() const
     * \brief Get the number of variables
     */
    int get_variable_number() const;

    /*!
     * \fn get_event_number() const
     * \brief Return the number of events
     */
    int get_event_number() const;
    
    /*!
     * \fn destroy(const Date &time)
     * \brief Sets the destruction time of the container
     * \param time Destruction tim
     */                         
    void destroy(const Date &time);
    
    /*!
     * \fn finish(const Date &time)
     * \brief Finish to initialize the container (flush temporary states)
     * \param time Time to set destruction time if it was not destroy
     */
    void finish(const Date &time);

    /*!
     * \fn fill_stat( Statistic * stat, Interval I)
     * \brief Fill the stat element with the corresponding data to be displayed
     */
    void fill_stat( Statistic * stat, Interval I);
};

    /*!
     * \fn browse_stat_state(Node<StateChange> * node, Statistic * stat, Interval I, Node<StateChange> ** prev)
     * \brief Recursive function that browses the tree
     * \param node : The node that is currently added
     * \param stat : The Statistic class that is filled
     * \param I : The interval we want the data in
     * \param prev : To save the previous state before the interval to be able to count it in the statistic
     */
void browse_stat_state(Node<StateChange> * node, Statistic * stats, Interval I, Node<StateChange> ** prev);
    /*!
     * \fn browse_stat_link(const Container * cont, Statistic * S, Interval I)
     * \brief Fill the stat element with the corresponding data to be displayed
     */
void browse_stat_link(const Container * cont, Statistic * S, Interval I);

#endif
