
// dcmsplit.cpp
// split a DICOM file into identifying and non-identifying parts
// Copyright (c) 2009-2010 by The VoxBo Development Team

// VoxBo is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// VoxBo is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with VoxBo.  If not, see <http://www.gnu.org/licenses/>.
// 
// For general information on VoxBo, including the latest complete
// source code and binary distributions, manual, and associated files,
// see the VoxBo home page at: http://www.voxbo.org/
//
// original version written by Dan Kimberg

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <sstream>
#include "vbutil.h"
#include "vbio.h"
#include "dcmsplit.hlp.h"

using namespace std;
using boost::format;

extern "C" {

#include "dicom.h"

int32 hextol(const string &str);
void dcmsplit_help();
void dcmsplit_version();

int
main(int argc,char **argv)
{
  if (argc<2) {
    dcmsplit_help();
    exit(0);
  }
  string infile,outfile,anonfile;

  vector<string>filelist;
  set<uint16> stripgroups;
  set<dicomge> stripges;
  set<string> stripvrs;

  tokenlist args;
  args.Transfer(argc-1,argv+1);
  for (int i=0; i<args.size(); i++) {
    if (args[i]=="-h") {
      dcmsplit_help();
      exit(0);
    }
    else if (args[i]=="-v") {
      dcmsplit_version();
      exit(0);
    }
    else if (args[i]=="-g" && i<args.size()-1)
      stripgroups.insert(hextol(args[++i]));
    else if (args[i]=="-e" && i<args.size()-2) {
      stripges.insert(dicomge(hextol(args[i+1]),hextol(args[i+2])));
      i+=2;
    }
    else if (args[i]=="-d" && i<args.size()-1)
      stripvrs.insert(args[++i]);
    else
      filelist.push_back(args[i]);
  }
  
  if (filelist.size()<1 || filelist.size()>3) {
    dcmsplit_help();
    exit(100);
  }
  infile=outfile=filelist[0];
  if (filelist.size()>1)
    outfile=filelist[1];
  if (filelist.size()>2)
    anonfile=filelist[2];

  cout << format("[I] dcmsplit: splitting %s\n")%infile;
  cout << format("[I] dcmsplit:   stripped data will be in: %s\n")%outfile;
  cout << format("[I] dcmsplit:   identifying data will be in: %s\n")%anonfile;

  string ids=(string)"id:"+VBRandom_nchars(30);

  int err;
  if ((err=anonymize_dicom_header(infile,outfile,anonfile,
                                  stripgroups,stripges,stripvrs,ids)))
    cout << "[E] dcmsplit: anonymize error "<<err<<endl;
  else
    cout << "[I] dcmsplit: done: " << ids << endl;;
  exit(0);
}

int32
hextol(const string &str)
{
  string tmp=xstripwhitespace(str);
  stringstream ss(tmp);
  int32 res;
  ss >> hex >> res;
  if (ss.fail() || (size_t)ss.tellg()<tmp.size())
    return 0;
  return res;
}



void
dcmsplit_help()
{
  cout << boost::format(myhelp) % vbversion;
}

void
dcmsplit_version()
{
  printf("VoxBo dcmsplit (v%s)\n",vbversion.c_str());
}



}
