/*---------------------------------------------------------------------------*\

    This program takes a line off hook and records a wave file.  To use:

    1. Dial a telephone connected to the Voicetronix card.
    2. Start this program, it will take the line off hook and record anything
       you say.
    3. Wait for the time specified in the command line or press any DTMF to 
       finish.


         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2007 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
         MA  02110-1301  USA

\*---------------------------------------------------------------------------*/

#include "vpbapi.h"

#include <cstdlib>
#include <cstring>


static int arg_exists(int argc, char *argv[], const char *arg)
{ //{{{
	for(int i = 0; i < argc; ++i)
		if(strcmp(argv[i],arg) == 0) return i;

	return 0;
} //}}}

static void usage(const char *argv0)
{ //{{{
	printf("usage: %s [options] file secs\n\n", argv0);
	printf("  --card num    Record from card num.  Default is 0\n");
	printf("  --port num    Record from port num.  Default is 0\n");
	printf("  --recgain db  Set the record gain to db.\n");
	printf("  --onhook      Do not take the port off hook before recording.\n");
	printf("  --mode [linear | alaw | mulaw | adpcm]\n");
	printf("                Specify the recording sample type.\n");
	printf("                Default is 8kHz 16 bit linear.\n");
	printf("  --dial number Dial the given number before recording.\n");
	printf("  --cutthrough  Terminate the recording if dtmf is detected.\n\n");
} //}}}

int main(int argc, char * argv[])
{ //{{{
	int cardnum = 0;
	int portnum = 0;
	int arg;

	if(argc < 3) {
		usage(argv[0]);
		exit(EXIT_FAILURE);
	}
	if((arg = arg_exists(argc,argv,"--card")) != 0) cardnum = atoi(argv[arg+1]);
	if((arg = arg_exists(argc,argv,"--port")) != 0) portnum = atoi(argv[arg+1]);

	int h = vpb_open(cardnum, portnum);

	if((arg = arg_exists(argc, argv, "--recgain")) != 0)
		vpb_record_set_gain(h, atof(argv[arg+1]));

	AudioCompress   mode = VPB_LINEAR;
	if((arg = arg_exists(argc, argv, "--mode")) != 0)
	{
		if(strcmp(argv[arg+1], "alaw") == 0)       mode = VPB_ALAW;
		else if(strcmp(argv[arg+1], "mulaw") == 0) mode = VPB_MULAW;
		else if(strcmp(argv[arg+1], "adpcm") == 0) mode = VPB_OKIADPCM;
		else if(strcmp(argv[arg+1], "linear") != 0) {
			usage(argv[0]);
			exit(EXIT_FAILURE);
		}
	}

	VPB_RECORD  record;
	record.time_out = atoi(argv[argc-1]) * 1000;

	if(arg_exists(argc,argv,"--cutthrough")) record.term_digits = "1234567890*#";
	else                                     record.term_digits = "";

	vpb_record_set(h, &record);

	int port_type = vpb_get_port_type(h);

	if(port_type == VPB_FXO && !arg_exists(argc,argv,"--onhook"))
		vpb_sethook_sync(h, VPB_OFFHOOK);

	if((arg = arg_exists(argc, argv, "--dial")) != 0)
		vpb_dial_sync(h, argv[arg+1]);

	printf("Recording to %s for %d ms ...\n", argv[argc-2], record.time_out);

	vpb_record_file_sync(h, argv[argc-2], mode);

	if(port_type == VPB_FXO)
		vpb_sethook_sync(h, VPB_ONHOOK);

	vpb_close(h);

	return EXIT_SUCCESS;
} //}}}

