#include "global.h"

#include "net.h"
#include "rtpsess.h"	// Session
#include "stat.h"

#include "world.h"	// World::getChannelName
#include "gui.h"	// GuiGetCycles


/*** network counters ***/
uint32_t pkts_sent = 0;
uint32_t sum_pkts_lost = 0;

static uint32_t pkts_recvd = 0;
static uint32_t bytes_sent = 0;
static uint32_t bytes_recvd = 0;
static uint32_t pkts_rtp_sent = 0;
static uint32_t bytes_rtp_sent = 0;
static uint32_t pkts_rtcp_sent = 0;
static uint32_t bytes_rtcp_sent = 0;
static uint32_t pkt_max = 0;
static uint32_t pkt_min = 512;
static uint32_t sum_pkts_sent = 0;
static uint32_t sum_pkts_recvd = 0;
static uint32_t sum_pkts_rtp_sent = 0;
static uint32_t sum_pkts_rtcp_sent = 0;
static uint32_t sum_bytes_sent = 0;
static uint32_t sum_bytes_rtp_sent = 0;
static uint32_t sum_bytes_rtcp_sent = 0;
static uint32_t sum_bytes_recvd = 0;
static float kbps_sent = 0.0;
static float kbps_recvd = 0.0;

/*** timers ***/
ProfileTime ptime_init;
ProfileTime ptime_net;
ProfileTime ptime_events;
ProfileTime ptime_world;
ProfileTime ptime_render;
ProfileTime ptime_textures;
ProfileTime ptime_solids;
ProfileTime ptime_buffer;

/*** memory ***/
int new_wobject;
int del_wobject;
int new_world;
int del_world;
int new_solid;
int del_solid;
int new_texture;
int del_texture;
int new_image;
int del_image;
int new_channel;
int del_channel;
int new_session;
int del_session;
int new_source;
int del_source;
int new_payload;
int del_payload;
int new_netobject;
int del_netobject;
int new_objectlist;
int del_objectlist;
int new_netproperty;
int del_netproperty;

/** stat memory usage (new, delete) */
void statMemory(void)
{
  fprintf(stderr, "### Memory ###\n");
  fprintf(stderr, "WObject    : n=%8d d=%8d\n", new_wobject, del_wobject);
  fprintf(stderr, "World      : n=%8d d=%8d\n", new_world, del_world);
  fprintf(stderr, "Channel    : n=%8d d=%8d\n", new_channel, del_channel);
  fprintf(stderr, "Session    : n=%8d d=%8d\n", new_session, del_session);
  fprintf(stderr, "Source     : n=%8d d=%8d\n", new_source, del_source);
  fprintf(stderr, "Solid      : n=%8d d=%8d\n", new_solid, del_solid);
  fprintf(stderr, "Texture    : n=%8d d=%8d\n", new_texture, del_texture);
  fprintf(stderr, "Image      : n=%8d d=%8d\n", new_image, del_image);
  fprintf(stderr, "NetObject  : n=%8d d=%8d\n", new_netobject, del_netobject);
  fprintf(stderr, "Payload    : n=%8d d=%8d\n", new_payload, del_payload);
  fprintf(stderr, "NetProperty: n=%8d d=%8d\n", new_netproperty, del_netproperty);
  fprintf(stderr, "ObjectList : n=%8d d=%8d\n", new_objectlist, del_objectlist);
}

/** clear time */
void clearTime(ProfileTime *pt)
{
  memset(pt, 0, sizeof(ProfileTime));
}

/* start time */
void startTime(ProfileTime *pt)
{
  gettimeofday(&(pt->start), NULL);
}

/* stop time & cumul time, return last diff */
double stopTime(ProfileTime *pt)
{
  gettimeofday(&(pt->stop), NULL);
  double d = diffDates(pt->start, pt->stop);
  pt->cumul += d;
  return d;
}

/* compute kbps_sent & co */
void statAdjust(void)
{
  static struct timeval then;
  struct timeval now;

  gettimeofday(&now, NULL);
  float d = diffDates(then, now);
  if (fabs(d) > 0.1) {
    kbps_sent = bytes_sent/d/1000*8;
    kbps_recvd = bytes_recvd/d/1000*8;
    trace(DBG_NET, "kbps sent: %5.2f recvd: %5.2f",
	   bytes_sent/d/1000*8, bytes_recvd/d/1000*8);
    then = now;
    bytes_sent = bytes_recvd = 0;  
  }
}

void statSendPacket(int pkt_len)
{
  bytes_sent += pkt_len;
  sum_bytes_sent += pkt_len;
  pkts_sent++;
  sum_pkts_sent++;
  pkt_max = MAX((int) pkt_max, pkt_len);
  pkt_min = MIN((int) pkt_min, pkt_len);
}

void statSendRTP(int pkt_len)
{
  bytes_rtp_sent += pkt_len;
  sum_bytes_rtp_sent += pkt_len;
  pkts_rtp_sent++;
  sum_pkts_rtp_sent++;
}

void statSendRTCP(int pkt_len)
{
  bytes_rtcp_sent += pkt_len;
  sum_bytes_rtcp_sent += pkt_len;
  pkts_rtcp_sent++;
  sum_pkts_rtcp_sent++;
}

void statReceivePacket(int pkt_len)
{
  bytes_recvd += pkt_len;
  sum_bytes_recvd += pkt_len;
  pkts_recvd++;
  sum_pkts_recvd++;
  pkt_max = MAX((int) pkt_max, pkt_len);
  pkt_min = MIN((int) pkt_min, pkt_len);
}

static FILE *flog;

void Session::statRTP()
{
  if ((flog = writelog("--- RTP stat ---")) == (FILE *) NULL)
    return;
  writelog("worldname       : %s", World::getCurrentName());
  writelog("channel         : %s", World::getChannelName());
  writelog("source ssrc     : %x", source->ssrc);
  writelog("sources number  : %d", nbsources);
  writelog("source received : %d", source->s.received);
  writelog("source sr psent : %d", sr.psent);
  writelog("source sr osent : %d", sr.osent);
  writelog("source lost     : %d", source->lost);
  writelog("source next     : %x", source->next);
}

float getRate(void)
{
  double time_cycles;

  time_cycles = ptime_events.cumul + ptime_world.cumul + ptime_render.cumul + ptime_buffer.cumul;
  if (time_cycles == 0)
    return 5.0;
  return (float) (GuiGetCycles() / time_cycles);
}

bool checkRate(const uint16_t rate)
{
  int ratio = (int) getRate() / rate;
  return (ratio <= 1 || (ratio > 1) && (GuiGetCycles() % ratio) == 1);
}

void statNetwork(void)
{
  if (flog != (FILE *) NULL) {
    closelog(flog);
    printlog();
  }
  fprintf(stderr, "### NET stat ###\n");

  float d = stopTime(&ptime_net);
  float bw = (float) ((sum_bytes_sent + sum_bytes_recvd + 2 * (8 + 20)) * 8) / d;

  if (sum_pkts_sent) {
    fprintf(stderr, "pkts sent       : %d\n", sum_pkts_sent);
    fprintf(stderr, "pkts_rtp sent   : %d\n", sum_pkts_rtp_sent);
    fprintf(stderr, "pkts_rtcp sent  : %d\n", sum_pkts_rtcp_sent);
    fprintf(stderr, "pkts sent/s     : %.0f/s\n", sum_pkts_sent/d);
  }
  if (sum_pkts_recvd) {
    fprintf(stderr, "pkts received   : %d\n", sum_pkts_recvd);
    fprintf(stderr, "pkts received/s : %.0f/s\n", sum_pkts_recvd/d);
    fprintf(stderr, "pkts lost       : %d\n", sum_pkts_lost);
    fprintf(stderr, "pkts lost %%     : %2.2f%%\n", ((double) 100*sum_pkts_lost)/(double) sum_pkts_recvd);
  }
  fprintf(stderr, "bytes sent      : %d\n", sum_bytes_sent);
  fprintf(stderr, "bytes_rtp sent  : %d\n", sum_bytes_rtp_sent);
  fprintf(stderr, "bytes_rtcp sent : %d\n", sum_bytes_rtcp_sent);
  fprintf(stderr, "bytes sent/s    : %.0f/s\n", sum_bytes_sent/d);
  if (sum_bytes_recvd) {
    fprintf(stderr, "bytes received  : %d\n", sum_bytes_recvd);
    fprintf(stderr, "bytes received/s: %.0f/s\n", sum_bytes_recvd/d);
  }
  fprintf(stderr, "pkt max         : %d\n", pkt_max);
  fprintf(stderr, "pkt min         : %d\n", pkt_min);
  if (sum_pkts_sent)
    fprintf(stderr, "bytes/pkt sent  : %d\n", sum_bytes_sent / sum_pkts_sent);
  if (sum_pkts_recvd)
    fprintf(stderr, "bytes/pkt recvd : %d\n", sum_bytes_recvd / sum_pkts_recvd);
  fprintf(stderr, "bw IP+UDP+RTP+PL: %.0f bps\n", bw);

  fprintf(stderr, "### Timings ###\n");
  fprintf(stderr, "session time    : %5.2f s\n", d);
  fprintf(stderr, "init time       : %5.2f s\n", (float) ptime_init.cumul);
  fprintf(stderr, "events time     : %5.2f s\n", (float) ptime_events.cumul);
  fprintf(stderr, "world time      : %5.2f s\n", (float) ptime_world.cumul);
  fprintf(stderr, "render time     : %5.2f s\n", (float) ptime_render.cumul);
  fprintf(stderr, "buffer time     : %5.2f s\n", (float) ptime_buffer.cumul);
  fprintf(stderr, "cycles          : %d\n", GuiGetCycles());
  fprintf(stderr, "cycles/s        : %5.2f/s\n", getRate());

#if 0 //segfault
  fprintf(stderr, "### Sources ###\n");
  Source::dumpAll();
#endif //segfault
  statMemory();
}
