/* ==================================================== ======== ======= *
 *
 *  gui.cc : generic interface and callback functions for the VREng GUI
 *  NOTE: this file should be common to all GUI variants
 *
 *  VREng Project [Elc::2000]
 *  Author: Eric Lecolinet
 *
 *  (C) 1999-2000 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 */

#include <ubit/ubit.hpp>

#include "global.h"
#include "world.h"
#include "keys.h"
#include "wo.h"
#include "user.h"	// User
#include "vnc.h"	// Vnc
#include "cart.h"	// Cart
#include "move.h"	// chaneKey
#include "vrhelp.h"	// HELPSTRING
#include "web.h"	// WEB_TYPE
#include "icon.h"	// ICON_TYPE

#include "net.h"
#include "vgl.h"	// Solid
#include "app.h"
#include "channel.h"

#include "gui.h"
#include "guiImpl.hh"
#include "widgets.hh"


// global variable - initialized by GuiCreateMainWindow
Options *options;

// private variable which contain GUI data
static GUI *gui;


GUI::GUI()
{
  sceneInitialized = false;
  readyAndVisible = false;
  cycles = 0;
  appli = NULL;
  guiWidgets = NULL;
  display = NULL;
#if !defined(WANT_UBIT)
  appContext = NULL;
  toplevel = NULL;
  appWidth = appHeight = 0;
  inputTable[0] = inputTable[1] = NULL;
#endif
  postponedKRmask = 0;
  postponedKRcount = 0;

  glwin = None;
  glvisual = NULL;

  current_solid = NULL; // the solid that is currently selected
  vrengInitCB = NULL;
}

Options::Options()
{
  version = PACKAGE_VERSION;
  url = NULL;
  world = NULL;
  channel = NULL;
  nick = NULL;
  skinf = strdup(DEF_FRONT_GIF);
  skinb = strdup(DEF_BACK_GIF);
  // size of the GL rendering window
  width3D = DEF_WIDTH3D;
  height3D = DEF_HEIGHT3D;
  dbg = 0;
  quality = false;
  flashy = false;
  counters = true;
  reflector = true;
  maxsimcon = DEF_MAXSIMCON;
  cachetime = 3600 * 24 * 3;	// 3 days
}

void GuiInitAndGetOptions(int argc, char *argv[])
{
  options = new Options();

  extern char *optarg;
  int c, v;

  // parses command line and initializes the VREng 'options'
  while ((c = getopt(argc, argv, "?hvq2FCMRD:u:w:c:n:t:")) != -1) {
    switch (c) {
      case '?':
      case 'h':
        fprintf(stderr, "%s\n", HELPSTRING);
        exit(0);
      case 'v':
        printf("%s\n", PACKAGE_VERSION);
        exit(0);
      case 'D':
        options->dbg = debug = atoi(optarg);
        break;
      case 'q':
        options->quality = true;
        options->counters = false;
        break;
      case 'u':
      case 'w':
        options->url = strdup(optarg);
        char chanstr[CHAN_LEN];
        memset(chanstr, 0, sizeof(chanstr));
        Vac::resolveWorldUrl(options->url, chanstr);
        options->channel = strdup(chanstr);
        break;
      case 'c':
        options->channel = strdup(optarg);
        options->reflector = false;
        break;
      case 'F':
        options->flashy = true;
        break;
      case 'M':
        options->reflector = false;
        break;
      case 'R':
        options->reflector = true;
        break;
      case 'C':
        cleanCache(0L);
        break;
      case 'n':
        options->nick = strdup(optarg);
        break;
      case '2':
        options->width3D *= 2;
        options->height3D *= 2;
        break;
      case 't':
        v = atoi(optarg);
	if (v < 0 || v > 365)
          v = 3;
        options->cachetime = v * 3600 * 24;
        break;
    }
  }

  if (options->url == NULL)
    options->url = strdup(DEF_URL_CFG);
  if (options->channel == NULL)
    options->channel = strdup(DEF_VRE_CHANNEL);
  // nickname
  if (options->nick == NULL) {
#if HAVE_GETPWUID
    struct passwd *pwd = getpwuid(getuid());
    if (pwd)
      options->nick = strdup(pwd->pw_name);
    else
      options->nick = strdup("unknown");
#else
    options->nick = strdup("unknown");
#endif
  }

  gui = new GUI();	// create the GUI

  // creates the toplevel
  gui->createToplevel(argc, argv, GuiWidgets::getFallbackOptions());

  // creates the widgets of the GUI and initializes the glwin position
  // NB: the toplevel must have been be created with correct depth and colormap
  gui->guiWidgets = new GuiWidgets(*gui);

  // initializes X vars, X TimeOuts and GLX window
  gui->initX();
}

/** prend une fct. en arg. qui sera lancee apres l'ouverture de la fenetre */
void GuiMainLoop( void (*initCB)(void) )
{
  gui->vrengInitCB = initCB;
  gui->mainLoop();
}

int GuiGetCycles(void)
{
  return gui->cycles;
}

void GuiWriteMessage(const char *mode, const char *from, const char *mess)
{
  // printf("GuiWriteMessage: [%s] %s> %s\n", mode, (from?from:""), mess);
  gui->guiWidgets->writeMessage(mode, from, mess);
}

/** sets the 'busy' cursor */
void GuiSetBusy(bool state)
{
  printf("GuiSetBusy: state=%d\n", state);
}

/** indicates work progress. value from 0 to 100 */
void GuiInProgess(int progress)
{
  printf("GuiInProgess: progress=%d\n", progress);
}

/** NOTE: table_no=0 for tabFd / table_no=1 for tabManagerFd */
void GuiAddInputTable(int table_count, int *table, int table_no)
{
  gui->addInputTable(table_count, table, table_no);
}

void GuiRemoveInputTable(int table_count, int table_no)
{
  gui->removeInputTable(table_count, table_no);
}

// PLD handling cart

UBox * GuiAddCartItem(WObject *po)
{
  if (!po) {
    warning("GuiAddCartItem: NULL item!");
    return NULL;
  }
  notice("Item %s added to cart", NN(po->name.instance_name));
  return gui->guiWidgets->addCartItem(po);
}

void GuiRemoveCartItem(WObject *po, int target)
{
  if (!po) {
    warning("GuiRemoveCartItem: NULL item!");
    return;
  }

  if (po->guip) {
    gui->guiWidgets->removeCartItem((GuiCartItem) po->guip);
    po->guip = NULL;
  }

  switch (target) {
  case CART_DROP:
    notice("Item %s dropped into the %s world", NN(po->name.instance_name),NN(worlds->name));
    worlds->plocaluser->cart->dropObjectIntoWorld(po);
    break;
  case CART_REMOVE:
    notice("Item %s removed from cart", NN(po->name.instance_name));
    worlds->plocaluser->cart->removeObjectFromCart(po);
    break;
  }
}

// Handling User

UBox * GuiAddUser(User *user) 	// when a new user comes in
{
  if (!user) {
    warning("GuiAddUser: NULL user!");
    return NULL;
  }
  notice("Avatar %s joins %s", NN(user->name.instance_name), NN(user->name.world_name));
  return gui->guiWidgets->addUser(user);
}

void GuiRemoveUser(User *user)	// when an user quits
{
  if (!user) {
    warning("GuiRemoveUser: NULL user!");
    return;
  }
  notice("Avatar %s leaves %s", NN(user->name.instance_name), NN(user->name.world_name));
  if (user->guip) {
    gui->guiWidgets->removeUser(user->guip);
    // MS : for Ubit at least, removeUser does a delete on
    // the guip structure. Safer to mark it a NULL than to
    // carry an invalid pointer around.
    user->guip = NULL;
  }
}

void GuiUpdateUser(User *user)
{
  if (!user) {
    warning("GuiUpdateUser: NULL user!");
    return;
  }
  notice("Avatar %s is in %s", NN(user->name.instance_name), NN(user->name.world_name));
  if (user->guip)
    gui->guiWidgets->updateUser(user->guip, user);
}

// Handling World

UBox * GuiAddWorld(World *world, bool isCurrent)
{
  if (!world) {
    warning("GuiAddWorld: NULL world!");
    return NULL;
  }
  // notice("New world %s", NN(world->name));
  return gui->guiWidgets->addWorld(world, isCurrent);
}

void GuiRemoveWorld(World *world)
{
  if (!world) {
    warning("GuiRemoveWorld: NULL world!");
    return;
  }
  // notice("Remove world %s", NN(world->name));
  if (world->guip)
    gui->guiWidgets->removeWorld(world);
}

void GuiUpdateWorld(World *world, bool isCurrent)
{
  if (!world) {
    warning("GuiUpdateWorld: NULL world!");
    return;
  }
  // notice("Update world %s", NN(world->name));
  if (world->guip)
    gui->guiWidgets->updateWorld(world, isCurrent);
}

void GuiRedirectToVnc(Vnc *vnc)
{
  gui->guiWidgets->redirectToVnc(vnc);
}

void GuiLaunchVncConnect(Vnc *vnc)
{
  gui->guiWidgets->launchVncConnect(vnc);
}

void GuiRedirectToVrelet(Vrelet *vrelet)
{
  gui->guiWidgets->redirectToVrelet(vrelet);
}

void GuiRedirectToCarrier(Carrier *carrier)
{
  gui->guiWidgets->redirectToCarrier(carrier);
}

void GuiAlert(const char *message)
{
#if WANT_UBIT
  gui->guiWidgets->alert(message);
#else
  trace(DBG_FORCE, "%s", message);
#endif
}

void GuiLaunchIconDisplay(Icon *icon)
{
  gui->guiWidgets->launchIconDisplay(icon);
}


// CALLBACK FUNCTIONS

void GUI::callVrengAction(int spaction)
{
  struct timeval t;
  gettimeofday(&t, NULL);
  trace(DBG_GUI, "SpecialAction: action=%d", spaction);
  Solid *ps = NULL;	// user
  ps->specialAction(spaction, NULL, t.tv_sec, t.tv_usec);
}

void GUI::changeVrengKey(int key, bool ispressed)
{
  struct timeval t;
  gettimeofday(&t, NULL);
  //printf("GuiProcessKey: PRESS: ts: %d - tus: %d \n", t.tv_sec, t.tv_usec);
  if (key >= MAXKEYS || key < 0)
    return;
  changeKey(key, ispressed, t.tv_sec, t.tv_usec);
}

// with current solid!
static void objectActionCB(int meth)
{
  if (gui->current_solid) {
    struct timeval t;
    gettimeofday(&t, NULL);
    trace(DBG_GUI, "Call method: %d of object %p", meth, gui->current_solid);
    gui->current_solid->specialAction(meth, NULL, t.tv_sec, t.tv_usec);
  }
}

/** returns info about the pointed object but do NOT select it */
Solid* GUI::getPointedObject(int x, int y, ObjInfo *objinfo)
{ 
  Solid *solid = VglGetBufferSelection(x, y);

  char *classname = NULL, *instancename = NULL, *actionnames = NULL;

  if (! solid) {
    objinfo[0].name = "World";
    objinfo[1].name = worlds->name; //World::getCurrentName();
    objinfo[2].name = NULL; // NULL terminaison
    return NULL;
  }

  /* an object was selected */
  trace(DBG_GUI, "Pointed: solid=%p id=%d obj=%p", solid, solid->id, solid->object);
#if 0
  {
    float *m;

    for (int i=0; i<4; i++) {
      for (int j=0; j<4; j++)
        printf("[%d,%d]=%.2f ", i, j, solid->posmat.m[i][j]);
      printf("\n");
    }
#if 0
    printf("\n");
    m = getGLMatrix(&solid->posmat);
    for (int i=0; i<4; i++) {
      for (int j=0; j<4; j++, m++)
	printf("m[%d][%d]=%.2f ", i, j, *m);
      printf("\n");
    }
    printf("\n");
#endif
  }
#endif

  /* get the object hname or avatar name */
  solid->getObjectHumanName(&classname, &instancename, &actionnames);

  if (classname == NULL)
    return NULL;

  objinfo[0].name = classname;
  if (instancename == NULL)
    instancename = "";
 
 trace(DBG_GUI, "Pointed: %s:%s", classname, instancename);
  objinfo[1].name = instancename;   // TO BE COMPLETED

  int index = 0;
  if (actionnames) {
    for (index = 0; index < BUTTONSNUMBER && *actionnames; index++) {
      objinfo[index+2].name = actionnames;
      objinfo[index+2].fun  = objectActionCB;
      objinfo[index+2].farg = index;
      actionnames += HNAME_LEN;	// AWFULL !!!
    }
  }
  objinfo[index+2].name = NULL; // NULL terminated
  //pd solid->setSolidFlashy(true);
  return solid;
}

/** returns info about the pointed object AND selects it */
Solid* GUI::selectPointedObject(int x, int y, ObjInfo *objinfo)
{
  return (gui->current_solid = getPointedObject(x, y, objinfo));
}

/** inform the GUI that a (possibly selected) object was destroyed */
void Solid::clearObjectBar()
{
  if (this == gui->current_solid)
    gui->current_solid = NULL;
  // delete the objectBar and makes it disappear
  gui->guiWidgets->updateObjectInfo(NULL, 0);
}

void GUI::setPref(int tool)	// callback d'un des boutons pref
{
  trace(DBG_GUI, "Selected tool: %d", tool);
  if (tool & AUDIO_MASK) audiotool = tool;
  if (tool & VIDEO_MASK) videotool = tool;
  if (tool & WHITEBOARD_MASK) whiteboardtool = tool;
  if (tool & HTML_MASK) browsertool = tool;
  if (tool & VRML_MASK) vrmltool = tool;
}

void GUI::setAudio(int on)	 // callback du bouton audio
{
  if (on) {
    audioactive = true;
    App::startaudio(World::getChannelName());
  }
  else {
    App::quitaudio();
    audioactive = false;
  }
}

void GUI::setVideo(int on) 	// callback du bouton video
{
  if (on) App::startvideo(World::getChannelName());
  else App::quitvideo();
}

void GUI::setWhiteboard(int on)	// callback du bouton whiteboard
{
  if (on) App::startwhiteboard(World::getChannelName());
  else App::quitwhiteboard();
}

void GUI::setSlidecast(int on)	// callback du bouton slidecast
{
  if (on) App::startslidecast(World::getChannelName());
  else App::quitslidecast();
}

void GUI::setModeler(int on)	// callback du bouton modeler
{
  if (on) App::startmodeler();
  else App::quitmodeler();
}

void GUI::setJmrc(int on)	// callback du bouton jmrc
{ }

void PutInfo::putIcon(const UStr& val)
{
  if (generalActionList[ICON_CREAT][ICON_TYPE].method)
    generalActionList[ICON_CREAT][ICON_TYPE].method(worlds->plocaluser, 
						    (void*)val.chars(), 0, 0);
}

void PutInfo::putIcon()
{
  char infos[BUFSIZ];
  memset(infos, 0, sizeof(infos));
  if (put_url->chars()) {
    strcat(infos, "<url=\"");
    strcat(infos, put_url->chars());
    strcat(infos, "\">&");
  }
  if (put_file->chars()) {
    strcat(infos, "<file=\"");
    strcat(infos, put_file->chars());
    strcat(infos, "\">&");
  }
  if (put_ofile->chars()) {
    strcat(infos, "<ofile=\"");
    strcat(infos, put_ofile->chars());
    strcat(infos, "\">&");
  }
  if (put_name->chars()) {
    strcat(infos, "<name=\"");
    strcat(infos, put_name->chars());
    strcat(infos, "\">&");
  }
  if (put_icon->chars()) {
    strcat(infos, "<icon=\"");
    strcat(infos, put_icon->chars());
    strcat(infos, "\">&");
  }

  if (generalActionList[ICON_CREAT][ICON_TYPE].method)
    generalActionList[ICON_CREAT][ICON_TYPE].method(worlds->plocaluser, infos, 0, 0);
}

void GUI::getIcon(bool status)
{
  if (gui->current_solid && gui->current_solid->object) {
    if (gui->current_solid->object->type == ICON_TYPE) {
      trace(DBG_FORCE, "getIcon:");
    }
    else
      warning("not an icon!");
  }
}

void GUI::openIcon(bool status)
{
  if (gui->current_solid && gui->current_solid->object) {
    if (gui->current_solid->object->type == ICON_TYPE)
#if 1
      if (generalActionList[ICON_OPEN][ICON_TYPE].method)
        generalActionList[ICON_OPEN][ICON_TYPE].method(gui->current_solid->object, NULL, 0, 0);
#else
      App::startbrowser(gui->current_solid->object->name.url, false);
#endif
  }
}

void GUI::removeIcon(bool status)
{
#if 0
  if (gui->current_solid && gui->current_solid->object) {
    if (gui->current_solid->object->type == ICON_TYPE) {
      if (! strcmp(gui->current_solid->object->name.owner, worlds->plocaluser->name.instance_name)) {
        gui->current_solid->object->toDelete();
      }
      else {
        warning("Permission denied, not owner!");
      }
    }
    else {
      warning("not an icon!");
    }
  }
#endif
}

void GUI::quit(int status)
{
  quitVreng(status);
  exit(status);
}

void GUI::back(int)		// callback du bouton Back
{
  World::back();
}

void GUI::forward(int)		// callback du bouton Forward
{
  World::forward();
}

void GUI::addbookmark(int)	// callback du bouton Bookmarks
{
  FILE *fp;
  char bookmark[URL_LEN + CHAN_LEN + 2];
  char buf[URL_LEN + CHAN_LEN + 2];

  sprintf(bookmark, "%s %s\n", worlds->url, worlds->chan);
  if ((fp = fopen(vrengbookmarks, "r")) != NULL) {
    while (fgets(buf, sizeof(buf), fp)) {
      if (!strcmp(buf, bookmark)) {
        fclose(fp);
        return;
      }
    }
    fclose(fp);
  }
  if ((fp = fopen(vrengbookmarks, "a")) != NULL) {
    fputs(bookmark, fp);
    fclose(fp);
  }
}

void GUI::home(int)		// callback du bouton Home
{
  char chan_home_str[CHAN_LEN];
  sprintf(chan_home_str, "%s/%u/%d", universe->groupinitial, universe->portinitial, currentTtl());
  trace(DBG_IPMC, "WO: goto %s at %s", universe->urlinitial, chan_home_str);

  worlds->quit();
  delete Channel::getCurrentChannel();	// delete Channel
  World::newWorld(universe->urlinitial, chan_home_str, true);
  World::setChannelNameAndJoin(chan_home_str);	// join new channel

  if (audioactive)
    App::startaudio(chan_home_str);
}

void GUI::world(int, const char *vre)	// callback world
{
  if (! *vre)
    return;
  char *p, urlvre[URL_LEN], chanstr[CHAN_LEN];
  if ((p = strchr(vre, '/'))) {	// url or path
    strcpy(urlvre, vre);
    if (! App::checkUrl(urlvre))
      return;	// bad url
    if (! Vac::resolveWorldUrl(urlvre, chanstr))
      return;	// url not found
  }
  else {	// worldname
    if (! Vac::getUrlAndChannel(vre, urlvre, chanstr))
      return;	// world not found
  }
  trace(DBG_IPMC, "goto %s at %s", urlvre, chanstr);

  worlds->quit();
  delete Channel::getCurrentChannel();	// delete Channel
  World::newWorld(urlvre, chanstr, true);
  World::setChannelNameAndJoin(chanstr);	// join new channel

  if (audioactive)
    App::startaudio(chanstr);
}

void GUI::help(int)		// callback du bouton Help
{
  char cmd[BUFSIZ];
  sprintf(cmd, "IFS=' '; mozilla -remote 'openURL(http://%s/)' &", DEF_HTTP_SERVER);
  system(cmd);
}

#if NOT_YET
void GUI::pref(int) {		// callback du bouton Pref
}
void GUI::edit(int) {		// callback du bouton Edit
}
void GUI::stats(int) {		// callback du bouton Stats
}
void GUI::views(int) {		// callback du bouton Views
}
#endif

PutInfo::PutInfo()
{
  put_url = &ustr();
  put_file = &ustr();
  put_ofile = &ustr();
  put_name = &ustr();
  put_icon = &ustr();
}
