/* ==================================================== ======== ======= *
 *
 *  xkeys.cc : key management and correspondance with VREng
 *  NOTE: this file should be common to all X-Window GUI variants
 *
 *  VREng Project [Elc::2000]
 *  Author: Eric Lecolinet
 *
 *  (C) 1999-2000 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 */

#include <X11/keysym.h>

#include "global.h"
#include "wo.h"		// WObject
#include "user.h"	// FOVY*
#include "move.h"	// changeKey

#include "gui.h"
#include "guiImpl.hh"
#include "keys.h"


// *** Use this callback function for handling AUTOREPEAT Key event 
// Autorepeat is logically suppressed by annulating corresponding Press 
// and Release events.
// Principle: Release events are postponed and temporarily stored in KRmask
// They are processed when coming back to the RenderingWorkProc of the
// mainLoop if not annulated in the meantime by subsequent Press events.


void GUI::processKey(KeySym keysym, bool press) {

  // converts the X keysym into a Vreng change key (vrkey) and returns
  // a keymask which is an hexa value for marking released keys in the KRmask
  int vrkey;
  long keymask = convertKey(keysym, vrkey);
  if (keymask == 0) return;    // return if null (undefined or not a vrkey)

  if (postponedKRcount < 0) {
    fprintf(stderr, "!negative KRcount => reset\n"); // should never happen!
    postponedKRmask = 0;
    postponedKRcount = 0;
  }

  if (press) {
    if (postponedKRmask & keymask) { 
      // le Press annule le Release correspondant dans le KRmask

      postponedKRmask &= ~keymask;	// remove keymask from KRmask
      postponedKRcount--;		// une touche en moins dans KRlist
      // rien d'autre a faire puisque c'etait juste une anulation
    }
    else {  // traitement normal d'un Press
      struct timeval time;
      gettimeofday(&time, NULL);
      //fprintf(stderr, "KPress change or activate Key( %d ) \n", vrkey);
      if (vrkey >= MAXKEYS || vrkey < 0) {
        return;
      }
      changeKey(vrkey, press, time.tv_sec, time.tv_usec);
    }
  }

  else { // release

    // too many keys stored ==> flush the KRlist
    if (postponedKRcount > KRBUF_MAXCOUNT) {
      fprintf(stderr, "-->too many keys stored => flush: KRmask=%lx\n", postponedKRmask);
      flushPostponedKRs();
    }

    else { // normal case: postpone until we come back to the mainLoop
      // so that this Key Release event can possibly be annulate
      // if there is a subsequent (and corresponding) Key Press event

      postponedKRmask |= keymask;	// add keymask to KRmask

      // add this event to the KRlist of postponed Key Releases
      postponedKRbuf[postponedKRcount].vrkey = vrkey;
      gettimeofday(&(postponedKRbuf[postponedKRcount].time), NULL);
      postponedKRcount++;
    }
  }
}


void GUI::flushPostponedKRs() {
    //fprintf(stderr, "-->postponed release: KRmask=%x\n", gui->KRmask);

  for (int ix = 0; ix < postponedKRcount; ix++) {
    KRKey &k = postponedKRbuf[ix];
    //fprintf(stderr, "-->changeKey( %d )\n", k.vrkey);
    changeKey(k.vrkey, FALSE, k.time.tv_sec, k.time.tv_usec);
  }
  postponedKRmask = 0;
  postponedKRcount = 0;
}


long GUI::convertKey(KeySym keysym, int &vrkey) {
  long keymask = 0;

  switch (keysym) {

  case XK_Up:		// move forward
  case XK_KP_Up:
    keymask = 1 << 0; vrkey = KEY_AV; break;
  case XK_Down:		// move backward
  case XK_KP_Down:
    keymask = 1 << 1; vrkey = KEY_AR; break;
  case XK_Left:		// turn left
  case XK_KP_Left:
    keymask = 1 << 2; vrkey = KEY_GA; break;
  case XK_Right:	// turn right
  case XK_KP_Right:
    keymask = 1 << 3; vrkey = KEY_DR; break;
  case XK_Meta_L:	// left translation 
  case XK_Alt_L:
  //case XK_Control_L:
  case XK_less:
  //pd case XK_Shift_L:
    keymask = 1 << 4; vrkey = KEY_SG; break;
  case XK_Meta_R:	// right translation 
  case XK_Alt_R:
  //case XK_Control_R:
  case XK_greater:
  //pd case XK_Shift_R:
    keymask = 1 << 5; vrkey = KEY_SD; break;
  case XK_Page_Up:	// move up
  case XK_KP_Page_Up:
    keymask = 1 << 11; vrkey = KEY_JU; break;
  case XK_Page_Down:	// move down
  case XK_KP_Page_Down:
    keymask = 1 << 12; vrkey = KEY_JD; break;
  case XK_Insert:	// tilt up
  case XK_KP_Insert:
    keymask = 1 << 6; vrkey = KEY_MT; break;
  case XK_Delete:	// tilt down
  case XK_KP_Delete:
    keymask = 1 << 7; vrkey = KEY_DE; break;
  case XK_Home:		// stand up
  case XK_KP_Home:
    keymask = 1 << 8; vrkey = KEY_HZ; break;
  //pd case XK_KP_Left:
  case XK_l:		// tilt left
    keymask = 1 << 9; vrkey = KEY_TL; break;
  //pd case XK_KP_Right:
  case XK_r:		// tilt right
    keymask = 1 << 10; vrkey = KEY_TR; break;

  case XK_u:
    keymask = 1 << 11 | 1 << 13; vrkey = KEY_JU; break;

  case XK_space:	// accelerators
  case XK_KP_Space:
  case XK_End:
  case XK_KP_End:
    keymask = 1 << 13; vrkey = KEY_VI; break;

  case XK_equal:	// original fovy
    GUI::callVrengAction(FOVYORIGINAL); return 0;
  case XK_minus:	// decrease fovy
    GUI::callVrengAction(FOVYLESS); return 0;
  case XK_plus:		// increase fovy
    GUI::callVrengAction(FOVYGREATER); return 0;

  case XK_period:	// original lspeed
    GUI::callVrengAction(LSPEEDORIGINAL); return 0;
  case XK_s:		// decrease lspeed
    GUI::callVrengAction(LSPEEDLESS); return 0;
  case XK_f:		// increase lspeed
    GUI::callVrengAction(LSPEEDGREATER); return 0;

  case XK_comma:	// original aspeed
    GUI::callVrengAction(ASPEEDORIGINAL); return 0;
  case XK_BackSpace:	// decrease aspeed
    GUI::callVrengAction(ASPEEDLESS); return 0;
  case XK_Tab:		// increase aspeed
    GUI::callVrengAction(ASPEEDGREATER); return 0;

  case XK_b:
    GUI::callVrengAction(BULLETUSER); return 0;
  case XK_d:
    GUI::callVrengAction(DARTUSER); return 0;  // done, don't call on release

  default:
    return 0; // == 0 => undefined key
    break;
  }

  return keymask;
}
