/* ==================================================== ======== ======= *
 *
 *  ubnavig.cc : Ubit Navigator for the VREng GUI
 *
 *  VREng / Ubit Project
 *  Author: Eric Lecolinet
 *  Date:   (rev) 19 March 03
 *
 *  Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *  Please refer to the Ubit GUI Toolkit Home Page for details.
 *
 *  (C) 2002 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 */

#include <ubit/ubit.hpp>

#include "global.h"
#include "wo.h"		// WObject
#include "world.h"
#include "keys.h"
#include "move.h"	// changekey
#include "user.h"	// FOVY*
#include "carrier.h"	// Carrier

#include "net.h"
#include "vgl.h"
#include "app.h"

#include "gui.h"
#include "guiImpl.hh"
#include "widgets.hh"
#include "theme.hh"
#include "vnc.h"	// methodes de VNC

const float MOUSE_LINEAR_ACCEL  = 0.055;
const float MOUSE_ANGULAR_ACCEL = 0.0045;

static ObjInfo objinfo[BUTTONSNUMBER + 6];

/* ==================================================== ======== ======= */
/* ==================================================== ======== ======= */

void GuiWidgets::redirectToVnc(Vnc *_vnc) {
  vnc = _vnc;
}

void GuiWidgets::redirectToVrelet(Vrelet *_vrelet) {
  vrelet = _vrelet;
}

void GuiWidgets::redirectToCarrier(class Carrier *_car) {
  car = _car;
}

// Movements
class Mvt {
  int minuskey, pluskey, fun;
  float accel;
public:
  static Mvt xtrans, ytrans, ztrans, zrot;

  Mvt(int minuskey, int pluskey, int fun, float accel);
  void move(int);
  void stop(); // = move(0)
};

Mvt::Mvt(int _minuskey, int _pluskey, int _fun, float _accel) {
  minuskey = _minuskey;
  pluskey  = _pluskey;
  fun      = _fun;
  accel    = _accel;
}

void Mvt::stop() {
  move(0);
}

void Mvt::move(int speed) {
  struct timeval t;
  gettimeofday(&t, NULL);

  if (speed == 0.0) {
    changeKey(pluskey, FALSE, t.tv_sec, t.tv_usec);
    changeKey(minuskey, FALSE, t.tv_sec, t.tv_usec);
  }
  else {
    Solid *ps = NULL;	// user
    float new_speed;  //MUST be a float!
    if (accel > 0) new_speed = accel * speed;
    else  new_speed = USER_LSPEED;

    if (minuskey) {
      if (new_speed > 0) {
	ps->specialAction(fun, &new_speed, t.tv_sec, t.tv_usec);
	changeKey(minuskey, FALSE, t.tv_sec, t.tv_usec);
	changeKey(pluskey, TRUE, t.tv_sec, t.tv_usec);
      }
      else {
	new_speed = -new_speed;
	ps->specialAction(fun, &new_speed, t.tv_sec, t.tv_usec);
	changeKey(minuskey, TRUE, t.tv_sec, t.tv_usec);
	changeKey(pluskey, FALSE, t.tv_sec, t.tv_usec);
      }
    }
    else {
      ps->specialAction(fun, &new_speed, t.tv_sec, t.tv_usec);
      changeKey(pluskey, TRUE, t.tv_sec, t.tv_usec);
    }
    //if (new_speed > 0) {
      //}
      //else {
      //new_speed = -new_speed;
      //ps->specialAction(fun, &new_speed, t.tv_sec, t.tv_usec);
      //changeKey(minuskey, TRUE, t.tv_sec, t.tv_usec);
      //}
  }
}
//forward, backward
Mvt Mvt::ytrans(0/*KEY_AV*/, KEY_AR, USERSETLSPEED, MOUSE_LINEAR_ACCEL);
//turn left, right
Mvt Mvt::zrot(0/*KEY_GA*/, KEY_DR, USERSETASPEED, MOUSE_ANGULAR_ACCEL);
//move left, right
Mvt Mvt::xtrans(0/*KEY_SG*/, KEY_SD, USERSETLSPEED, MOUSE_LINEAR_ACCEL);
//move down, up
Mvt Mvt::ztrans(0/*KEY_JU*/, KEY_JD, USERSETLSPEED, MOUSE_LINEAR_ACCEL);

static Mvt *mvt1 = null, *mvt2 = null;

/* ==================================================== ======== ======= */
/* ==================================================== ======== ======= */
// Primary Callbacks (on glzone)

void GuiWidgets::setMouseRef(UEvent& e) {
  uint32_t bid = e.getButtons();

  if (bid == UEvent::MButton1) button = 1;
  else if (bid == UEvent::MButton2) button = 2;
  else if (bid == UEvent::MButton3) button = 3;
  else button = 0;

  x_ref = e.getX();
  y_ref = e.getY();
}

void GuiWidgets::mpress(UEvent& e) {
  uint32_t bid = e.getButtons();

  if (bid == UEvent::MButton1) button = 1;
  else if (bid == UEvent::MButton2) button = 2;
  else if (bid == UEvent::MButton3) button = 3;
  else button = 0;

  if (vnc)			// events are redirected to VNC
    vnc->redirectEvent(e.getX(), e.getY(), button);

  else if (car->iscarring)	// events are redirected to Carrier
    car->redirectEvent(e.getX(), e.getY(), button);

  else {			// normal behaviour -> navigation
    Solid *solid = GUI::selectPointedObject(e.getX(), e.getY(), objinfo);

    //new: 19sep03 
    if (solid && solid->object && solid->object->name.url[0])
      selected_object_url->set(solid->object->name.url);
    else
      selected_object_url->clear();

    setMouseRef(e);
    //printf("press x:%d y%d\n", e.getX(), e.getY());

    // VRELET
    // Calculate the click vector and do the click method on the object
    // if the object has that method defined
    if (vrelet && button == 1) {
      if (solid) {
        V3 dir;
        VglClickToVector(e.getX(), e.getY(), &dir);
        solid->doClickMethod(dir);
      }
    }

    if (solid)
      solid->setSolidFlashy(true);

    //openedMenu = the menu that is now opened (and that will be refreshed
    //in the main loop after the OGL scene is rendered)

    //pd openedMenu = updateObjectInfo((solid ? objinfo : null), button);
    openedMenu = updateObjectInfo(objinfo, button);
    if (openedMenu) {
      u_dim ww = 0, hh = 0;

      if (openedMenu == navigMenu) {
	// si c'est navigMenu
	// on calcule sa taille pour pouvoir centrer le menu
	//(NB: il faut forcer l'initialisation graphique du menu car
	// il se peut qu'il n'ait pas deja ete affiche donc que 
	// sa taille soit nulle
	UUpdate upd = UUpdate::all;
	upd.layoutIfNotShown();
	openedMenu->update(upd);
	//printf("ww=%d hh=%d \n", ww, hh);
	openedMenu->getSize(ww, hh);
      }
      // IMPORTANT: il faut afficher le menu dans la glzone, pas dans 
      // la Frame car ce sont des X Windows differentes (a cause d'OGL)
      // et car ce menu est une "softwin" (= directement dessine la
      // ou il est affiche sans utiliser de X Window specifique
      // afin de permetre des effets de transparence)

      openedMenu->move(e, -ww/2, -hh/2);
      openedMenu->open();
    }
  }
}


void GuiWidgets::mrelease(UEvent& e) {
  if (button > 0) stopMotion(e);

  //close the menu when the mouse is released
  if (openedMenu) openedMenu->close(0);
  openedMenu = null;

  button = 0;
  if (vnc) vnc->redirectEvent(e.getX(), e.getY(), button);
 
  else if (gui.current_solid)
    gui.current_solid->setSolidFlashy(false);
}

void GuiWidgets::mdrag(UEvent& e) {
  if (button > 0) motion(e);
}

void GuiWidgets::mmove(UEvent& e) {
  if (vnc) {  //events are redirected to VNC
    vnc->redirectEvent(e.getX(), e.getY(), button);
  }
  else if (button == 0) {
    // mode followMouse continuously indicates object under pointer
    if (followMouseMode) {
      Solid *solid = GUI::getPointedObject(e.getX(), e.getY(), objinfo);

      // btn = 0 ==> the popup menu is NOT created
      updateObjectInfo((solid ? objinfo : null), 0);
    }
  }
}

/* ==================================================== ======== ======= */
// Secondary Callbacks (when menu is open)

void GuiWidgets::startMotion(UEvent& e, Mvt* m1, Mvt *m2) {
  if (mvt1) mvt1->move(0);
  if (mvt2) mvt2->move(0);
  mvt1 = m1;
  mvt2 = m2;

  //le mesu se ferme des l'activation si mouse btn 2
  if (button == 2 && openedMenu) openedMenu->close(0);
}

//default Mvts
void GuiWidgets::startMotion(UEvent& e) {
  startMotion(e, &Mvt::zrot, &Mvt::ytrans);
}

void GuiWidgets::stopMotion(UEvent& e) {
  setMouseRef(e);
  button = 0;
  if (mvt1) mvt1->move(0);
  if (mvt2) mvt2->move(0);
  mvt1 = null;
  mvt2 = null;
}

void GuiWidgets::motion(UEvent& e) {
  int delta_x = e.getX() - x_ref;
  int delta_y = e.getY() - y_ref;
  if (mvt1) mvt1->move(delta_x);
  if (mvt2) mvt2->move(delta_y);
}

/* ==================================================== ======== ======= */

void GuiWidgets::kpress(UEvent& e) {
  if (vnc) {     // all the events are redirected to VNC
    XEvent* ev = e.getXEvent();
    char keyname[256];
    KeySym ks;
    XLookupString(&ev->xkey, keyname, 256, &ks, NULL);
    vnc->redirectEvent(XKeysymToString(ks), true/*is_down*/);
  }
  else {  // normal behaviour
    Solid *ps = NULL;	// user
    // reset speed (NB: ne tient plus compte des controles) !!!!
    struct timeval t;
    gettimeofday(&t, NULL);

    float user_lspeed = USER_LSPEED; //MUST be a float!
    ps->specialAction(USERSETLSPEED, &user_lspeed, t.tv_sec, t.tv_usec);
    float user_aspeed = USER_ASPEED; //MUST be a float!
    ps->specialAction(USERSETASPEED, &user_aspeed, t.tv_sec, t.tv_usec);
    gui.processKey(e.getKeySym(), TRUE);
  }
}

void GuiWidgets::krelease(UEvent& e) {
  if (vnc) {     // all the events are redirected to VNC
    XEvent* ev = e.getXEvent();
    char keyname[256];
    KeySym ks;
    XLookupString(&ev->xkey, keyname, 256, &ks, NULL);
    vnc->redirectEvent(XKeysymToString(ks), false/*is_up*/);
  }
  else // normal behaviour
    gui.processKey(e.getKeySym(), FALSE);
}

// ======================================================================
// ======================================================================
// NAVIGATOR 

UBox* GuiWidgets::makeNavigator(int popup_mode) {
  UVmargin  &vm = uvmargin(0);
  UHmargin  &hm = uhmargin(0);
  UVspacing &vs = uvspacing(0);
  UHspacing &hs = uhspacing(0);

  UBgcolor *bg = popup_mode ? &UBgcolor::none : &UBgcolor::black;

  UTrow &row1 = utrow
    (
     vm + hm + vs + hs 
     + utcell(bg + uhcenter() + uvcenter()
	      + UOn::arm / theme.activeBg
	      + "^"
	      // for menubar
	      + UOn::mpress   / ucall(this, &GuiWidgets::setMouseRef)
	      + UOn::mdrag    / ucall(this, &GuiWidgets::mdrag)
	      + UOn::mrelease / ucall(this, &GuiWidgets::mrelease)
	      // for all (and after mpress)
	      + UOn::arm / ucall(this, (Mvt*)0, &Mvt::ztrans, &GuiWidgets::startMotion)
	      )
     + utcell(bg + vm + hm                 //FORWARD
	      + UOn::arm / theme.activeBg
	      + theme.Up
	      + UOn::mpress   / ucall(this, &GuiWidgets::setMouseRef)
	      + UOn::mdrag    / ucall(this, &GuiWidgets::mdrag)
	      + UOn::mrelease / ucall(this, &GuiWidgets::mrelease)
	      + UOn::arm      / ucall(this, &GuiWidgets::startMotion)
	      )
     + utcell(bg + uhcenter() + uvcenter()
	      + UOn::arm / theme.activeBg
	      + "m"
	      )
     );

  UTrow &row2 = utrow
    (
     vm + hm + vs + hs 
     + utcell(bg + vm + hm             //LEFT
	      + UOn::arm / theme.activeBg+ UMode::canArm
	      + theme.Left
	      // for menubar
	      + UOn::mpress   / ucall(this, &GuiWidgets::setMouseRef)
	      + UOn::mdrag    / ucall(this, &GuiWidgets::mdrag)
	      + UOn::mrelease / ucall(this, &GuiWidgets::mrelease)
	      // for all (and after mpress)
	      + UOn::arm      / ucall(this, &GuiWidgets::startMotion)
	      )
     + utcell(bg + uhcenter() + uvcenter()
	      + " "
	      + UOn::mpress   / ucall(this, &GuiWidgets::setMouseRef)
	      + UOn::mdrag    / ucall(this, &GuiWidgets::mdrag)
	      + UOn::mrelease / ucall(this, &GuiWidgets::mrelease)
	     )
     + utcell(bg + vm + hm               //RIGHT
	      + UOn::arm / theme.activeBg + UMode::canArm
	      + theme.Right
	      + UOn::mpress   / ucall(this, &GuiWidgets::setMouseRef)
	      + UOn::mdrag    / ucall(this, &GuiWidgets::mdrag)
	      + UOn::mrelease / ucall(this, &GuiWidgets::mrelease)
	      + UOn::arm      / ucall(this, &GuiWidgets::startMotion)
	      )
    );

 UTrow &row3 = utrow
   (
    vm + hm + vs + hs 
    + utcell(bg + uhcenter() + uvcenter()
	     + UOn::arm / theme.activeBg
	     + "<"
	     + UOn::mpress   / ucall(this, &GuiWidgets::setMouseRef)
	     + UOn::mdrag    / ucall(this, &GuiWidgets::mdrag)
	     + UOn::mrelease / ucall(this, &GuiWidgets::mrelease)
	     + UOn::arm / ucall(this, &Mvt::xtrans, (Mvt*)0, &GuiWidgets::startMotion)
	     )
    + utcell(bg + vm + hm        //BACKWARD
	     + UOn::arm / theme.activeBg
	     + theme.Down
	     + UOn::mpress   / ucall(this, &GuiWidgets::setMouseRef)
	     + UOn::mdrag    / ucall(this, &GuiWidgets::mdrag)
	     + UOn::mrelease / ucall(this, &GuiWidgets::mrelease)
	     + UOn::arm      / ucall(this, &GuiWidgets::startMotion)
	     )
    + utcell(bg + uhcenter() + uvcenter()
	     + UOn::arm / theme.activeBg
	     + ">"
	     + UOn::mpress   / ucall(this, &GuiWidgets::setMouseRef)
	     + UOn::mdrag    / ucall(this, &GuiWidgets::mdrag)
	     + UOn::mrelease / ucall(this, &GuiWidgets::mrelease)
	     + UOn::arm      / ucall(this, &Mvt::xtrans, (Mvt*)0, &GuiWidgets::startMotion)
	     )
    );

 return &utable
   ( 
    vs + hs
    + UBorder::none + UFont::bold
    + UBgcolor::none + UColor::white
    + row1
    + row2
    + row3
    );
}
