/* ==================================================== ======== ======= *
 *
 *  ubwidgets.cc : Ubit Widgets for the VREng GUI
 *
 *  VREng / Ubit Project
 *  Author: Eric Lecolinet
 *  Date:   (rev) 12 May 03
 *
 *  Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *  Please refer to the Ubit GUI Toolkit Home Page for details.
 *
 *  (C) 2002 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 */

#include <iostream>
using namespace std;

#include <ubit/ubit.hpp>

#include "global.h"
#include "wo.h"		// WObject
#include "world.h"
#include "keys.h"
#include "move.h"	// changeKey
#include "user.h"	// FOVY*
#include "vnc.h"
#include "cart.h"	// CART_DROP
#include "icon.h"	// Icon

#include "net.h"
#include "vgl.h"
#include "app.h"

#include "gui.h"
#include "guiImpl.hh"
#include "widgets.hh"
#include "theme.hh"

#include "landmark.h"	// buildDialog

#include "README.h"
#include "COPYRIGHT.h"
#include "COPYING.h"
#include "ChangeLog.h"
#include "TODO.h"


WTheme GuiWidgets::theme;    // the Theme of the GUI

// No Fallbacks for the Ubit version
char** GuiWidgets::getFallbackOptions() {
  return NULL;
}

/* ==================================================== ======== ======= */
/* ==================================================== ======== ======= */

GuiWidgets::GuiWidgets(GUI &_gui) : 
  gui(_gui),      // points to the containing GUI class
  selected_object_url(new UStr())
{
  //Initialise le theme
  //NB: ca ne peut pas etre fait par le constructeur de WTheme
  // -1- parce qu'on veut pouvoir changer le theme dynamiquement
  //     en appelant une fonction
  // -2- car les initialisations dans setXXXStyle font reference
  //     a des constantes predefinies qui ne sont peut-etre pas
  //     encore initialisees quand static WTheme theme est evalue
  if (options->flashy)
    theme.setFlashyStyle();
  else
    theme.setClassicStyle();

  alertBox = null;
  vnc = null;
  vnc_dialog = null;
  file_dialog = null;
  followMouseMode = false;

  // size of the GL rendering zone
  glzoneWidth  = &uwidth(options->width3D);
  glzoneHeight = &uheight(options->height3D);

  put_info = new PutInfo();

  UTable &puturl_table = utable
    (
     UBgcolor::grey
     + utrow(utcell(UFont::bold
                    + UColor::black
                    + "Url"   
                    + UFont::normal
                    + " (required)")   
             + utcell(utextbox(uedit()
                      + uwidth(300) // *new UWidth(300)
                      + put_info->put_url)))
     + utrow(utcell(UFont::bold
                    + UColor::black
                    + "Alias"
                    + UFont::normal
                    + " (short name)")   
             + utcell(utextbox(uedit()
                      + uwidth(300)
                      + put_info->put_name)))
     + utrow(utcell(UFont::bold
                    + UColor::black
                    + "Icon"
                    + UFont::normal
                    + " (optional)")   
             + utcell(utextbox(uedit()
                      + uwidth(300)
                      + put_info->put_icon)))
    );

  UTable &putfile_table = utable
    (
     UBgcolor::grey
     + utrow(utcell(UFont::bold
                    + UColor::black
                    + "Input File"   
                    + UFont::normal
                    + " (required)")   
             + utcell(utextbox(uedit()
                      + uwidth(300)
                      + put_info->put_file)))
     + utrow(utcell(UFont::bold
                    + UColor::black
                    + "Output File"   
                    + UFont::normal
                    + " (public location)")   
             + utcell(utextbox(uedit()
                      + uwidth(300)
                      + put_info->put_ofile)))
     + utrow(utcell(UFont::bold
                    + UColor::black
                    + "Alias"
                    + UFont::normal
                    + " (short name)")   
             + utcell(utextbox(uedit()
                      + uwidth(300)
                      + put_info->put_name)))
     + utrow(utcell(UFont::bold
                    + UColor::black
                    + "Icon"   
                    + UFont::normal
                    + " (optional)")   
             + utcell(utextbox(uedit()
                      + uwidth(300)
                      + put_info->put_icon)))
    );

  UBox &put_control = uhbox
    (
     uhcenter() + UFont::bold 
     + ubutton("   Put   " 
               + ucall(put_info, &PutInfo::putIcon)
               + ucloseWin()
              )
     + ubutton("Cancel" + ucloseWin())
    );

  puturl_dialog = &udialog
    (
     utitle("Put new Url document")
     + uhcenter() + uvcenter()
     + UBgcolor::black
     + puturl_table
     + put_control
    );

  putfile_dialog = &udialog
    (
     utitle("Put new File document")
     + uhcenter() + uvcenter()
     + UBgcolor::black
     + putfile_table
     + put_control
    );

  UMenu *file_menu = &umenu
    (
     uhbox(UBorder::shadowOut)
     + ubutton(UColor::navy
               + UFont::bold
               + "Put & Publish Url"
               + puturl_dialog
               )
     + uhbox(UBorder::shadowOut)
     + ubutton(UColor::navy
               + UFont::bold
               + "Put & Publish File"
               + putfile_dialog
               )
#if 0 //pdicon
     + uhbox(UBorder::shadowOut)
     + ubutton(UColor::navy
               + UFont::bold
               + "Get & Save"
               //+ get_dialog
               //+ ucall(get_dialog, &GuiWidgets::getIcon)
               )
     + uhbox(UBorder::shadowOut)
     + ubutton(UColor::navy
               + UFont::bold
               + "Open"
               + ucall(true, &GUI::openIcon)
               )
     + uhbox(UBorder::shadowOut)
     + ubutton(UColor::navy
               + UFont::bold
               + "Remove"
               + ucall(true, &GUI::removeIcon)
               )
#endif //pdicon
     + uhbox(UBorder::shadowOut)
     + ubutton(UPix::stop
               + "Quit"
               + ucall(0, &GUI::quit)
               )
    );

  UMenu *tool_menu = &umenu
    (
     ucheckbox("Audio"
	       + UOn::select   / ucall(TRUE, &GUI::setAudio)
	       + UOn::unselect / ucall(FALSE, &GUI::setAudio)
	       )
     + ucheckbox("Video"
	         + UOn::select   / ucall(TRUE, &GUI::setVideo)
		 + UOn::unselect / ucall(FALSE, &GUI::setVideo)
		 )
     + ucheckbox("Whiteboard"
		 + UOn::select   / ucall(TRUE, &GUI::setWhiteboard)
		 + UOn::unselect / ucall(FALSE, &GUI::setWhiteboard)
		 )
     + ucheckbox("Slidecast"
		 + UOn::select   / ucall(TRUE, &GUI::setSlidecast)
		 + UOn::unselect / ucall(FALSE, &GUI::setSlidecast)
		 )
     + ucheckbox("Jmrc"
		 + UOn::select   / ucall(TRUE, &GUI::setJmrc)
		 + UOn::unselect / ucall(FALSE, &GUI::setJmrc)
		 )
     + ucheckbox("Modeler"
                 + UOn::select   / ucall(TRUE, &GUI::setModeler)
                 + UOn::unselect / ucall(FALSE, &GUI::setModeler)
                 )
    );

  URadioSelect
    &pref1 = uradioSelect(),
    &pref2 = uradioSelect(),
    &pref3 = uradioSelect(),
    &pref5 = uradioSelect(),
    &pref6 = uradioSelect();

  UMenu *setting_menu = &umenu
    ( UColor::navy + UFont::bold
      + uhbox("Audio            : " + UFont::normal
              + ucheckbox("Rat" + UMode::selected + pref1 
	                  + UOn::select / ucall(RAT_TOOL, &GUI::setPref))
              + ucheckbox("Vat" + pref1 
	                  + UOn::select / ucall(VAT_TOOL, &GUI::setPref))
              + ucheckbox("Fphone" + pref1 
	                  + UOn::select / ucall(FPHONE_TOOL, &GUI::setPref))
             )
     + uhbox("Video            : " + UFont::normal
             + ucheckbox("Vic" + UMode::selected
		         + UOn::select / ucall(VIC_TOOL, &GUI::setPref))
            )
     + uhbox("Whiteboard : " + UFont::normal
             + ucheckbox("Wb" + UMode::selected + pref2
	                 + UOn::select / ucall(WB_TOOL, &GUI::setPref))
             + ucheckbox("Wbd" + pref2
		         + UOn::select / ucall(WBD_TOOL, &GUI::setPref))
             + ucheckbox("Nte" + pref2 
		         + UOn::select / ucall(NTE_TOOL, &GUI::setPref))
            )
     + uhbox("Browser      : " + UFont::normal
             + ucheckbox("Mozilla" + UMode::selected + pref3 
		         + UOn::select / ucall(MOZILLA_TOOL, &GUI::setPref))
             + ucheckbox("Netscape" + pref3 
		         + UOn::select / ucall(NETSCAPE_TOOL, &GUI::setPref))
             + ucheckbox("mMosaic" + pref3 
		         + UOn::select / ucall(MMOSAIC_TOOL, &GUI::setPref))
            )
     + uhbox("Modeler       : " + UFont::normal
             + ucheckbox("Vred" + UMode::selected + pref5
                         + UOn::select / ucall(VRED_TOOL, &GUI::setPref))
             + ucheckbox("Vrem" + pref5
                         + UOn::select / ucall(VREM_TOOL, &GUI::setPref))
            )
     + uhbox("Session       : " + UFont::normal
             + ucheckbox("Ssh" + UMode::selected + pref6
		         + UOn::select / ucall(SSH_TOOL, &GUI::setPref))
             + ucheckbox("Telnet" + pref6
		         + UOn::select / ucall(TELNET_TOOL, &GUI::setPref))
            )
    );


  UBox *source_box = &uvbox();
  UDialog& source_dialog = udialog
          (utitle("World source")
           + UBgcolor::white
           + ulabel("World source")
           + uvflex()
           + uscrollpane(uheight(400) + uwidth(600) + source_box)
           + ubutton(UFont::bold + uhcenter() + " Cancel " + ucloseWin())
	  );

  UBox *worlds_box = &uvbox();
  UDialog& worlds_dialog = udialog
          (utitle("Worlds list")
           + ulabel("Worlds list")
           + uvflex()
           + uscrollpane(uheight(300) + uwidth(440) + worlds_box)
           + ubutton(UFont::bold + uhcenter() + " Cancel " + ucloseWin())
	  );

  UBox *prefs_box = &uvbox(UBgcolor::white + uhmargin(2) + uvmargin(2));
  UDialog& prefs_dialog = udialog
          (utitle("Preferences")
           + ulabel("Preferences: " + UColor::red + vrengprefs)
           + uvflex()
           + uedit()
           + uscrollpane(uheight(300) + uwidth(400) + prefs_box)
           + ubutton(UFont::bold + uhcenter() + " Cancel " + ucloseWin())
	  );
  FILE *fpprefs;
  char bufprefs[128];
  if ((fpprefs = fopen(vrengprefs, "r")) != NULL) {
    while (fgets(bufprefs, sizeof(bufprefs), fpprefs)) {
      if (isalpha(*bufprefs))
        prefs_box->add(uitem(UColor::red + UFont::bold + bufprefs), false);
      else
        prefs_box->add(uitem(UColor::black + bufprefs), false);
    }
    prefs_box->update();
    fclose(fpprefs);
  }

  UDialog *landmark_dialog = Landmark::buildDialog();

  UMenu *view_menu = &umenu
    (
     uhbox(UBorder::shadowOut)
     + ubutton(UColor::navy
               + UFont::bold
               + "Source"
               + source_dialog
               + ucall(source_box, &World::sourceDialog)
               )
     + uhbox(UBorder::shadowOut)
     + ubutton(UColor::navy
               + UFont::bold
               + "Worlds"
               + worlds_dialog
               + ucall(worlds_box, &World::worldsDialog)
               )
     + uhbox(UBorder::shadowOut)
     + ubutton(UColor::navy
               + UFont::bold
               + "Prefs"
               + prefs_dialog
               )
     + uhbox(UBorder::shadowOut)
             + ubutton(UColor::navy
             + UFont::bold
             + "Axis/Grid"
             + *landmark_dialog
             )
     );

  UBox *readme_box = &uflowbox(uhmargin(2) + uvmargin(2) + README);
  UDialog& readme_dialog = udialog
    (
     utitle("README") + ulabel("README ")
     + uvflex()
     + uscrollpane(uheight(400) + uwidth(500) + readme_box)
     + ubutton(UFont::bold + uhcenter() + " Cancel " + ucloseWin())
     );

  UBox *copyright_box = &uflowbox(uhmargin(2) + uvmargin(2) + COPYRIGHT);
  UDialog& copyright_dialog = udialog
    (
     utitle("COPYRIGHT") + ulabel("COPYRIGHT ")
     + uvflex()
     + uscrollpane(uheight(300) + uwidth(500) + copyright_box)
     + ubutton(UFont::bold + uhcenter() + " Cancel " + ucloseWin())
     );

  UBox *license_box = &uflowbox(uhmargin(2) + uvmargin(2) + COPYING);
  UDialog& license_dialog = udialog
    (
     utitle("LICENSE") + ulabel("LICENSE ")
     + uvflex()
     + uscrollpane(uheight(300) + uwidth(500) + license_box)
     + ubutton(UFont::bold + uhcenter() + " Cancel " + ucloseWin())
     );

  UBox *changelog_box = &uflowbox(uhmargin(2) + uvmargin(2) + ChangeLog);
  UDialog& changelog_dialog = udialog
    (
     utitle("ChangeLog") + ulabel("ChangeLog ")
     + uvflex()
     + uscrollpane(uheight(400) + uwidth(500) + changelog_box)
     + ubutton(UFont::bold + uhcenter() + " Cancel " + ucloseWin())
     );

  UBox *todo_box = &uflowbox(uhmargin(2) + uvmargin(2) + TODO);
  UDialog& todo_dialog = udialog
    (
     utitle("TODO") + ulabel("TODO ")
     + uvflex()
     + uscrollpane(uheight(300) + uwidth(500) + todo_box)
     + ubutton(UFont::bold + uhcenter() + " Cancel " + ucloseWin())
     );

  UMenu *help_menu = &umenu
    (
     uhbox(UBorder::shadowOut)
     + ubutton(UColor::navy + UFont::bold + "Home Page" + ucall(0, &GUI::help))
     + uhbox(UBorder::shadowOut)
     + ubutton(UColor::navy + UFont::bold + "README" + readme_dialog)
     + uhbox(UBorder::shadowOut)
     + ubutton(UColor::navy + UFont::bold + "COPYRIGHT" + copyright_dialog)
     + uhbox(UBorder::shadowOut)
     + ubutton(UColor::navy + UFont::bold + "LICENSE" + license_dialog)
     + uhbox(UBorder::shadowOut)
     + ubutton(UColor::navy + UFont::bold + "ChangeLog" + changelog_dialog)
     + uhbox(UBorder::shadowOut)
     + ubutton(UColor::navy + UFont::bold + "TODO" + todo_dialog)
     );

#if NOT_YET
  UMenu *edit_menu =
    &umenu(ucheckbox("Preferences"
		     + UOn::select / ucall(TRUE, &GUI::pref)
		     + UOn::unselect / ucall(FALSE, &GUI::pref))
	  );
  UMenu *views_menu =
    &umenu(ucheckbox("From Top"
		     + UOn::select / ucall(TRUE, &GUI::views)
		     + UOn::unselect / ucall(FALSE, &GUI::views))
	  );
#endif

  UBox *bookmark_box = &uvbox();
  UMenu *bookmark_menu = &umenu
    (
     UBgcolor::none
     + uscrollpane(uheight(100) + UBgcolor::none + bookmark_box)
     );
  
  bookmark_box->add
    (
     ubutton(UBgcolor::none + "Add Bookmark" 
	     + ucall(TRUE, &GUI::addbookmark))
     );

  FILE *fpbookmark = NULL;
  char bufbookmark[URL_LEN + CHAN_LEN + 2];

  if ((fpbookmark = fopen(vrengbookmarks, "r")) != NULL) {
    while (fgets(bufbookmark, sizeof(bufbookmark), fpbookmark)) {
      bookmark_box->add( uitem(UBgcolor::none
                               + UColor::yellow
                               + UOn::enter/UColor::red
                               + UOn::arm/UColor::navy
                               + bufbookmark),
                         false
                       );
    }
    fclose(fpbookmark);
  }

  // ===== menubar ====== 

  UBox& menubar = umenubar
    ( 
     UFont::bold + UFont::x_large
     + uscale(-1)
     + ubutton(*theme.File      + *file_menu)
     + ubutton(*theme.Back1     + ucall(0, &GUI::back))
     + ubutton(*theme.Forward1  + ucall(0, &GUI::forward))
     + ubutton(*theme.Home1     + ucall(0, &GUI::home))
     + ubutton(*theme.Tools     + *tool_menu)
     + ubutton(*theme.Settings  + *setting_menu)
     + ubutton(*theme.View      + *view_menu)
     + ubutton(*theme.Bookmarks + *bookmark_menu)
     + uhflex() + ulabel("") + uright() 
     + ubutton(*theme.Help      + *help_menu)
    );

  // ===== toolbar & infobox ====== 

  infobox = &ulinkbutton
    (
     UBorder::none //UBorder::etchedIn
     + UFont::bold + UFont::italic
     + UPix::ray
     + UOn::idle/UColor::orange + " Welcome to VREng "
     + UPix::ray
     );

  toolbar = &ubar
    ( 
     *theme.toolbarBg + UBorder::etchedIn //UBorder::none
     + uheight(UHeight::KEEP_SIZE) + uwidth(UWidth::KEEP_SIZE)
     + infobox
    );

  // ===== worlds, users and basket ====== 

  cartWidget = &uvbox(uwidth(LIST_WIDTH) + *theme.listBg);
  worlds     = &uvbox(uwidth(LIST_WIDTH) + *theme.listBg);
  users      = &uvbox(uwidth(LIST_WIDTH) + *theme.listBg);

  UBox& rightPanel = uvbox
    ( 
     utop() 
     + ulabel(uvmargin(3) + UFont::bold + *theme.World + "Worlds")
     + uvflex()
     + (worldsPane = 
	&uscrollpane(true, true, UBorder::shadowIn + worlds)
       )
     
     + utop() 
     + ulabel(uvmargin(3) + UFont::bold + UPix::cross + "Avatars")
     + uvflex()
     + (usersPane = 
	&uscrollpane(true, false, UBorder::shadowIn + users)
       )

     + utop()
     + ulabel(uvmargin(3) + UFont::bold +  UPix::folder + "Basket")
     + uvflex()
     + (cartPane =
        &uscrollpane(true, true, UBorder::shadowIn + cartWidget)
       )
    );
  
  // ===== Short messages =====

  messages = &uvbox
    (
     //*theme.messBg 
     UBgcolor::black
     + uleft()
     + ulabel( uhcenter() + UFont::bold + UColor::orange
	       + "Have fun with VREng ;-)")
    );
  
  messagesPane = &uscrollpane
    (
     true, true,
     uheight(70) + UBgcolor::grey //+ UBorder::etchedOut
     + messages
    );

  // ===== Text Entry =====

  entryStr = &ustr("");
  entry = &utextbox
    (
     //*theme.entryBg 
     UBgcolor::grey + UColor::navy
     //+ UFont::large
     + uedit() + USymbol::right + " "
     + UOn::action / ucall(this, &GuiWidgets::entryAction)
     + entryStr
    );


  // ===== OpenGL drawing zone  ====== 

  // UIncrust = a True X Window that is "incrusted" in the Ubit Window
  // (this is necessary for displaying OpenGL data)
  glzone = &uincrust
    (
     UBgcolor::none + glzoneWidth + glzoneHeight
     + UOn::mmove    / ucall(this, &GuiWidgets::mmove)
     + UOn::mdrag    / ucall(this, &GuiWidgets::mdrag)
     + UOn::mpress   / ucall(this, &GuiWidgets::mpress)
     + UOn::mrelease / ucall(this, &GuiWidgets::mrelease)
     + UOn::kpress   / ucall(this, &GuiWidgets::kpress)
     + UOn::krelease / ucall(this, &GuiWidgets::krelease)
    );

  // ===== navigation, info menus, control panel ====== 

  // created by updateInfo:
  infoMenu = null;

  //the menu that is currently opened (and that will be refreshed
  //in the main loop after the OGL scene is rendered)
  openedMenu = null;

  // shared navigation box
  //navigator = makeNavigator();

  // the navigation menu (now created in advance)
  navigMenu = &upopmenu
    (
     UBgcolor::none + UBorder::none + UOrient::horizontal
     + makeNavigator(true)
     );

  // NOTE: ce menu est dans la glzone, pas dans la Frame car ce sont 
  // des X Windows differentes (a cause d'OGL) et car ce menu est une 
  // "softwin" (cf: navigMenu->setSoftwinMode()) qui est directement 
  // dessine la ou il est affiche sans utiliser de X Window specifique
  // afin de permetre des effets de transparence)

  navigMenu->setSoftwinMode(true);
  glzone->add(navigMenu);

  // creates the control panel (NB: uses navigator!)
  controlPanel = makeControlPanel();

  // ===== Main Frame ======

  frame = &uframe
    (
     *theme.bg + *theme.fg
     + uhflex()
     + utop()
     + menubar
     + toolbar
     + uvflex()
     + uhbox(uhflex() + uvflex()
	     + uvbox(uhflex() + uvflex() + glzone)
	     + uright() + rightPanel
	    )
     + ubottom()
     + controlPanel
     + UOn::umessage / ucall(this, &GuiWidgets::processMessage)
    );
  
  // adds the main frame to the UAppli
  gui.appli->add(frame);

  // shows the main win
  frame->show();

  glzone->show();
}

/* ==================================================== ======== ======= */
/* ==================================================== ======== ======= */
// updates object info (infobox in the toolbar and contextual info menu)

UMenu* GuiWidgets::updateObjectInfo(ObjInfo *objinfo, int btn){
  UMenu *which_menu = null;

  if (infobox) delete infobox;
  infobox = null;
  if (infoMenu) delete infoMenu;
  infoMenu = null;

  // show the navigation menu if left button was pressed
  if (btn == 1) which_menu = navigMenu;

  if (objinfo) {
    // NB: objinfo[0].name = class_name; objinfo[1].name = instance_name;

    // create and show the contextual info menu if right button
    if (btn == 3) {
      infoMenu = &upopmenu
	(
	 UBgcolor::none 
	 + uvspacing(0)
	 + ulabel(UBgcolor::navy + UColor::white + UFont::bold 
		  + USymbol::right // + UPix::right 
		  + objinfo[0].name + " " + UColor::yellow + objinfo[1].name
		 )
	);
      // pseudo-menu transparent qui est directement dessine dans GLzone
      // avantage: meme quand la scene bouge l'effet de transprence reste
      // preserve (mais attention: il faut constamment reafficher le menu
      // dans la main loop sinon il serait efface par OpenGL)
      infoMenu->setSoftwinMode(true);
      glzone->add(infoMenu);
      which_menu = infoMenu;
    }

    // creates the new updated infobox
    infobox = &uhbox
      (
       UBorder::none
       + ulabel(*theme.objectNameFg + UFont::bold 
		+ USymbol::right + objinfo[0].name + " "
                + UColor::yellow + objinfo[1].name
               )
      );

    // add buttons to infobox and infoMenu
    for (ObjInfo *poi = objinfo + 2; poi->name != NULL; poi++) {

      UButton *but1 = &ubutton
	(
	 UBorder::none
	 + UOn::idle/UBgcolor::none
	 + UColor::orange
	 + poi->name
	);
      // add callback function
      if (poi->fun) but1->add(UOn::action / ucall(poi->farg, poi->fun));
      // add to box & menu
      infobox->add(but1);

      if (infoMenu) {
	UBox *but2 = &uitem
	  (
	   uvmargin(0)
	   + UBgcolor::none
	   + *theme.actionNameFg
	   + UFont::bold
	   + poi->name
          );
	if (poi->fun)
          but2->add(UOn::action / ucall(poi->farg, poi->fun));
	infoMenu->add(but2);
      }
    }

    // put infobox at the beginning of toolbar
    toolbar->add(infobox, 0);
  }

  // update graphics
  toolbar->update();
  return which_menu;  //can be null!
}

/* ==================================================== ======== ======= */
/* ==================================================== ======== ======= */
// External API

void GuiWidgets::entryAction(UEvent&) {
  const char *mess = entryStr->chars();
  if (! mess) return;
  if (! isalnum(*mess)) return;

  // display in message zone
  writeMessage("chat", options->nick, mess);
  // send to World Management 
  User::userWriting(mess);

  // clear input zone 
  entryStr->set("");
}

void GuiWidgets::resizeGLZone(int width, int height) {
  glzoneWidth->set(width);
  glzoneHeight->set(height);
  frame->update();
}

// chat, warning, notice...
void GuiWidgets::writeMessage(const char *mode, 
			      const char *from, const char *mess) {
  char *prefix = NULL;
  UColor *prefix_color = null, *mess_color = null;
  
  if (!mess) mess = ""; //securite
  if (from) {
    prefix = new char[strlen(from) + 15];
    sprintf(prefix, "Chat from %s> ", from);  
    prefix_color = &UColor::white;
    mess_color   = &UColor::white;
  }
  else if (mode) {
    prefix = new char[strlen(mode) + 5];
    sprintf(prefix, "%s> ", mode);
    prefix_color = &UColor::red;
    mess_color   = &UColor::orange;
  }
  else {
    prefix = new char[5];
    sprintf(prefix, "> ");
    mess_color = &UColor::black;
    mess_color = &UColor::green;
  }
  messages->add(uhbox(UFont::bold + prefix_color + prefix 
		      + UFont::normal +mess_color + mess));
  delete[] prefix;
  messages->update();
  messagesPane->getVScrollbar()->setValue(100.);
}

//
// Handling cart items (PLD)
//
void GuiWidgets::updateCartItemInfo(WObject *po)
{
  if (infobox) delete infobox;
  infobox = NULL;

  // creation de l'infobox de l'objet selectionn dans le panier
  infobox = &uhbox
      (
       UBorder::none
       + ulabel(*theme.objectNameFg
                + UFont::bold
                + USymbol::right
                + po->name.class_name
                + " " + po->name.instance_name
                )
       + &ubutton
                (
                UBorder::none
                + UOn::idle/UBgcolor::none
                + UColor::orange
                + "Drop"
                + ucall(po, CART_DROP, GuiRemoveCartItem)
                )
       + &ubutton
                (
                UBorder::none
                + UOn::idle/UBgcolor::none
                + UColor::orange
                + "Destroy"
                + ucall(po, CART_REMOVE, GuiRemoveCartItem)
                )
      );
  // put infobox at the beginning of toolbar
  toolbar->add(infobox, 0);

  // update graphics
  toolbar->update();
}

GuiCartItem GuiWidgets::addCartItem(WObject *po) {
  UBox *gu = &ulinkbutton();

  // delete toolbar's infobox
  if (infobox) {
    toolbar->remove(infobox, 0);
    delete infobox;
  }
  toolbar->update();
  infobox = NULL;

  // create item's box
  gu->addlist
    (
     UFont::bold + UColor::navy
     + ustr(po->name.instance_name)
     + ucall(this, po, &GuiWidgets::updateCartItemInfo)
     );
  cartWidget->add(gu, 2);
  cartWidget->update();

  return gu;
}

void GuiWidgets::removeCartItem(GuiCartItem gu) {
  // delete toolbar's infobox
  if (infobox) {
    toolbar->remove(infobox, true);
    //pd delete infobox;	//pd segfault in UBrick::removeFromParents
  }
  toolbar->update();
  infobox = NULL;

  // delete item's box
  cartWidget->remove(gu, true);
  cartWidget->update();
}


/* ==================================================== ======== ======= */

static void setUser(UBox *gu, User *user){
  gu->addlist
    (
     UFont::bold + UColor::navy 
     + ustr(user->name.instance_name)
     + umenu(UFont::bold
	     + uhbox(" Name: "
		     + UFont::normal + UColor::navy + user->name.instance_name)
	     + uhbox(" World: " 
		     + UFont::normal + UColor::navy + user->name.world_name)
	     + uhbox(" Email: "
		     + UFont::normal + UColor::navy + user->email)
	     + uhbox(" Cname: "
		     + UFont::normal + UColor::navy + user->rtcpname)
	     + uhbox(" Vreng: "
		     + UFont::normal + UColor::navy + user->tool)
	     + uhbox(" Web: "
		     + UFont::normal + UColor::navy + user->web)
	    )
    );
}

UBox *GuiWidgets::addUser(User *user) {
  UBox *gu = &ulinkbutton();
  setUser(gu, user);
  users->add(gu, 2);
  users->update();
  return gu;
}

void GuiWidgets::updateUser(UBox *gu, User *user) {
  gu->removeAll(true);
  setUser(gu, user);
  users->update();
}

void GuiWidgets::removeUser(UBox *gu) {
  users->remove(gu, true);
  users->update();
}

/* ==================================================== ======== ======= */

static void setWorld(UBox *gw, World *world, bool isCurrent) {

  if (! gw) return;

  UFont *ft = isCurrent ? &UFont::bold : &UFont::normal;
  gw->addlist
    (
     UColor::navy + ft 
     + world->name
     + umenu(UFont::bold
	     + uhbox(" URL: "
		     + UFont::normal + UColor::navy + world->url)
	     + uhbox(" Channel: "
		     + UFont::normal + UColor::navy + world->chan)
	     )
     );
}

UBox *GuiWidgets::addWorld(World *world, boolean isCurrent) {
  if (! world) return null;

  UBox *gw = &ulinkbutton();
  setWorld(gw, world, isCurrent);
  worlds->add(gw, 2);
  worlds->update();
  return gw;
}

void GuiWidgets::updateWorld(World *world, boolean isCurrent) {
  if (! world || ! world->guip) return;

  UBox *gw = world->guip;
  if (! gw) return;
#if 1	//pd this call causes segfault
  gw->removeAll(true);
#else
  gw->removeAll(false);
#endif
  setWorld(gw, world, isCurrent);
  gw->update();
  worlds->update();
}

void GuiWidgets::removeWorld(World *world) {
  UBox *gw = world->guip;
  if (! gw) return;

  worlds->remove(gw, TRUE);
  world->guip = NULL;
  worlds->update();
}

/* ==================================================== ======== ======= */
// Control Panel

static void setKey(int key, boolean ispressed) {
  struct timeval t;
  gettimeofday(&t, NULL);  // get time
  changeKey(key, ispressed, t.tv_sec, t.tv_usec);
}

static void action(int spaction) {
  struct timeval t;
  gettimeofday(&t, NULL);  // get time
  trace(DBG_WIN, "action: action=%d", spaction);
  Solid *ps = NULL;	// user
  ps->specialAction(spaction, NULL, t.tv_sec, t.tv_usec);
}

static void setmode(boolean *variable, boolean value) {
  *variable = value;
}

UBox* GuiWidgets::makeControlPanel() {
  UMode& enterHL = UMode::enterHighlight;

  UBox& projectiles = uvbox
    (
     uvcenter()
     + ubox(enterHL
	    + UOn::enter / *theme.activeBg
	    + UOn::arm   / ucall(BULLETUSER, action)
	    + uhcenter() + UPix::rball)
     + " " 
     + ubox(enterHL
	    + UOn::enter / *theme.activeBg
	    + UOn::arm   / ucall(DARTUSER, action)
	    + uhcenter() + UPix::ray)
    );

  UBox& up_down = uvbox
    (
     uvcenter()
     + ubox(enterHL
	    + UOn::enter  / *theme.activeBg
	    + UOn::arm    / ucall(KEY_JU, TRUE, setKey)
	    + UOn::disarm / ucall(KEY_JU, FALSE, setKey)
	    + uhcenter() + UPix::up)  // move up...
     + ubox(enterHL
	    + UOn::enter  / *theme.activeBg
	    + UOn::arm    / ucall(USERPAUSE, GUI::callVrengAction)
	    + uhcenter() + UPix::stop )  // pause
     + ubox(enterHL
	    + UOn::enter/theme.activeBg
	    + UOn::arm    / ucall(KEY_JD, TRUE, setKey)
	    + UOn::disarm / ucall(KEY_JD, FALSE, setKey)
	    + uhcenter() + UPix::down)  // move down...
    );

  UTrow &tilt = utrow
    (
     UFont::bold
     + utcell(UFont::small + "tilt:")
     + utcell(enterHL
	      + UOn::enter  / *theme.activeBg
	      + UOn::arm    / ucall(KEY_DE, TRUE, setKey)
	      + UOn::disarm / ucall(KEY_DE, FALSE, setKey)
	      + " - ")      // tilt up
     + utcell(enterHL
	      + UOn::enter  / *theme.activeBg
	      + UOn::arm    / ucall(KEY_HZ, TRUE, setKey)
	      + UOn::disarm / ucall(KEY_HZ, FALSE, setKey)
	      + " = ")      // stand up
     + utcell(enterHL
	      + UOn::enter  / *theme.activeBg
	      + UOn::arm    / ucall(KEY_MT, TRUE, setKey)
	      + UOn::disarm / ucall(KEY_MT, FALSE, setKey)
	      + " + ")      // tilt down
    );


  UTrow &linear_speed = utrow
    (
     UFont::bold
     + utcell(UFont::small + "lin. speed:")
     + utcell(enterHL
	      + UOn::enter / *theme.activeBg
	      + UOn::arm   / ucall(LSPEEDLESS, action)
	      + " - ")
     + utcell(enterHL
	      + UOn::enter / *theme.activeBg
	      + UOn::arm   / ucall(LSPEEDORIGINAL, action)
	      + " = ")
     + utcell(enterHL 
	      + UOn::enter / *theme.activeBg
	      + UOn::arm   / ucall(LSPEEDGREATER, action)
	      + " + ")
    );

  UTrow &angular_speed = utrow
    (
     UFont::bold
     + utcell(UFont::small + "ang. speed:")
     + utcell(enterHL
	      + UOn::enter / *theme.activeBg
	      + UOn::arm   / ucall(ASPEEDLESS, action)
	      + " - ")
     + utcell(enterHL
	      + UOn::enter / *theme.activeBg
	      + UOn::arm   / ucall(ASPEEDORIGINAL, action)
	      + " = ")
     + utcell(enterHL
	      + UOn::enter / *theme.activeBg
	      + UOn::arm   / ucall(ASPEEDGREATER, action)
	      + " + ")
    );

  UTrow &follow_mode = utrow
    (
     UFont::bold
     + utcell(UFont::small + "show names:")
	      + ucheckbox(UBgcolor::none
			+ UOn::enter / *theme.activeBg
			+ UOn::select / ucall(&followMouseMode, TRUE, setmode)
			+ UOn::unselect/ucall(&followMouseMode,FALSE, setmode)
			 )
    );

  UTable& settings = utable(tilt + linear_speed + angular_speed + follow_mode);

  /* (
       uhcenter() + *theme.Camera
      + ubox(ucall(GUI::callVrengAction, FOVYLESS, UOn::arm)
	     + " - ")
      + ubox(ucall(GUI::callVrengAction, FOVYORIGINAL, UOn::arm)
	     + " = ")
      + ubox(ucall(GUI::callVrengAction, FOVYGREATER, UOn::arm)
	     + " + ")
     )
    );
  */ 

  // === Control Panel =======

  UBox* control_panel = &uhbox
  (
   *theme.navigatorBg
   + uvcenter()
   + uvmargin(6) + uhmargin(7)
   + UColor::white
   + uvbox(UFont::bold 
	   + uitem(UOn::enter / *theme.activeBg
		   + *theme.Back2 + ucall(0, &GUI::back)
		   )
	   + uitem(UOn::enter / *theme.activeBg
		   + *theme.Forward2 + ucall(0, &GUI::forward)
		   )
	   + uitem(UOn::enter / *theme.activeBg
		   + *theme.Home2 + ucall(0, &GUI::home)
		   )
	   )
   + "     "
   //Navig Panel
   + umenubar( UBgcolor::none + UBorder::none 
	       + UOrient::horizontal
	       + makeNavigator(false)	       )
   + "     "
   + uhflex()
   + uvbox(uscale(-1) + messagesPane + entry)
   + "    "
   + uright()
   + projectiles
   + up_down
   + "  " 
   + settings
  );

  return control_panel;
}

/* ==================================================== ======== ======= */
// ALERT BOX

void GuiWidgets::alert(const char* message) {
  if (! alertBox) {
    alertStr = &ustr();
    alertBox = &udialog
      (
       uhmargin(20) + uvmargin(20)
       + uhcenter() + uvcenter() 
       + UBgcolor::black
       + uhbox(uhmargin(20) + uvmargin(20)
	       + uhcenter() + uvcenter() 
	       + UColor::orange
	       + UFont::bold + UFont::large
	       + alertStr
	       )
       + ubutton(UFont::bold + " OK "+ ucloseWin())
      );
    frame->add(alertBox);
    alertBox->moveAndCenter();
  }
  if (message) {
    alertStr->set(message);
    alertBox->show(true);
  }
  else alertBox->show(false);
  alertBox->update();
}

/* ==================================================== ======== ======= */
// VNC Dialog

void GuiWidgets::launchVncConnect(Vnc *pvnc) {
  if (! vnc_dialog) {
    vnc_server = &ustr();
    vnc_port   = &ustr();
    vnc_passwd = &ustr();

    UTable &vnc_table = utable
      (
       UBgcolor::grey
       + utrow(utcell(UFont::bold + UColor::black + "Server name:")   
	       + utcell(utextbox(uedit() + "                        " + vnc_server)))
       + utrow(utcell(UFont::bold + UColor::black + "Port number:") 
	       + utcell(utextbox(uedit() + "5901 " + vnc_port)))
       + utrow(utcell(UFont::bold + UColor::black + "Password:")
	       + utcell(utextbox(uedit() + "                " + vnc_passwd)))
      );

    UBox &vnc_control = uhbox
      (
       uhcenter() + UFont::bold 
       + ubutton("Connect" 
		 + ucloseWin()
		 + ucall(this, pvnc, &GuiWidgets::initVncConnect)
		 )
       + ubutton("Cancel" + ucloseWin())
      );

    vnc_dialog = &udialog
      (
       utitle("New VNC server")
       + uhcenter() + uvcenter()
       + uhmargin(2) + uvmargin(2)
       + UBgcolor::black
       + vnc_table
       + ""
       + vnc_control
      );

    frame->add(vnc_dialog);
    vnc_dialog->moveAndCenter();
  }
  vnc_dialog->show(true);
}

void GuiWidgets::initVncConnect(Vnc *vnc) {
  vnc->convert(vnc_server->chars(), vnc_port->chars(), vnc_passwd->chars());
}

// Icon Dialog
void GuiWidgets::launchIconDisplay(Icon *icon)
{
#if 0 //pdicon
  if (! file_dialog) {
    UBox *file_box = &uvbox(UBgcolor::white + uhmargin(2) + uvmargin(2));
    file_dialog = &udialog
      (
       utitle(icon->name.url)
       + uvflex()
       + uscrollpane(uheight(400) + uwidth(500) + file_box)
       + ubutton(UFont::bold + uhcenter() + " Cancel " + ucloseWin())
       );

    FILE *fp;
    char buf[BUFSIZ];
    if ((fp = fopen(icon->name.url, "r")) != NULL) {
      while (fgets(buf, sizeof(buf), fp)) {
        file_box->add(uitem(UColor::black + buf), false);
      }
    }
    frame->add(file_dialog);
    file_dialog->moveAndCenter();
    fclose(fp);
  }
  file_dialog->show(true);
#endif //pdicon
}

/* ==================================================== ======== ======= */

void GuiWidgets::processMessage(class UEvent& e) {
  UStr msg;
  bool stat = e.getMessage(msg);

  if (!stat || msg.empty()) return;

  UStr arg;
  for (int k = 0; k < msg.length(); k++) {
    if (msg.charAt(k) == ':') {
      arg = msg.split(k);
      arg.remove(0, 1);
      break;
    }
  }

  //cerr << "processMessage: '" << msg << "'" << " arg: " << arg <<endl;

  if (msg == "turn") {
    if (arg == "left 1") setKey(KEY_GA, TRUE);
    else if (arg == "left 0") setKey(KEY_GA, FALSE);

    else if (arg == "right 1") setKey(KEY_DR, TRUE);
    else if (arg == "right 0") setKey(KEY_DR, FALSE);
  }

  else if (msg == "move") {
    if (arg == "left 1") setKey(KEY_SG, TRUE);
    else if (arg == "left 0") setKey(KEY_SG, FALSE);

    else if (arg == "right 1") setKey(KEY_SD, TRUE);
    else if (arg == "right 0") setKey(KEY_SD, FALSE);

    else if (arg == "forward 1") setKey(KEY_AV, TRUE);
    else if (arg == "forward 0") setKey(KEY_AV, FALSE);

    else if (arg == "backward 1") setKey(KEY_AR, TRUE);
    else if (arg == "backward 0") setKey(KEY_AR, FALSE);

    else if (arg == "up 1") setKey(KEY_JU, TRUE);
    else if (arg == "up 0") setKey(KEY_JU, FALSE);

    else if (arg == "down 1") setKey(KEY_JD, TRUE);
    else if (arg == "down 0") setKey(KEY_JD, FALSE);
  }

  else if (msg == "back") GUI::back(0);
  else if (msg == "forward") GUI::forward(0);
  else if (msg == "home")  GUI::home(0);

  else if (msg == "enter") {
    // a completer
    cerr << "enter" << endl;
  }

  else if (msg == "file") {
    if (!arg.empty())
      //cerr << "file:" << arg << endl;
      GUI::world(0, arg.chars());
  }

  else if (msg == "get") {
    // a completer
    cerr << "get: " << *selected_object_url << endl; 
  }

  else if (msg == "put") {
    if (!arg.empty()) {
      UStr file_name = arg.getFileName();
      UStr val;
      if (file_name.empty())
	val = "<url=\"" & arg & "\">";
      else
#if 0
	val = "<url=\"" & arg & "\">&<name=\"" & file_name & "\">&<action=\"push\">";
#else
	val = "<url=\"" & arg & "\">&<name=\"" & file_name & "\">";
#endif
      
      //cerr << "put:" << val << endl;
      if (! App::checkUrl(arg.chars()))
        return;		// bad url
      put_info->putIcon(val);
    }
  }
}
