/* ==================================================== ======== ======= *
 *
 *  ubxwin.cc : X-Window initialization and management for the VREng GUI
 *  NOTE: this file should be common to all X-Window GUI variants
 *
 *  VREng / Ubit Project [Elc::01/02]
 *  Author: Eric Lecolinet
 *  Date:   (rev) 12 May 03
 *
 *  (C) 2002 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 */

#include <ubit/ubit.hpp>
#include <ubit/ugraph.hpp>
#include <ubit/unatwin.hpp>
#include <ubit/unatdisp.hpp>

#include "global.h"
#include "world.h"	// World::updateWorld
#include "net.h"	// NetTimeout
#include "vgl.h"	// VglInit
#include "landmark.h"	// isCounterVisible

#include "guiclass.h"
#include "gui.h"
#include "guiImpl.hh"
#include "widgets.hh"


//catches some X undesirable errors
static void catchXErrors();

// checks regularly for various updates
static void timeoutNet(UAppli*);

static int glxminor = 0;

#undef GLX13
#ifdef GLX13
static GLXFBConfig *glconfig;
#endif

/* ==================================================== ======== ======= */

void GUI::createToplevel(int argc, char *argv[], char **fallback_options)
{
  // creates the Ubit Application Context
  appli = new UAppli(&argc, argv);
  if (!appli) fatal("GUI::createToplevel: the UAppli could not be created");

  // UNatDisp: gives acces to the X Display that is used by the UAppli
  UNatDisp* natdisp = appli->getNatDisp(); 
  display    = natdisp->getXDisplay();
  screen     = natdisp->getXScreen();
  int scr_no = XScreenNumberOfScreen(screen);
  int dummy, glxmajor = 0;

  // Check if GLX is supported
  if (!glXQueryExtension(display, &dummy, &dummy))
    fprintf(stderr, "X server has no OpenGL GLX extension\n");

#if !defined(WITH_TINYGL)
#if 0 //info
  printf("GLX client: %s, %s\n",
          glXGetClientString(display, GLX_VENDOR),
          glXGetClientString(display, GLX_VERSION));
  printf("GLX server: %s, %s\n",
          glXQueryServerString(display, scr_no, GLX_VENDOR),
          glXQueryServerString(display, scr_no, GLX_VERSION));
#endif //info

  /* Get GLX version */
  glXQueryVersion(display, &glxmajor, &glxminor);
  fprintf(stderr, "glXQueryVersion: GLX%d.%d\n", glxmajor, glxminor);
#endif

  static int attributeListGLX12[] = {
    GLX_RGBA,
    GLX_DEPTH_SIZE, 16,
    GLX_DOUBLEBUFFER,
    None
  };
#ifdef GLX13
  static int attributeListGLX13[] = {
    GLX_RED_SIZE, 4,
    GLX_GREEN_SIZE, 4,
    GLX_BLUE_SIZE, 4,
    GLX_DEPTH_SIZE, 16,
    GLX_DOUBLEBUFFER, True,
    None
  };
  int nelem;
#endif

  /* TODO: test if GLX1.3 to do glxCreateWindow */
  // find a Visual that match requested depth and OpenGL options
#ifdef GLX13
  if (glxminor <= 2) {
#endif
    if (! (glvisual = glXChooseVisual(display, scr_no, attributeListGLX12)))
      fprintf(stderr, "could not get appropriate OpenGL visual\n");
    else
      trace(DBG_INIT, "glvisual: depth=%d, visualid=%x, class=%d [PseudoColor=%d, TrueColor=%d]", 
	    glvisual->depth, glvisual->visualid, getVisualClass(glvisual),
	    PseudoColor, TrueColor);
#ifdef GLX13
  }
  else {
    if (! (glconfig = glXChooseFBConfig(display, scr_no, attributeListGLX13, &nelem)))
      fprintf(stderr, "could not get appropriate OpenGL config\n");
    else {
      fprintf(stderr, "get appropriate OpenGL config nelem=%d\n", nelem);
      if (! (glvisual = glXGetVisualFromFBConfig(display, glconfig[0])))
        fprintf(stderr, "could not get appropriate OpenGL visual\n");
    }
  }
#endif
  
  // tells Ubit to use this GL specific visual
  // (note this fct also initializes the Colormap)
  natdisp->setVisual(*glvisual);

  // allocates graphical resources
  appli->realize();

  // must be done after realize()
  cmap = natdisp->getXColormap();
  sceneInitialized = false;
}

/* ==================================================== ======== ======= */

/** VREng Mainloop */
void GUI::mainLoop()
{
  catchXErrors();	// catches some X undesirable errors

  // gets the X Window of the GL Zone that was created by Ubit
  // Note: must be done after 'new GuiWidgets" (of course!)
  glwin = guiWidgets->getGLZone()->getXWin();

  // Create an OpenGL rendering context
#ifdef GLX13
  if (glxminor <= 2)
#endif
    if (! (glxc = glXCreateContext(display, glvisual, None, True)))
      fatal("glXCreateContext: could not create OpenGL rendering context");
#ifdef GLX13
  else
    if (! (glxc = glXCreateNewContext(display, glconfig[0], GLX_RGBA_TYPE, 0, GL_TRUE)))
      fatal("glXCreateNewContext: could not create OpenGL rendering context");
#endif

  // here, we call GL & VGL initialization
#ifdef GLX13
  if (glxminor <= 2)
#endif
    glXMakeCurrent(display, glwin, glxc);
#ifdef GLX13
  else
    glXMakeContextCurrent(display, glwin, (GLXDrawable) NULL, glxc);
#endif

  // load fonts
  glfontdl = glGenLists(256);
  XFontStruct *font = XLoadQueryFont(display,
                            "-misc-fixed-medium-r-*-*-12-*-*-*-*-*-*-*");
  //pd XColor black = { 0, 0, 0, 0, 0, 0 };
  //pd Cursor null_cursor = XCreateGlyphCursor(display, font->fid, font->fid, ' ', ' ', &black, &black);
  glXUseXFont(font->fid, 0, 256, glfontdl);
  XFreeFont(display, font);

  // init 3D
  VglInit(options->quality);
  VglSetViewPort(options->width3D, options->height3D);
  trace(DBG_INIT, "GUI X-Window initialized");

  // winResized() is called when the GLZone is resized to tell OGL
  // that the size of the RenderBuffer has changed
  guiWidgets->getGLZone()->add(UOn::viewResize / ucall(this, &GUI::resizeGLBuffer));

  // !!TODO: add a callback to stop processing when the window is iconified

  // calculates the New World Order and renders it when mainloop is idle
  // - 1st arg: delay (0 means immediatly when idle)
  // - 2nd arg: times (-1 means infinity)
 
  UTimer* render_timer = appli->openTimer(0, -1);
  render_timer->onAction(ucall(this, &GUI::renderScene));

  // timeoutNet is called regularly during the session
  UTimer* net_timer = appli->openTimer(NET_TIMEOUT, -1);
  net_timer->onAction(ucall(appli, timeoutNet));
  
  // opens the "Please Wait" alert box
  // guiWidgets->alert("Please wait: loading the World...");

  appli->mainLoop();
}

/* ==================================================== ======== ======= */

void GUI::initX() {}	// to be removed one day

/** Called once after the mainloop is started for init, Wmgt and rendering */
void GUI::initScene()
{
  // Note: as this init. may be quite long, it is postponed so that
  // we can start the main loop immediately then run this init.
  // (en pratique c'est initWO(), qui DOIT etre lancee APRES VglInit())
  if (vrengInitCB) vrengInitCB();

  // closes the "Please Wait" alert box
  guiWidgets->alert(null);
  stopTime(&ptime_init);

  sceneInitialized = true; // the scene is initialized and ready for rendering
  // note: this var is supposed to change when the windo is iconified
  readyAndVisible = true;
}

/* ==================================================== ======== ======= */

/** Calculates the New World Order and renders it */
void GUI::renderScene()
{
  if (!sceneInitialized) {
    initScene();
    sceneInitialized = true;
  }

  // at least one postponed Key Release event
  if (pendingPostponedKRs())
    flushPostponedKRs();
  
  // the GUI is ready for rendering and is visible
  // (no rendering is performed while iconified)
  if (readyAndVisible) {      

    // Performs world calculation
    startTime(&ptime_world);
    World::updateWorld(ptime_world.start.tv_sec, ptime_world.start.tv_usec);
    stopTime(&ptime_world);
    
    // General rendering
    startTime(&ptime_render);
    VglRender();
    stopTime(&ptime_render);

    // Displays counters
    if (options->counters && Landmark::isCounterVisible()) {
      gl_printf(0, 1, 0, 1, options->width3D-60, options->height3D-8, glfontdl,
                "%.2f fps", getRate());
      gl_printf(0, 1, 0, 1, 1, options->height3D-8, glfontdl,
                "%3.0f s", stopTime(&ptime_net));
    }

    // Displays on the GL rendering window
    // swap the display and drawing buffers (double buffering mode):
    // the scene we just rendered in the doublebuffer is now shown on the screen
    startTime(&ptime_buffer);
    glXSwapBuffers(display, glwin);
    stopTime(&ptime_buffer);

    // redraw the Ubit menu (when it is open) on top of the scene 
    // that was just drawn
    if (guiWidgets->getOpenedMenu()) {
      guiWidgets->getOpenedMenu()->update(UUpdate::paint);
      XFlush(display);	// this is necessary to avoid flicker
    }
    cycles++;		// count cycles
  }
}

/* ==================================================== ======== ======= */
/* ==================================================== ======== ======= */
// Handlers and TimeOuts

void GUI::resizeGLBuffer(UEvent& e)
{
  UView* glview = e.getView();
  if (glview) {
    options->width3D  = glview->getWidth();
    options->height3D = glview->getHeight();
    VglSetViewPort(options->width3D, options->height3D);
  }
}

/** This timeOut checks regularly if various updates are needed */
static void timeoutNet(UAppli* appli)
{
  NetTimeout(); // NetTimeout() checks if various updates are needed
}

/** Called when something occurs on a file-descriptor */
void GUI::addInputTable(int cnt, int *table, int table_no)
{
  inputTable[table_no] = new UInput*[cnt];

  for (int k=0; k < cnt; k++) {
    inputTable[table_no][k] = appli->openInput(table[k]);
    inputTable[table_no][k]->onAction(ucall(table[k], NetIncoming));
  }
}

void GUI::removeInputTable(int cnt, int table_no)
{
  if (! inputTable[table_no]) return;

  for (int k=0; k < cnt; k++) {
    //delete inputTable[table_no][k];
    appli->closeInput(inputTable[table_no][k]);
  }

  delete inputTable[table_no];
  inputTable[table_no] = NULL;
}

/* ==================================================== ======== ======= */
// for Handling X errors

// this variable will point to the previous (= standard) error handler
static int (*std_x11error_handler)(Display *, XErrorEvent *);

static int MyX11ErrorHandler(Display *d, XErrorEvent *ev)
{
  /* errors for shared-mem */
  if (ev->request_code == 129) {
    fprintf(stderr, "Shared memory unavailable\n");
  }
  else {
    quitVreng(0);
    std_x11error_handler(d, ev);
    exit(1);
  }
  return 0;
}

static void catchXErrors()
{
  std_x11error_handler = XSetErrorHandler(MyX11ErrorHandler);
}
