#ifndef CHANNEL_H
#define CHANNEL_H

#include "global.h"
#include "net.h"
#include "rtpsess.h"

#define MANAGER_MODE	0
#define WORLD_MODE	1

/**
 * Channel Class
 */
class Channel {
 public:
  uint32_t	ssrc;		///< ssrcid
  uint32_t	group;		///< group address
  uint16_t	port;		///< RTP port
  uint8_t	ttl;		///< ttl scope
  uint8_t	cntfds;		///< fd numbers
  int		sd[5];		///< sockets descriptors
  struct sockaddr_in * sa[5];	///< sockets addresses
  struct sockaddr_in sa_rtp;	///< RTP data
  struct sockaddr_in sa_rtcp;	///< RTCP data
  struct sockaddr_in sa_rtp_r;	///< RTP data reflector
  struct sockaddr_in sa_rtcp_r;	///< RTCP data reflector
  struct sockaddr_in sa_data;	///< UDP datas
  struct ip_mreq mreq;		///< IGMPv2
  Session *session;		///< RTP session
  Channel *next;		///< next channel

  Channel();
  /**< Constructor */

  virtual ~Channel();
  /**< Destructor */

  virtual int createChannel(const char *chanstr, int **fds);
  /**<
   * Create a Channel
   * channel string is given by decode
   * channel structure is given bye Channel
   * return number of fd to survey
   *
   * Usage:
   * int *fds;
   * int count = createChannel(chan_str, &fds);
   * for (i=0; i < count; i++) {
   *   addThisFdToWatchList(fds[i]);
   * }
   * create sockets mcast_recv_rtp and mcast_send_rtp
   *        sockets mcast_recv_rtcp and mcast_send_rtcp
   */

  virtual void naming();
  /**<
   * Channel naming
   * init my_host_id, my_port_id, my_obj_id
   */

  virtual void deleteFromList();
  /**< Deletes a channel from  a list */

  virtual void quitChannel();
  /**< Quits a channel */

  virtual int createMcastRecvSocket(struct sockaddr_in *sa);
  /**<
   * Creates a Multicast listen socket on the channel defined by group/port.
   * Returns sock else -1 if problem.
   */

  virtual void closeMcastSocket();
  /**< Closes Multicast sockets */

  virtual void closeUcastSocket();
  /**< Closes Unicast sockets */

  virtual void sendBYE();
  /**< Sends a BYE RTCP */

  virtual int buildSR(rtcp_common_t *prtcp_hdr, uint8_t *pkt);
  /**< Builds a SR RTCP packet */

  virtual int buildSDES(rtcp_common_t *prtcp_hdr, uint8_t* pkt);
  /**<
   * Builds a SDES packet.
   * Fills rtcp buffer and return length.
   */

  virtual int buildBYE(rtcp_common_t *prtcp_hdr, uint8_t *pkt);
  /**< Builds a BYE packet */

  virtual int sendRTCPPacket(const struct sockaddr_in *to, const uint8_t pt);
  /**< Sends a RTCP packet */

  virtual int sendSRSDES(const struct sockaddr_in *to);
  /**<  Sends a RTCP compound packet (SR + SDES) */

  virtual int joinGroup(const int sock);
  /**<
   * Joins group (IP_ADD_MEMBERSHIP).
   * "group" in network format,
   * returns sock if OK, else -1.
   */

  virtual int leaveGroup(const int sock);
  /**<
   * Leaves group (IP_DROP_MEMBERSHIP).
   * Returns sock if OK, else -1.
   */

  static void initNetwork();
  /**< Network Initialization */

  static void decode(const char *chan_str, uint32_t *group, uint16_t *port, uint8_t *ttl);
  /**<
   * Decodes the string format "group[/port[/ttl]]".
   * Returns group, port, ttl.
   * If chan_str == "" or NULL, we get the default values.
   */

  static char * join(const char *chanstr);
  /**<
   * Joins the channel and return the new channel string.
   * Sets the global variables: currentChannel, cntFd, tabFd
   */

  static char * joinManager(const char *chanstr);
  /**<
   * Only for vreng client.
   * Sets the global variables: managerChannel, cntManagerFd, tabManagerFd.
   */

  static Channel * getCurrentChannel();
  /**< Gets current channel instance */

  static Channel * getManager();
  /**< Gets manager channel */

  static void switchChannel();

  static Channel * getbysa(const struct sockaddr_in *sa);

#ifdef VRENGD
  static char * joinDaemon(const char *chanstr);
  /**< Only for vrengd.
   * Sets managerChannel, cntManagerFd, tabManagerFd: are global.
   */
#endif VRENGD
};

/**
 * Vac class
 */
class Vac {
 public:
  char url[URL_LEN];
  char channel[CHAN_LEN];
  Vac *next;

  static int connectVac();
  static void getList();
  static bool resolveWorldUrl(const char *_url, char *chanstr);
  static bool getChannel(const char *_url, char *chanstr);
  static bool getUrlAndChannel(const char *name, char *_url, char *chanstr);
};

int getFdSendRTP(const struct sockaddr_in *sa);
int getFdSendRTCP(const struct sockaddr_in *sa);
struct sockaddr_in * getSaRTCP(const struct sockaddr_in *sa);
int getfdbysa(const struct sockaddr_in *sa, const int sd);
struct sockaddr_in * getsabysa(const struct sockaddr_in *sa, const int sd);

#endif // CHANNEL_H
