#include "global.h"
#include "net.h"
#include "netdefs.h"	// RECV_RTP
#include "wo.h"		// user.h
#include "user.h"	// USER_TYPE
#include "world.h"	// isValidType


// network ids
static uint32_t mySsrcId = 0;	// ssrc network format
static uint32_t myHostId = 0;	// addr_IP network format
static uint16_t myPortId = 0;	// port network format
static uint16_t myObjId = 0;	// new object'id (naming), fmt host
static uint32_t myMgrSsrcId = 0; // manager ssrc network format

static NetObject *netobjectList = NULL;	// netobjects list
static uint8_t propertiesnumber[OBJECTSNUMBER+1];


/*
 * Handling id names and properties
 */

/* NetObject constructors */

void NetObject::initCommun()
{
  new_netobject++;
  nbprop = 0;
  netprop = NULL;
  next = NULL;
  prev = NULL;
}

/* Creates local generic NetObject */
NetObject::NetObject()
{
  initCommun();
  type = 0;
  pobject = NULL;
  permanent = NET_PERMANENT;
#if 0
  setSrcId(0);
  setPortId(0);
  setObjId(0);
#endif
  setNetObjectId();
}

/* Creates local permanent NetObject */
NetObject::NetObject(WObject *po, uint8_t nprop, uint16_t oid)
{
  initCommun();
  type = po->type;
  pobject = po;
  permanent = NET_PERMANENT;
  setPropertiesNumber(nprop);

  char str[80];
  sprintf(str, "%d/%d", type, oid);
  createNetObjectIdFromString(str, permanent);     // net objname
  trace(DBG_WO, "NetObject: str=%s", str);
}

/* Creates local volatile NetObject */
NetObject::NetObject(WObject *po, uint8_t nprop)
{
  initCommun();
  type = po->type;
  pobject = po;
  permanent = NET_VOLATILE;
  setPropertiesNumber(nprop);
  createNetObject(permanent);
#if 0
  setSrcId(0);
  setPortId(0);
  setObjId(0);
#endif
  setNetObjectId();
}

/* Creates replicated volatile NetObject */
NetObject::NetObject(WObject *po, uint8_t nprop, NetObjectId _noid)
{
  initCommun();
  type = po->type;
  pobject = po;
  permanent = NET_VOLATILE;
  noid = _noid;
  setPropertiesNumber(nprop);
}

NetObject * NetObject::getNetObjectsList()
{
  return netobjectList;
}

void NetObject::clearNetObjectsList()
{
  netobjectList = NULL;
}

void setMyMgrSsrcId(const uint32_t mgr_ssrc_id)
{
  myMgrSsrcId = mgr_ssrc_id;
}

uint32_t getMyMgrSsrcId(void)
{
  return myMgrSsrcId;
}

void setMySsrcId(const uint32_t ssrc_id)
{
  mySsrcId = ssrc_id;
}

// get current ssrc
uint32_t getMySsrcId(void)
{
  return mySsrcId;
}

void setMyHostId(const uint32_t host_id)
{
  myHostId = host_id;
}

uint32_t getMyHostId(void)
{
  return myHostId;
}

void setMyPortId(const uint16_t _port_id)
{
  myPortId = _port_id;
}

uint16_t getMyPortId(void)
{
  return myPortId;
}

void setMyObjId(const uint16_t _obj_id)
{
  myObjId = _obj_id;
}

uint16_t getMyObjId(void)
{
  return myObjId;
}

void NetObject::setSrcId(const uint32_t _src_id)
{
  noid.src_id = _src_id;
}

uint32_t NetObject::getSrcId()
{
  return noid.src_id;
}

void NetObject::setPortId(const uint16_t _port_id)
{
  noid.port_id = _port_id;
}

uint16_t NetObject::getPortId()
{
  return noid.port_id;
}

void NetObject::setObjId(const uint16_t _obj_id)
{
  noid.obj_id = _obj_id;
}

uint16_t NetObject::getObjId()
{
  return noid.obj_id;
}

/* Builds a concataned string name */
char * NetObjectId::getNetNameById()
{
  static char str[80];

  sprintf(str, "%x/%x/%x", ntohl(src_id), ntohs(port_id), ntohs(obj_id));
  return str;
}

/* Returns 0 if different, other if equal */
bool NetObjectId::equalNetObjectId(const NetObjectId n2)
{
  return src_id == n2.src_id && port_id == n2.port_id && obj_id == n2.obj_id;
}

bool NetObject::isPermanent()
{
  return permanent;
}

/* Gets a NetObject by name */
NetObject * NetObjectId::getNetObject()
{
  NetObject *pn;

  for (pn = netobjectList; pn && !equalNetObjectId(pn->noid); pn = pn->next) {
    if (! isValidType(pn->type)) {
      error("getNetObject: bad type=%d", pn->type);
      return NULL;
    }
#if 0
    assert(pn->next == NULL || pn->next->prev == pn);
    assert(pn->prev == NULL || pn->prev->next == pn);
    assert(pn->prev || pn == netobjectList);
#endif
  }
#if 0 //debug
  if (! pn) {
    for (NetObject *n = netobjectList; n ; n = n->next)
      error("type=%d noid=%x/%d/%d mynoid=%x/%d/%d", n->type, n->noid.src_id, n->noid.port_id, n->noid.obj_id, src_id, port_id, obj_id);
  }
#endif
  return pn;
}

/* Inserts netobject in head of NetObject list */
void NetObject::insertNetObject()
{
  if ((next = netobjectList) != NULL)
    netobjectList->prev = this;
  netobjectList = this;
  prev = NULL;  
}

void NetObject::initProperties(bool _responsible)
{
  if (netprop) {
    //pd warning("initProperties: netprop already exists (type=%d)", type);
    return;
  }

  uint8_t n = propertiesNumber();
  if (!n)
    return;
  trace(DBG_NET, "initProperties: type=%d nobj=%s nprop=%d resp=%d", type, pobject->name.instance_name, n, _responsible);

  netprop = new NetProperty[n];

  for (int i=0; i<n; i++) {
    NetProperty *pprop = netprop + i;
    pprop->responsible = _responsible;
    pprop->version = 0;
    pprop->resetDates();
  }
}

/* Returns the number of properties of this type */
uint8_t NetObject::propertiesNumber()
{
  if (nbprop == 0)
    return propertiesnumber[type];
  return nbprop;
}

uint8_t NetObject::propertiesNumber(const uint8_t _type_id)
{
  return propertiesnumber[_type_id];
}

/* Sets the number of properties of this type */
void NetObject::setPropertiesNumber(const uint8_t _nbprop)
{
  propertiesnumber[type] = nbprop = _nbprop;
}

void NetObject::setPropertiesNumber(const uint8_t _type_id, const uint8_t _nbprop)
{
  propertiesnumber[_type_id] = _nbprop;
}

/* Assigns an unique identifier to each Vreng netobject */
void NetObject::setNetObjectId()
{
  noid.src_id = mySsrcId;		// Application's identifier
  noid.port_id = myPortId;		// Comm port identifier
  noid.obj_id = htons(myObjId++);	// Application wide unique number
}

/* Creates a new netobject name */
void NetObject::createNetObject(bool netbehave)
{
  // MS.: Objects need a unique ID from the start,
  // not just for networked objects, so that the
  // Vjc controller apps can tell them apart.
  if (next) {
    warning("createNetObject: already named, type=%d", type);
    return;
  }
  setNetObjectId();

  permanent = netbehave;
  insertNetObject();
  initProperties(true);	// new: then we are responsible
}

/* Builds a netobject name from the string "scene_id/obj_id" */
void NetObject::createNetObjectIdFromString(const char *s, bool netbehave)
{
  //PD if (noid.src_id != 0 || next != NULL) {
  if (next) {
    warning("createNetObjectIdFromString: already named, type=%d src_id=%x next=%p", type, noid.src_id, next);
    return;
  }

  uint16_t scene_id, obj_id;
  int c = sscanf(s, "%hu/%hu", &scene_id, &obj_id);

  if (c != 2 || scene_id == 0) {
    error("createNetObjectIdFromString: invalid name %s", s);
    return;
  }
  noid.src_id = htonl(1); 
  noid.port_id = htons(scene_id);
  noid.obj_id = htons(obj_id);
  if (noid.getNetObject()) {
    warning("createNetObjectIdFromString: already seen %d/%d", scene_id, obj_id);
    return;
  }
  permanent = netbehave;	// should be true
  insertNetObject();
  initProperties(false);	// we are not responsible
}

/* Deletes a netobject from the list */
void NetObject::deleteNetObjectFromList()
{
  if (! noid.getNetObject()) {
    //warning("deleteNetObjectFromList: already unnamed/deleted type=%d", type);
    return;
  }
  if (prev)
    prev->next = next;
  else {
    if (this != netobjectList) {
      error("deleteNetObjectFromList: type=%d pn=%p netobjectList=%p",
            type, this, netobjectList);
      return;
    }
    netobjectList = next;
  }
  if (next)
    next->prev = prev;
  next = NULL;
  prev = NULL;
}

NetObject::~NetObject()
{
  deleteNetObjectFromList();

  memset(&noid, 0, sizeof(noid));
  if (netprop)
    delete[] netprop; 
  netprop = NULL;
  del_netobject++;
}

bool NetObject::isResponsible()
{
  //pd BUG! comment this line because netprop->responsible is false
  return (netprop && netprop->responsible);
  //pd return true;
}

/* Finds a WObject pointer by its noid */
WObject * NetObject::getWObjectByNoid()
{
  for (ObjectList *pl = mobileList; pl; pl = pl->next) {
    if (pl->pobject && pl->pobject->noh) {
      if (noid.equalNetObjectId(pl->pobject->noh->noid)) {
        trace(DBG_FORCE, "getWObjectByNoid: noid %x:%x:%x name=%s", noid.src_id, noid.port_id, noid.obj_id, pobject->name.instance_name);
        return pl->pobject;
      }
    }
  }
  return NULL;
}

/* Gets the property (its local copy) got from Network */
void NetObject::getProperty(const uint8_t prop_id, Payload *pp)
{
  if (pobject) {
    pobject->getProperty(prop_id, pp);
    return;
  }

#if 0 //unnecessary
  // Find the WObject by its noid
  WObject *po = NULL;
  
  if ((po = getWObjectByNoid()))
    po->getProperty(prop_id, pp);
  return;
#endif //unnecessary
}

/* Puts the property (its local copy) to be sent to Network */
void NetObject::putProperty(const uint8_t prop_id, Payload *pp)
{
  if (pobject)
    pobject->putProperty(prop_id, pp);
}

void NetObject::getAllProperties(Payload *pp)
{
  uint8_t _nbprop = propertiesNumber();
  for (int p=0; p < _nbprop; p++)
    getProperty(p, pp);
}

/* Puts all properties of the netobject */
void NetObject::putAllProperties(Payload *pp)
{
  uint8_t _nbprop = propertiesNumber();
  for (int p=0; p < _nbprop; p++)
    putProperty(p, pp);
}

/* Removes netobject */
void NetObject::requestDeletionFromNetwork()
{
  if (pobject)
    pobject->deleteReplica();
}

/* Creates a replicated object */
NetObject * NetObject::replicateObject(uint8_t type_id, NetObjectId noid, Payload *pp)
{
  WObject *po = WClass::replicatorInstance(type_id, noid, pp);  // factory

  if (po) {
    if (! po->noh) {
      error("replicateObject: no po->noh for type=%d", type_id);
      return NULL;
    }
    return po->noh;	// OK
  }
  return NULL;	// BAD
}

#if 0 //notused
// get an netobject by SSRC
NetObject * getObjectBySSRC(const uint32_t _ssrc)
{
  NetObject *pn = netobject_list;

  for ( ; pn && pn->noid.src_id != _ssrc; pn = pn->next)
	;
  return pn;
}
#endif //notused

/*
 * NetProperty
 */

NetProperty::NetProperty()
{
  min_assume_delay = DEF_REFRESH_TIMEOUT * 2;
  max_assume_delay = DEF_REFRESH_TIMEOUT * 5;
  new_netproperty++;
}

/* Initializes responsibilities */
void NetProperty::setResponsible(bool _responsible)
{
  responsible = _responsible;
}

NetProperty::~NetProperty()
{
  del_netproperty++;
}

/** Adds d seconds to the date */
static void addToDate(struct timeval *t, const double d)
{
  double f = floor(d);

  t->tv_sec += (time_t) f;
  t->tv_usec += (time_t) ((d-f)*1e6);
  while (t->tv_usec >= 1000000) {
    t->tv_usec -= 1000000;
    t->tv_sec++;
  }
}

/**
 * Computes a new date for the assume_at of the property
 * and sets the last_seen at "now" 
 */
void NetProperty::resetDates()
{
  struct timeval now;
  gettimeofday(&now, NULL);

  last_seen = now;
  assume_at = now;

  double delay = min_assume_delay + rand()/(RAND_MAX+1.0)*(max_assume_delay - min_assume_delay);
  //trace(DBG_FORCE, "delay=%.2fs min=%.2fs max=%.2fs", delay, min_assume_delay, max_assume_delay);

  addToDate(&assume_at, delay); 
}
