#include "global.h"
#include "net.h"
#include "payload.h"


Payload::Payload()
{
  new_payload++;
  len = 0;
  idx = 0;
}

Payload::~Payload()
{
  del_payload++;
}

Payload * Payload::resetPayload()
{
  len = 0;
  idx = 0;
  return this;
}

/* Checks validity */
bool Payload::isValidPayload()
{
  return (len <= PAYLOAD_LEN && idx <= len);
}

/* Fills the payload */
int Payload::putPayload(const char *format, ...)
{
  va_list ap;

  if (! isValidPayload()) {
    error("putPayload: invalid Payload %s", this);
    return -1;
  }
  if (format == NULL) {
    error("putPayload: NULL format");
    return -1;
  }

  va_start(ap, format);
  len = idx;	// "rewrite" mode rather than "append"
  trace(DBG_13, "putPayload: idx=%d", idx);

  /* Parse format */
  while (*format) {
    switch (*format) {
    case 'c': // char
      {
	uint8_t c = va_arg(ap, int);  	

	data[idx++] = 'c';
	data[idx++] = c;
	break;
      }
    case 'h': // short
      {
	int16_t h = htons(va_arg(ap, int));  	

	data[idx++] = 'h';
	memcpy(data + idx, &h, sizeof(int16_t));
	idx += sizeof(int16_t);
	break;
      }
    case 'd': // int32_t
      {
	int32_t d = htonl(va_arg(ap, int));

	data[idx++] = 'd';
	memcpy(data + idx, &d, sizeof(int32_t));
	idx += sizeof(int32_t);
	break;
      }
    case 'f': // float
      {
	float f = va_arg(ap, double);
	int32_t *p = (int32_t*) &f;
	int32_t j = htonl(*p);

	data[idx++] = 'f';
	memcpy(data + idx, &j, sizeof(int32_t));
	idx += sizeof(int32_t);
	break;
      }
    case 's': // string
      {
	uint8_t *s = va_arg(ap, uint8_t *);
	uint16_t l;
        l = (*s) ? strlen((char *) s) : 0;

	data[idx++] = 's';
	data[idx++] = l;
        if (l)
	  memcpy(data + idx, s, l);
	idx += l;
	break;
      }
    case 'n': // NetObjectId
      {
	NetObjectId n = va_arg(ap, NetObjectId);

	data[idx++] = 'n';

	/* everything is already in network format */
	memcpy(data + idx, &n.src_id, sizeof(n.src_id));
	idx += sizeof(n.src_id);
	memcpy(data + idx, &n.port_id, sizeof(n.port_id));
	idx += sizeof(n.port_id);
	memcpy(data + idx, &n.obj_id, sizeof(n.obj_id));
	idx += sizeof(n.obj_id);
	break;
      }
    case 't': // timeval
      {
	struct timeval t = va_arg(ap, struct timeval);
	time_t sec = htonl(t.tv_sec);
	time_t usec = htonl(t.tv_usec);

	data[idx++] = 't';
	memcpy(data + idx, &sec, sizeof(int32_t));
	idx += sizeof(int32_t);
	memcpy(data + idx, &usec, sizeof(int32_t));
	idx += sizeof(int32_t);
	break;
      }	

    default: // unknown type in format
      error("putPayload: invalid format '%c' in %s", *format, format);
      va_end(ap);
      return -1;
    }

    format++;
    if (idx > len)
      len = idx;

    /* check the length */
    if (len >= PAYLOAD_LEN) {
      error("putPayload: Payload too long (%d > %d)", len, PAYLOAD_LEN);
      dumpPayload(stdout);
      len = idx = 0; // just in case
      va_end(ap);
      return -1;
    }
  }
  va_end(ap);
  return 0;
}    

/* Gets the payload */
int Payload::getPayload(const char *format, ...)
{
  va_list ap;
  const char *pformat = format;

  if (! isValidPayload()) {
    //pd char *pp = (char *) this;	// hack
    error("getPayload: invalid len=%d idx=%d %02x%02x%02x%02x", len, idx,
           data[idx], data[idx+1], data[idx+2], data[idx+3]); 
    return -1;
  }
  if (! format) {
    error("getPayload: NULL format"); 
    return -1;
  }

  va_start(ap, format); 

  while (*format) { 
    /* Format known ? */
    if (strchr("chdfsnt", *format) == NULL) {
      error("getPayload: invalid format [%c] in %s", *format, format);
      format++;
      continue;
    }

#if 0 //debug
    switch (*format) {
    case 'c':
      trace(DBG_FORCE, "format=%c idx=%d len=%d data=[%c %02x]", *format, idx, len, data[idx+0], data[idx+1]);
      break;
    case 'h':
      trace(DBG_FORCE, "format=%c idx=%d len=%d data=[%c %02x%02x]", *format, idx, len, data[idx+0], data[idx+1], data[idx+2]);
      break;
    case 'd':
    case 'f':
      trace(DBG_FORCE, "format=%c idx=%d len=%d data=[%c %02x%02x%02x%02x]", *format, idx, len, data[idx+0], data[idx+1], data[idx+2], data[idx+3], data[idx+4]);
      break;
    case 'n':
      trace(DBG_FORCE, "format=%c idx=%d len=%d data=[%c %02x%02x%02x%02x %02x%02x %02x%02x]", *format, idx, len, data[idx+0], data[idx+1], data[idx+2], data[idx+3], data[idx+4], data[idx+5], data[idx+6], data[idx+7], data[idx+8]);
      break;
    case 's':
      trace(DBG_FORCE, "format=%c idx=%d len=%d data=[%c %02x %02x%02x%02x%02x%02x%02x%02x]", *format, idx, len, data[idx+0], data[idx+1], data[idx+2], data[idx+3], data[idx+4], data[idx+5], data[idx+6], data[idx+7], data[idx+8]);
      break;
    }
#endif //debug

    /* test matching Payload - format */
    if (data[idx] != *format) {
      error("getPayload: mismatch '%c'[x'%02x'], format='%s' [%s], len=%d, idx=%d[x'%02x']",
	    data[idx], data[idx], format, pformat, len, idx, idx);
      dumpPayload(stdout);
      //pd format++;
      //pd continue; // return -1;
      return -1;
    }
    idx++;	// points on following data

    switch (*format) { 
    case 'c': // char
      { 
	uint8_t *p = va_arg(ap, uint8_t *);

	memcpy(p, data + idx, sizeof(uint8_t)); 
	idx++;
	break; 
      } 
    case 'h': // int16_t
      { 
	int16_t h; 
	int16_t *p = va_arg(ap, int16_t *);

	memcpy(&h, data + idx, sizeof(int16_t)); 
	*p = ntohs(h);
	idx += sizeof(int16_t);
	break; 
      } 
    case 'd': // int32_t
      { 
	int32_t d; 
	int32_t *p = va_arg(ap, int32_t *);

	memcpy(&d, data + idx, sizeof(int32_t)); 
	*p = ntohl(d);
	idx += sizeof(int32_t);
	break; 
      } 
    case 'f': // float
      { 
	int32_t f;
	float *p = va_arg(ap, float *);

	memcpy(&f, data + idx, sizeof(int32_t)); 
	f = ntohl(f);
	memcpy(p, &f, sizeof(int32_t));
	idx += sizeof(int32_t);
	break; 
      } 
    case 's': // string
      { 
	/* Note: no length check */
	uint8_t *s = va_arg(ap, uint8_t *);
	uint16_t l = data[idx++];

        if (l)
	  memcpy(s, data + idx, l);
	s[l] = 0; // NULL terminated
	idx += l;
	break; 
      }     
    case 'n': // NetObjectId
      {
	NetObjectId *n = va_arg(ap, NetObjectId*);

	memcpy(&n->src_id, data + idx, sizeof(n->src_id));
	idx += sizeof(n->src_id);
	memcpy(&n->port_id, data + idx, sizeof(n->port_id));
	idx += sizeof(n->port_id);
	memcpy(&n->obj_id, data + idx, sizeof(n->obj_id));
	idx += sizeof(n->obj_id);
	break;
      }
    case 't': // timeval
      {
	time_t sec, usec;
	struct timeval *p = va_arg(ap, struct timeval*);

	memcpy(&sec, data + idx, sizeof(int32_t));
	idx += sizeof(int32_t);
	memcpy(&usec, data + idx, sizeof(int32_t));
	idx += sizeof(int32_t);
	p->tv_sec = ntohl(sec);
	p->tv_usec = ntohl(usec);
	break;
      }	
    default:
      error("getPayload: format unimplemented [%c] in %s", *format, format);
      va_end(ap);
      return -1;
    }
    
    /* verify if not too far */
    if (idx > len) {
      error("getPayload: past end of Payload: idx=%d len=%d", idx, len); 
      dumpPayload(stdout);
      idx = len = 0; 
      va_end(ap);
      return -1; 
    }
    format++;
  }
  va_end(ap); 
  return 0; 
}

void Payload::seekPayload(const uint16_t _idx)
{
  idx = _idx;
}

uint16_t Payload::tellPayload()
{
  return idx; 
}

int Payload::tellStrInPayload(const char *str)
{
  uint16_t save_idx = idx;

  for (idx = 0; idx < len; ) { 
    uint8_t format = data[idx];

    /* checks format */
    if (strchr("chdfsnt", format) == NULL) {
      error("tellStrInPayload: invalid format [%c]", format);
      idx = save_idx;
      return -1;
    }
    idx++;

    switch (format) { 
    case 'c': // char
	idx++;
	break; 
    case 'h': // int16_t
	idx += sizeof(int16_t);
	break; 
    case 'd': // int32_t
    case 'f': // float
	idx += sizeof(int32_t);
	break; 
    case 's': // string
	if (!strncmp((char *) (data + idx + 1), str, strlen(str))) {
          idx--;
	  return idx;
        }
	idx += data[idx++];
	break; 
    case 'n': // NetObjectId
	idx += sizeof(uint32_t);
	idx += sizeof(uint16_t);
	idx += sizeof(uint16_t);
	break;
    case 't': // timeval
	idx += sizeof(int32_t);
	idx += sizeof(int32_t);
	break;
    default:
        if (format != 0) {
	  error("tellStrInPayload: format unimplemented [%c 0x%02x]", format, format);
          dumpPayload(stdout);
        }
	idx = save_idx;
	return -1;
    }
  }
  error("tellStrInPayload: past end of Payload: idx=%d len=%d", idx, len); 
  idx = save_idx;
  return -1; 
}

void Payload::dumpPayload(FILE *f)
{
  fprintf(f, "dumpPayload: len=%d 0x%03x, idx=%d 0x%03x\n", len, len, idx, idx);

  char adr[4], hex[49], asc[17];

  for (int i=0; i<len; ) {
    memset(adr, 0, sizeof(adr));
    memset(hex, 0, sizeof(hex));
    memset(asc, 0, sizeof(asc));
    sprintf(adr, "%03x", i);
    for (int j=0; j<16; j++, i++) {
      int c = (uint8_t) data[i];
      sprintf(&hex[j*3], "%02x ", c);
      if (isprint(c))
        sprintf(&asc[j], "%c", c);
      else
        sprintf(&asc[j], ".");
      if (i == len-1) {
        sprintf(&hex[(j+1)*3], ">  ");
        sprintf(&asc[j+1], ">");
        i = len;
        break;
      }
    }
    fprintf(f, "%s:  %s  | %s |\n", adr, hex, asc);
  }
  fflush(f);
}

#if 0 // UNUSED
Payload * Payload::rewindPayload()
{
  idx = 0;
  return this;
}

void Payload::appendPayload(const Payload *a)
{
  if (! isValidPayload() || ! a->isValidPayload())
    error("appendPayload: invalid Payload");
  memcpy(data + idx, a->data, a->len);
  idx += a->len;
  if (idx > len)
    len = idx;
} 
#endif
