#include "global.h"

#include "net.h"
#include "netdefs.h"
#include "payload.h"	// Payload
#include "channel.h"	// getCurrentChannel

#include "wo.h"		// WObject
#include "world.h"	// isValidType
#include "user.h"	// USER_TYPE
#include "icon.h"	// ICON_TYPE


/* Send a multicast packet of type '0x02' = Delta */
void NetObject::sendDelta(const uint8_t prop_id)
{
  if (! netprop) {
    error("sendDelta: netprop NULL");
    return;
  }

  NetProperty *pprop = netprop + prop_id;
  pprop->setResponsible(true);
  pprop->resetDates();

  Payload pp;
  pp.putPayload("cnch", VREP_DELTA, noid, prop_id, pprop->version);

  /*** TODO: the timevals ***/

  if (prop_id >= propertiesNumber()) {
    error("sendDelta: prop_id wrong");
    return;
  }
  putProperty(prop_id, &pp);
  Channel *pchan = Channel::getCurrentChannel();
  if (pchan)
    pp.sendPayload(pchan->sa[SA_RTP]);
}

/* Send a '0x01' packet to mentionned unicast address for the current object */
void NetObject::sendCreate(const struct sockaddr_in *to)
{
  Payload pp;
  pp.putPayload("ccnc", VREP_CREATE, type, noid, permanent);
  putAllProperties(&pp);  

  uint8_t nprop = propertiesNumber();
  for (int i=0; i < nprop; i++)
    pp.putPayload("h", netprop[i].version);
  trace(DBG_NET, "sendCreate: nobj=%s to=%s", noid.getNetNameById(), inet4_ntop(&to->sin_addr));
  pp.sendPayload(to);
}

/* Heuristic to avoid to send bunch of Query */
int NetObjectId::filterQuery()
{
  static NetObjectId oldnoid;
  static int countDelta = 0;

  if (equalNetObjectId(oldnoid)) { 
    // last request was on this name
    if (++countDelta <= 15)
      // 15: 10 proprietes en moyenne, avec 15 on saute donc
      // sans doute un "bloc" de deltas, mais sans doute pas deux 
      return 0;    // cancel this request
  }
  // it's another one
  oldnoid = *this;
  countDelta = 0;
  return 1;
}

/* Send a Query '0x03' packet  to the unicast sender */
void NetObjectId::sendQueryNoid(const struct sockaddr_in *to)
{
  if (! filterQuery())
    return;

  Payload pp;
  pp.putPayload("cn", VREP_QUERY, this);
  trace(DBG_NET, "sendQuery: nobj=%s to=%s", getNetNameById(), inet4_ntop(&to->sin_addr));
  pp.sendPayload(to);
}

/* Send a Delete '0x04' packet */
void NetObjectId::sendDeleteNoid(const struct sockaddr_in *to)
{
  Payload pp;
  pp.putPayload("cn", VREP_DELETE, this);
  trace(DBG_NET, "sendDelete: nobj=%s to=%s", getNetNameById(), inet4_ntop( &to->sin_addr));
  pp.sendPayload(to);
} 

void NetObject::sendDelete(const struct sockaddr_in *to)
{
  Payload pp;
  pp.putPayload("cn", VREP_DELETE, noid);
  trace(DBG_NET, "sendDelete: nobj=%s to=%s", noid.getNetNameById(), inet4_ntop(&to->sin_addr));
  pp.sendPayload(to);
} 

#if 0 //notused
void NetObject::sendPersistSet(const struct sockaddr_in *to)
{
  Payload pp;
  pp.putPayload("cn", VREP_PERSIST_SET, noid);
  trace(DBG_NET, "sendObjPersist: nobj=%s to=%p", noid.getNetNameById(), to);
  pp.sendPayload(to);
} 

void NetObject::sendPersistInfo(const struct sockaddr_in *to)
{
  Payload pp;
  pp.putPayload("cn", VREP_PERSIST_INFO, noid);
  trace(DBG_NET, "sendPersistInfo: nobj=%s to=%p", noid.getNetNameById(), to);
  pp.sendPayload(to);
} 

void NetObject::sendPersistReset(const struct sockaddr_in *to)
{
  Payload pp;
  pp.putPayload("cn", VREP_PERSIST_RESET, noid);
  trace(DBG_NET, "sendPersistReset: nobj=%s to=%p", noid.getNetNameById(), to);
  pp.sendPayload(to);
} 

void NetObject::sendWorldSet(const struct sockaddr_in *to)
{
  Payload pp;
  pp.putPayload("cn", VREP_WORLD_SET, noid);
  trace(DBG_NET, "sendWorldSet: nobj=%s to=%p", noid.getNetNameById(), to);
  pp.sendPayload(to);
} 

void NetObject::sendWorldInfo(const struct sockaddr_in *to)
{
  Payload pp;
  pp.putPayload("cn", VREP_WORLD_INFO, noid);
  trace(DBG_NET, "sendWorldInfo: nobj=%s to=%p", noid.getNetNameById(), to);
  pp.sendPayload(to);
} 

void NetObject::sendWorldReset(const struct sockaddr_in *to)
{
  Payload pp;
  pp.putPayload("cn", VREP_WORLD_RESET, noid);
  trace(DBG_NET, "sendWorldReset: nobj=%s to=%p", noid.getNetNameById(), to);
  pp.sendPayload(to);
} 
#endif //notused

/* Exported to WO, to call for each new object */
void NetObject::declareObjCreation()
{
  if (! noid.getNetObject()) {
    warning("declareObjCreation: unnamed netobject (type=%d)", type);
    return;
  }
  if (ntohl(noid.src_id) == 1) {
    //warning("declareObjCreation: not a new netobject (type=%d)", type);
    return;
  }
  Channel *pchan = Channel::getCurrentChannel();
  if (pchan)
    sendCreate(pchan->sa[SA_RTP]);
}  

/* Updates netobject version */
void NetObject::declareObjDelta(const uint8_t prop_id)
{
  if (noid.src_id == 0) {
    error("declareObjDelta: unnamed netobject (type=%d)", type);
    return;
  }

  uint16_t nprop = propertiesNumber();
  if (prop_id >= nprop) {
    error("declareObjDelta: invalid prop_id=%d > nprop=%d (type=%d)", prop_id, nprop, type); 
    return;
  }
  NetProperty *pprop = netprop + prop_id;
  pprop->setResponsible(true);
  pprop->version += 1 + abs(rand() % MAX_VERSION_JUMP); /* %10 */
  sendDelta(prop_id);
  //error("declareObjDelta: prop_id=%d prop_version=%d (type=%d)", prop_id, pprop->version, type); 
}

/* Destroy the netobject (its local copy) */
void NetObject::declareDeletion()
{
  if (! noid.getNetObject())
    return;

  if (permanent) {
    warning("declareDeletion: on permanent object (type=%d)", type);
    return;
  }

  Channel *pchan = Channel::getCurrentChannel();
  if (pchan)
    sendDelete(pchan->sa[SA_RTP]);
}

#if 0 //notused
void NetObject::declarePersistSet()
{
  Channel *pchan = Channel::getManager();
  sendPersistSet(pchan->sa[SA_RTP]);
}
void NetObject::declarePersistInfo()
{
  Channel *pchan = Channel::getManager();
  sendPersistInfo(pchan->sa[SA_RTP]);
}
void NetObject::declarePersistReset()
{
  Channel *pchan = Channel::getManager();
  sendPersistReset(pchan->sa[SA_RTP]);
}
void NetObject::declareWorldSet()
{
  Channel *pchan = Channel::getManager();
  sendWorldSet(pchan->sa[SA_RTP]);
}
void NetObject::declareWorldInfo()
{
  Channel *pchan = Channel::getManager();
  sendWorldInfo(pchan->sa[SA_RTP]);
}
void NetObject::declareWorldReset()
{
  Channel *pchan = Channel::getManager();
  sendWorldReset(pchan->sa[SA_RTP]);
}
#endif //notused


/* Incoming Delta */
void Payload::incomingDelta(const struct sockaddr_in *from) 
{
  NetObjectId noid;	// name received
  uint8_t prop_id;	// property number received
  int16_t vers_id;	// version received

  if (getPayload("nch", &noid, &prop_id, &vers_id) < 0) {
    if (getPayload("nhh", &noid, &prop_id, &vers_id) < 0) // VREP_1
      return;
  }

  NetObject *pn;
  if ((pn = noid.getNetObject()) == NULL) {
    /* delta on an unknown object */
    trace(DBG_NET, "incomingDelta sendQuery on: %s, from=%s, p=%d, v=%d",
                   noid.getNetNameById(), inet4_ntop(&from->sin_addr), prop_id, vers_id);
    // send a Query to the sender in unicast
    noid.sendQueryNoid(from);
    return;
  }

  /* verify prop_id */
  uint8_t nprop = pn->propertiesNumber();
  if (prop_id >= nprop) {
    warning("incomingDelta: invalid property id (%d)"
	    "(type is %d, nprop is %d)", prop_id, pn->type, nprop);
    return;
  }
  NetProperty *pprop = pn->netprop + prop_id;

  /*
   * depends on version
   */
  // in complement to 2: d gives the distance, same throught the boundary
  int16_t d;		// versions difference
  d = pprop->version - vers_id;
  if (abs(d) > 5000) {	// very far
    warning("incomingDelta: very different versions: mine is %d, received %d",
             pprop->version, vers_id);
  }
  if (d > 0) {
    /* mine is more recent */
    return;
  }

  pprop->resetDates();
  if (d < 0) {
    /* mine is less recent, its own is more recent */
    pn->getProperty(prop_id, this);
    pprop->setResponsible(false); // leave responsibility to the other one
    pprop->version = vers_id;
    return;
  }
  /* same versions, 2 responsibles: conflict */
  else if (pprop->responsible) {
    // resolved by getting a new random version
    // publishes new version to sender in unicast
    pn->declareObjDelta(prop_id);
    // warning("Conflict resol: obj=%s, prop=%d, changing vers_num %d->%d",
    // pn->noid.getNetNameById(), prop_id, vers_id, pn->prop[prop_id].version);
  }
  // else, it's just a "recall" (then nothing to do)
}

/* Incoming Create */
void Payload::incomingCreate(const struct sockaddr_in *from) 
{
  NetObjectId noid;
  uint8_t type_id, _permanent;

  if (getPayload("cnc", &type_id, &noid, &_permanent) < 0) {
    if (getPayload("dnc", &type_id, &noid, &_permanent) < 0) // VREP_1
      return;
  }
  if (noid.getNetObject())
    return;	// local copy already exists -> ignore this request

  trace(DBG_NET, "incomingCreate: nobj=%s (type=%d), _permanent=%d", 
	         noid.getNetNameById(), type_id, _permanent);
  
  //
  // creates the replicated object
  // glue with WObject
  // Very very important !!!
  //
  NetObject *pn = NetObject::replicateObject(type_id, noid, this);
  if (!pn) {
    error("incomingCreate: can't replicate object, type=%d", type_id);
    return;
  }

#if 0
  if (pn->type != type_id) {
    error("incomingCreate: bad type=%d", type_id);
    return;
  }
#endif
  if (! noid.equalNetObjectId(pn->noid)) {
    error("incomingCreate: bad noid=%s", noid.getNetNameById());
    return;
  }
  pn->permanent = _permanent;
  pn->initProperties(false); // we are not responsible
  pn->insertNetObject();

  // get properties
  uint8_t nprop = pn->propertiesNumber();
  // dumpPayload(stdout);
  for (int i=0; i < nprop; i++) {
    if (getPayload("h", &(pn->netprop[i].version)) < 0)
      return;
  }
}

/* Incoming Query */
void Payload::incomingQuery(const struct sockaddr_in *from) 
{
  NetObjectId noid;

  if (getPayload("n", &noid) < 0)
    return;
  if (noid.port_id == 0)	//HACK!
    return;

  trace(DBG_NET, "incomingQuery: nobj=%s from=%s",
                 noid.getNetNameById(), inet4_ntop(&from->sin_addr));
  NetObject *pn;
  if ((pn = noid.getNetObject()) == NULL) {
    // unknown object: may be we have deleted it, we advertize the requester
    trace(DBG_NET, "incomingQuery: sendDelete nobj=%s from=%s",
                   noid.getNetNameById(), inet4_ntop(&from->sin_addr));
    noid.sendDeleteNoid(from);
  }
  else {
    // object known, but not properties, ask them to sender
    trace(DBG_NET, "incomingQuery: sendCreate nobj=%s from=%s",
                   noid.getNetNameById(), inet4_ntop(&from->sin_addr));
    pn->sendCreate(from);
  }
}

/* Incoming Delete */
void Payload::incomingDelete(const struct sockaddr_in *from) 
{
  NetObjectId noid;

  if (getPayload("n", &noid) < 0)
    return;

  trace(DBG_NET, "incomingDelete: nobj=%s from=%s",
                 noid.getNetNameById(), inet4_ntop(&from->sin_addr));
  NetObject *pn;
  if ((pn = noid.getNetObject()))
    pn->requestDeletionFromNetwork();
}

#if 0 //notused
void Payload::incomingPersist(const struct sockaddr_in *from) 
{
  NetObjectId noid;

  if (getPayload("n", &noid) < 0)
    return;

  error("incomingPersist: nobj=%s from=%p", noid.getNetNameById(), from);
#if 0
  NetObject *pn;
  if (pn = noid.getNetObject())
     requestPersistFromNetwork(pn);
#endif
}
void Payload::incomingWorld(const struct sockaddr_in *from) 
{
  NetObjectId noid;

  if (getPayload("n", &noid) < 0)
    return;

  error("incomingWorld: nobj=%s from=%p", noid.getNetNameById(), from);
}
#endif //notused

void Payload::incomingOther(const struct sockaddr_in *from, const int size) 
{
  trace(DBG_FORCE, "IncomingOther: size=%d from %lx/%x",
        size, ntohl(from->sin_addr.s_addr), ntohs(from->sin_port));
  if (size) {
    trace(DBG_FORCE,
          "%02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x",
          data[0], data[1], data[2], data[3],
          data[4], data[5], data[6], data[7],
          data[8], data[9], data[10], data[11],
          data[12], data[13], data[14], data[15]);
  }
}

/* When something is available on file-descriptor "fd" */
void NetIncoming(const int fd) 
{
  while (1) {
    fd_set set;
    struct timeval timeout;

    FD_ZERO(&set);
    FD_SET(fd, &set);
    timeout.tv_sec = timeout.tv_usec = 0;
    if (select(FD_SETSIZE, &set, NULL, NULL, &timeout) == 0)
      break;

    /*
     * read the packet
     */
    struct sockaddr_in from;	// sender unicast address
    int r;

    Payload *pp = new Payload();	// create the payload to get the data

    if ((r = pp->recvPayload(fd, &from)) <= 0) {
      delete pp;
      return;	// read error <0 or ignore == 0
    }

    /*
     * test if this packet belongs to VREP (VREng Protocol)
     */
    uint8_t cmd;
    if (pp->len > 0 && pp->getPayload("c", &cmd) >= 0) {
      //
      // parse VREP
      //
      switch (cmd) {
      case VREP_CREATE:
      case VREP_CREATE_V1:
           pp->incomingCreate(&from);
           delete pp;
           return;
      case VREP_DELTA:
      case VREP_DELTA_V1:
           pp->incomingDelta(&from);
           delete pp;
           return;
      case VREP_QUERY:
      case VREP_QUERY_V1:
           pp->incomingQuery(&from);
           delete pp;
           return;
      case VREP_DELETE:
      case VREP_DELETE_V1:
           pp->incomingDelete(&from);
           delete pp;
           return;
#if 0 //notused
      case VREP_PERSIST_INFO:
      case VREP_PERSIST_RESET:
           pp->incomingPersist(&from);
      case VREP_WORLD_INFO:
      case VREP_WORLD_RESET:
           pp->incomingWorld(&from);
           delete pp;
           return;
#endif //notused
      default:
           warning("Incoming unknown cmd: X'%02x'", cmd);
           error("Incoming on fd=%d from %lx/%x (mine is %lx/%x)",
	         fd, ntohl(from.sin_addr.s_addr), ntohs(from.sin_port),
	         getMyHostId(), getMyPortId());
           delete pp;
           return;
      }
    }
    else {
      // empty or invalid payload
      pp->incomingOther(&from, r);
      //error("NetIncoming other=%x", r);
      delete pp;
      return;
    }
  }
}


/* Check if some responsibilities should to be taken when a timeout occurs */
int NetTimeout()
{
  float refresh = DEF_REFRESH_TIMEOUT;
  
  //statAdjust();
  struct timeval now;
  gettimeofday(&now, NULL);

  /* timeout grows as log(nbsources) to keep bandwidth confined */
#ifdef REDUCE_BW
  nsrc = Source::getSourcesNumber() - 1;
  nsrc = (nsrc <= 1) ? 1 : nsrc;
  refresh = DEF_REFRESH_TIMEOUT * ((1.0 + (float) log((double) nsrc)) / 2);
  error("refresh=%.2f nsrc=%d", refresh, nsrc);
#endif
  
  /* 
   * for each netobject in netobjectlist
   */
  for (NetObject *pn = NetObject::getNetObjectsList() ; pn ; ) {
    if (! isValidType(pn->type)) {
      error("NetTimeout: invalid type (%d)", pn->type);
      return -1;
    }

    NetObject *next = pn->next;	// save it now, pn may disapear if death

    /*
     * scan its properties
     */
    uint8_t nprop = pn->propertiesNumber();
    for (int i=0; i < nprop; i++) {
      switch (pn->type) {
      case USER_TYPE:
        if (i < USER_PROPBEGINVAR || i > USER_PROPENDVAR)
          //trace(DBG_NET, "skip property %d", i); // skip static properties
          continue;
#if 0 //pd
      case ICON_TYPE:
        if (i < ICON_PROPHNAME || i > ICON_PROPAX)
          continue;
#endif
      }
      NetProperty *pprop = pn->netprop + i;

      /*
       * test if refresh timeout reached
       */
      if (pprop->responsible && diffDates(pprop->last_seen, now) > refresh) {
	// now - last_seen > refresh: we are responsible, we must refresh

	pn->sendDelta(i);	// publish a heartbeat

        //error("NetTimeout: sendDelta type (%d)", pn->type);
      }

      if (diffDates(pprop->assume_at, now) > 0) {
	// now > assume_at: assume's timeout on this property
	if (pn->permanent) {
          /*
           * permanent object (door, lift, anim, button, thing, book)
           */
	  if (pprop->version != 0) {
	    // if permanent prop hasn't its initial value,
	    // somebody must assume responsibility
	    trace(DBG_FORCE, "NetTimeout: (permanent) Assuming responsibility for %s prop=%d unseen for %5.2fs", 
		  pn->noid.getNetNameById(), i, diffDates(pprop->last_seen, now));
	    if (pprop->responsible) {
	      trace(DBG_FORCE, "NetTimeout: (permanent) should assume responsibility "
		    "of something I am responsible for");
              return -1;
            }
            /* heartbeat */
	    pn->declareObjDelta(i);	// assume responsibility: publish
	  }
	}
        else { 
	  /*
           * volatile object (user, ball, dart, bullet, sheet, icon)
           */
	  if (pprop->responsible) {
	    trace(DBG_FORCE, "NetTimeout: (volatile) should assume death of %s I am responsible for", pn->pobject->name.instance_name);
            return -1;
	  } 
	  trace(DBG_FORCE, "NetTimeout: (volatile) Assuming death of %s [%s] (unseen for %.2fs)", 
                  pn->pobject->name.instance_name, pn->noid.getNetNameById(), diffDates(pprop->last_seen, now));
	  pn->declareDeletion();
	  pn->requestDeletionFromNetwork();	// discard the dead
	  // no reason to continue after a requestDeletion
          break;
	} 
      }
    } // end scan properties
    pn = next;	// next object
  } // end list
  return (int) ceil(refresh * 1000);
}
