#include "global.h"

#include "net.h"
#include "netdefs.h"
#include "channel.h"	// Channel
#include "rtpsess.h"

#include "world.h"	// getCurrentName

extern uint32_t random32(const int type);


static Session *sessionList;		// RTP sessions
static uint16_t sourcesCounter;		// sources counter
static char rtpName[RTPNAME_LEN];	// rtp name
static char rtpEmail[EMAIL_LEN];	// rtp email
static char toolName[TOOL_LEN];		// tool name


#define CHECK_SESSION_LIST \
  { Session *tmp = sessionList; \
    while (tmp != NULL) { \
      if (tmp->next == tmp) \
        warning("RtpSession list invalid at %s:%d", __FILE__, __LINE__); \
        break; \
      tmp = tmp->next; \
    } \
  }


/*
 * RTP header
 */

uint16_t Rtp::createSeq()
{
  time_t t;

  srand((uint32_t) time(&t));
  return rand();
}

uint32_t Rtp::createSsrc(const int value)
{
  time_t t;

  srand((uint32_t) time(&t));
  return (uint32_t) random32(value);
}

void Session::buildRtpHeader(rtp_hdr_t *rtp_hdr, const uint32_t _ssrc)
{
  struct timeval ts;

  rtp_hdr->version = RTP_VERSION;
  rtp_hdr->p = 0;
  rtp_hdr->x = 0;
  rtp_hdr->cc = 0;
  rtp_hdr->m = 0;
  rtp_hdr->pt = PAYLOAD_TYPE;
  rtp_hdr->seq = htons(++(rtp_seq));
  gettimeofday(&ts, NULL);
  rtp_hdr->ts = htonl(ts.tv_sec*1000 + ts.tv_usec/1000);
  rtp_hdr->ssrc = htonl(_ssrc);

#ifdef DEBUG
  trace(DBG_RTP, "RTP: %02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x",
        hdr[0], hdr[1], hdr[2], hdr[3], hdr[4], hdr[5], hdr[6], hdr[7], hdr[8], hdr[9], hdr[10], hdr[11]);
#endif
}

void Rtp::initSeq(sourceInfos *s, const uint16_t seq)
{
  s->base_seq = seq - 1;
  s->max_seq = seq;
  s->bad_seq = RTP_SEQ_MOD + 1;
  s->cycles = 0;
  s->received = 0;
}

int Rtp::updateSeq(sourceInfos *s, const uint16_t seq)
{
  uint16_t udelta = seq - s->max_seq;
  const int MAX_DROPOUT = 3000;
  const int MAX_MISORDER = 100;
  const int MIN_SEQUENTIAL = 2;

  /*
   * Source is not valid until MIN_SEQUENTIAL packets with
   * sequential sequence numbers have been received.
   */
  if (s->probation) {
    /* packet is in sequence */
    if (seq == s->max_seq + 1) {
      s->probation--;
      s->max_seq = seq;
      if (s->probation == 0) {
	initSeq(s, seq);
	s->received++;
	return 1;
      }
    } else {
      s->probation = MIN_SEQUENTIAL - 1;
      s->max_seq = seq;
    }
    return 0;
  } else if (udelta < MAX_DROPOUT) {
    /* in order, with permissible gap */
    if (seq < s->max_seq) {
      /*
       * Sequence number wrapped - count another 64K cycle.
       */
      s->cycles += RTP_SEQ_MOD;
    }
    s->max_seq = seq;
  } else if (udelta <= RTP_SEQ_MOD - MAX_MISORDER) {
    /* the sequence number made a very large jump */
    if (seq == s->bad_seq) {
      /*
       * Two sequential packets -- assume that the other side
       * restarted without telling us so just re-sync
       * (i.e., pretend this was the first packet).
       */
      initSeq(s, seq);
    }
    else {
      s->bad_seq = (seq + 1) & (RTP_SEQ_MOD-1);
      return 0;
    }
  } else {
    /* duplicate or reordered packet */
  }
  s->received++;
  return 1;
}

/*
 * Handling RTCP SDES
 */

const char * Rtp::getRtpName(char *name)
{
  FILE *fp;
  char *p, buf[BUFSIZ];

  p = getenv("HOME");
  if (p == NULL || *p == '\0')
    return NULL;
  sprintf(buf, "%s/.RTPdefaults", p);
  if ((fp = fopen(buf, "r"))) {
    while (fgets(buf, sizeof(buf), fp)) {
      buf[strlen(buf) -1] = '\0';
      if (strncmp(buf, "*rtpName:", 9) == 0) {
        p = strchr(buf, ':');
        p += 2;
        strcpy(name, p);
        fclose(fp);
        return name;
      }
    }
    fclose(fp);
  }
  return NULL;
}

const char * Rtp::getRtcpEmail()
{
  char host[MAXHOSTNAMELEN], hostname[MAXHOSTNAMELEN];

  gethostname(hostname, MAXHOSTNAMELEN);

  struct hostent *ph = my_gethostbyname(hostname, AF_INET);
  strcpy(host, ph->h_name);
  my_free_hostent(ph);

#if HAVE_GETPWUID
  struct passwd *pwd;
  if ((pwd = getpwuid(getuid())) != NULL)
    sprintf(rtpEmail, "%s@%s", pwd->pw_name, host);
  else
    sprintf(rtpEmail, "unknown@%s", host);
#else
    sprintf(rtpEmail, "unknown@%s", host);
#endif
  return rtpEmail;
}

const char * Rtp::getRtcpName()
{
  const char *p;
  char name[RTPNAME_LEN];

  if ((p = getRtpName(name)) == NULL)
    p = getRtcpEmail();
  else
    p = name;
  strcpy(rtpName, p);
  return rtpName;
}

char * Rtp::getRtcpTool()
{
  sprintf(toolName, "VREng-%s", PACKAGE_VERSION);
  return toolName;
}

bool Rtp::isSdesType(const uint8_t sdes_type)
{
  switch (sdes_type) {
  case RTCP_SDES_CNAME:
  case RTCP_SDES_NAME:
  case RTCP_SDES_EMAIL:
  case RTCP_SDES_LOC:
  case RTCP_SDES_TOOL:
    return true;
  }
  return false;
}

char * Rtp::getSdesItemBySsrc(const uint8_t sdes_type, const uint32_t _ssrc)
{
  SdesItem *sitem;

  //pd CHECK_SESSION_LIST 
  for (Session *pse = sessionList; pse && pse->mode == WORLD_MODE; pse = pse->next) {
    for (Source *pso = pse->source; pso ; pso = pso->next) {
      if (pso == pso->next) {
        //pd warning("Infinite loop in getSdesItemBySsrc");	// BUG ?
        //pso->dump();
        return NULL;
      }

      if (pso->ssrc == _ssrc) {
        for (sitem = &pso->sdes; sitem ; sitem = sitem->si_next) {
          if (sitem < (SdesItem *) 999)		// bug: bad sitem
            return NULL;
          if (! isSdesType(sitem->si_type))
            continue;
          if (sitem->si_len == 0)
            continue;
          if (sitem->si_type == sdes_type) {
            return (char *) sitem->si_str;
          }
        }
      }
    }
  }
  return NULL;
}

char * Rtp::getRtcpNameBySsrc(const uint32_t _ssrc)
{
  return getSdesItemBySsrc(RTCP_SDES_NAME, _ssrc);
}

char * Rtp::getRtcpEmailBySsrc(const uint32_t _ssrc)
{
  return getSdesItemBySsrc(RTCP_SDES_EMAIL, _ssrc);
}

char * Rtp::getRtcpToolBySsrc(const uint32_t _ssrc)
{
  return getSdesItemBySsrc(RTCP_SDES_TOOL, _ssrc);
}

SdesItem * Rtp::allocSdesItem()
{
  return (SdesItem *) calloc(1, sizeof(SdesItem));
}

#ifdef OLD_SSRC
void Rtp::updateSSRC(uint32_t _ssrc)
{
  if ((NetObject *pn = getObjectBySSRC(_ssrc)) == 0) {
    warning("getObjectBySSRC: object not found");
  }
  else {
    if (pn->s == NULL) {
      warning("recvPacket: not yet sourced");
      pn->s = (sourceInfos *) calloc(1, sizeof(sourceInfos));
      /* Rtp::initSeq(pn->s, session.rtp_seq); */
    }
    /* (pn->s->received)++; */
  }
}
#endif /* OLD_SSRC */


/*
 * Handling Sources
 */

/**
 * creates a new Source
 */
Source::Source(const uint32_t _ssrc)
{
  new_source++;
  ssrc = _ssrc;
  extended_max = 0;
  expected = 0;
  lost = 0;
  sdes.si_type = 0;
  sdes.si_len = 0;
  sdes.si_str = NULL;
  sdes.si_next = NULL;
  next = this;	/* P. Bellot */
}

/**
 * finds source by its ssrc else creates a new source
 */
Source * Source::getSource(const uint32_t _ssrc)
{
  Channel *pchan = Channel::getCurrentChannel(); // maybe indeterminated...
  assert(pchan->session->source);

  Source *pso, *psolast;
  int i = 0;

  for (pso = psolast = pchan->session->source;
       pso && (i < sourcesCounter); pso = pso->next, i++) {
    if (pso->ssrc == _ssrc)
      return pso;
    psolast = pso;
  }
  /* source not yet registered */
  if (! pso) {
    /* no source found, we must create it now */
    pchan->session->source = new Source(_ssrc);

    pchan->session->incrSources();
    trace(DBG_RTP, "getSource: create source=%p", pchan->session->source);
    return pchan->session->source;
  }
  else {
    /* new incoming source: create this source */
    psolast->next = new Source(_ssrc);

    pchan->session->incrSources();
    return psolast->next;
  }
}

/**
 * Source destructor
 */
Source::~Source()
{
  del_source++;
}

void Session::deleteSourceBySsrc(const uint32_t _ssrc)
{
  Source *pso, *psolast;
  int i = 0;
  
  for (psolast = pso = source; pso && (i < nbsources); pso = pso->next, i++) {
    if (pso->ssrc == _ssrc) {
      if (psolast == NULL)	// no source found
        source = NULL;
      else {
        // MS : this is a NOOP if pso == source
        // Bad things happen after that
        // psolast->next = pso->next;
        if (pso == source)
          source = pso->next;
        else
          psolast->next = pso->next;
      }
      sum_pkts_lost += pso->lost;
      nbsources = --sourcesCounter;
      trace(DBG_RTP, "nbsources--=%d", nbsources);
      delete pso;		// delete Source
      pso = NULL;
      break;
    }
    psolast = pso;
  }
}

uint16_t Source::getSourcesNumber()
{
  return sourcesCounter;
}

uint16_t Source::getMembersNumber()
{
  uint16_t members = 0;

  //pd CHECK_SESSION_LIST 
  for (Session *pse = sessionList; pse && pse->mode; pse = pse->next) {
    for (Source *pso = pse->source; pso ; pso = pso->next)
      members++;
  }
  return members; 
}

void Source::dump()
{
  if (&sdes == NULL)
    return;	// BUG! segfault

  fprintf(stderr, "this=%p\n", this);
  fprintf(stderr, "ssrc=%x\n", ssrc);
  fprintf(stderr, "lost=%d\n", lost);
  fprintf(stderr, "next=%p\n", next);

  SdesItem *sitem;
  int i;
  for (i=0, sitem = &sdes; sitem ; sitem = sitem->si_next, i++) {
    if (sitem->si_type > RTCP_SDES_END && sitem->si_type <= RTCP_SDES_SOURCE && sitem->si_len >0 && sitem->si_len < 128 && sitem->si_str)
      fprintf(stderr, "  sdes[%d]=%s\n", i, sitem->si_str);
  }
  fflush(stderr);
}

void Source::dumpAll()
{
  uint32_t * ssrctab = new uint32_t[100];

  //pd CHECK_SESSION_LIST 
  for (Session *pse = sessionList; pse && pse->mode && pse->nbsources > 1; pse = pse->next) {
    int i = 0;
    for (Source *pso = pse->source; pso && i < 100; pso = pso->next, i++) {
      if (pso->ssrc == 0) {
        delete[] ssrctab;	// no more sources
        return;
      }
      int yet = 0;
      for (int j=0; ssrctab[j]; j++) {
        if (ssrctab[j] == pso->ssrc) {
          yet = 1;
          //pd continue;
          break;
        }
      }
      if (yet == 0) {
        ssrctab[i] = pso->ssrc;
        pso->dump();
      }
    }
  }
  delete[] ssrctab;
}


/*
 * Handling Sessions
 */

/**
 * creates a new Session
 */
Session::Session()
{
  new_session++;
  group = 0;
  rtp_port = 0;
  rtcp_port = 0;
  ttl = 0;
  mode = 0;
  nbsources = 0;
  source = NULL;
  mysdes  = NULL;
  next  = NULL;
  sr.psent = 0;
  sr.osent = 0;

  if (! sessionList) {
    sessionList = this;
    next = NULL;
  }
  else {
    next = sessionList;
    sessionList = this;
  }
}

void Session::clearSessionsList()
{
  sessionList = NULL;
}

void Session::incrSources()
{
  nbsources = ++sourcesCounter;
  trace(DBG_RTP, "getSource: nbsources=%d", nbsources);
}

/**
 * initialize variables for a session already allocated, return local ssrc
 */
uint32_t Session::createSession(uint32_t _group, uint16_t _rtp_port, uint8_t _ttl, uint32_t _ssrc)
{
  group = htonl(_group);
  rtp_port = htons(_rtp_port);
  rtcp_port = htons(_rtp_port + 1);
  ttl = _ttl;
  trace(DBG_RTP, "createSession: rtp_port=%x rtcp_port=%x", rtp_port, rtcp_port);

  /* seq number initialization */
  rtp_seq = Rtp::createSeq();

  /* SSRC number initialization */
  if (_ssrc)
    ssrc = _ssrc;	// ssrc already used
  else
    ssrc = htonl(Rtp::createSsrc(rtp_seq));	// new ssrc

  trace(DBG_RTP, "createSession: ssrc=%x", ssrc);
  rtp_hdr.ssrc = ssrc;
  sr.ssrc = ssrc;

  /* alloc Source */
  source = new Source(ssrc);

  nbsources = ++sourcesCounter;
  trace(DBG_RTP, "createSession: nbsources=%d", nbsources);

  Rtp::initSeq(&source->s, rtp_seq);
  createMySdes();

  return ssrc;
}

void Session::deleteSessionBySsrc(const uint32_t _ssrc)
{
  //pd CHECK_SESSION_LIST 
  for (Session *pse = sessionList; pse; pse = pse->next) {
    for (Source *pso = pse->source; pso; pso = pso->next) {
      if (pso->ssrc == _ssrc) {
        trace(DBG_RTP, "deleteSessionBySsrc: ssrc=%x found", _ssrc);
        pse->deleteSourceBySsrc(_ssrc);
        return;
      }
    }
  }
  error("deleteSessionBySsrc: ssrc=%x not found", _ssrc);
}

#if 0
/*
 * Free a Session - strange: never called
 */
void Session::freeSession()
{
  for (Session *pse = sessionList; pse ; pse = pse->next) {
    if (pse == this) {
      sessionList = pse->next;
      pse->next = NULL;
      if (pse == sessionList)
        sessionList = NULL;
      delete this;	// BAD -> DELETE SOMEWHERE !
      return;
    }
  }
  sessionList = NULL;
}
#endif

/**
 * Session destructor
 */
Session::~Session()
{
  del_session++;
  trace(DBG_RTP, "~Session");
  freeMySdes();
  deleteSourceBySsrc(getMySsrcId());
}

/**
 * creates my SDES item
 */
void Session::createMySdes()
{
  SdesItem *scname, *sname, *semail, *stool, *sloc;

  trace(DBG_RTP, "CreateSDES pse: %p", this);

  Rtp::getRtcpName();	// fill rtpName
  Rtp::getRtcpEmail();	// fill rtpEmail
  Rtp::getRtcpTool();	// fill toolName

  trace(DBG_RTP, "createMySdes: rtpName=%s, rtpEmail=%s", rtpName, rtpEmail);

  if ((scname = Rtp::allocSdesItem()) == NULL) return;
  mysdes = scname;
  scname->si_type = RTCP_SDES_CNAME;
  scname->si_len = strlen(rtpEmail);
  scname->si_str = (uint8_t *) rtpEmail;
  
  if ((sname = Rtp::allocSdesItem()) == NULL) return;
  scname->si_next = sname;
  sname->si_type = RTCP_SDES_NAME;
  sname->si_len = strlen(rtpName);
  sname->si_str = (uint8_t *) rtpName;
  
  if ((semail = Rtp::allocSdesItem()) == NULL) return;
  sname->si_next = semail;
  semail->si_type = RTCP_SDES_EMAIL;
  semail->si_len = strlen(rtpEmail);
  semail->si_str = (uint8_t *) rtpEmail;
  
  if ((sloc = Rtp::allocSdesItem()) == NULL) return;
  semail->si_next = sloc;
  sloc->si_type = RTCP_SDES_LOC;
  if (World::getCurrentName()) {
    sloc->si_len = strlen(World::getCurrentName());
    sloc->si_str = (uint8_t *) World::getCurrentName();
  }
  else {
    sloc->si_len = strlen(MANAGER_NAME);
    sloc->si_str = (uint8_t *) MANAGER_NAME;
  }
  
  if ((stool = Rtp::allocSdesItem()) == NULL) return;
  sloc->si_next = stool;
  stool->si_type = RTCP_SDES_TOOL;
  stool->si_len = strlen(toolName);
  stool->si_str = (uint8_t *) toolName;

  stool->si_next = NULL;
}

/**
 * refreshes my SDES item
 */
void Session::refreshMySdes()
{
  SdesItem *scname, *sname, *semail, *sloc;

  trace(DBG_RTP, "refreshMySdes: pse=%p", this);

  if ((scname = mysdes) == NULL) return;
  if ((sname = scname->si_next) == NULL) return;
  if ((semail = sname->si_next) == NULL) return;
  if ((sloc = semail->si_next) == NULL) return;
  sloc->si_type = RTCP_SDES_LOC;
  if (World::getCurrentName()) {
    sloc->si_len = strlen(World::getCurrentName());
    sloc->si_str = (uint8_t *) World::getCurrentName();
  }
  else {
    sloc->si_len = strlen(MANAGER_NAME);
    sloc->si_str = (uint8_t *) MANAGER_NAME;
  }
  trace(DBG_RTP, "sloc: %p %s", sloc, sloc->si_str);
}

void Session::freeMySdes()
{
  SdesItem *sitem, *tmp;
  for (sitem = mysdes; sitem; ) {
    tmp = sitem;
    sitem = sitem->si_next;
    free(tmp);
  }
  mysdes = NULL;
}

void Session::dump()
{
  fprintf(stderr, "group/port/ttl=%x/%x/%x\n", group, rtp_port, ttl);

  SdesItem *sitem;
  int i;
  for (i=0, sitem = mysdes; sitem ; sitem = sitem->si_next, i++) {
    if (sitem->si_type > RTCP_SDES_END && sitem->si_type <= RTCP_SDES_SOURCE && sitem->si_len > 0 && sitem->si_len < 128 && sitem->si_str)
      fprintf(stderr, "  sdes[%d]=%s\n", i, sitem->si_str);
  }
  fflush(stderr);
}

void Session::dumpAll()
{
  for (Session *pse = sessionList; pse ; pse = pse->next) {
    pse->dump();
  }
}
