#include "global.h"

#include "net.h"


/** Open a datagram socket */
int socketDatagram()
{
  int sock = -1;

  if ((sock = socket(PF_INET, SOCK_DGRAM, 0)) < 0)
    error("socket: %s", strerror(errno));
  return sock;
}

/** Open a stream socket */
int socketStream()
{
  int sock = -1;

  if ((sock = socket(PF_INET, SOCK_STREAM, 0)) < 0)
    error("socket: %s", strerror(errno));
  return sock;
}

/**
 * Do REUSEADDR on the socket
 * return sock if OK, else -1
 */
int setReuseAddr(const int sock)
{
  const int one = 1;
  
  if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, 
		 (char *) &one, sizeof(one)) < 0) {
    error("SO_REUSEADDR: %s (%d)", strerror(errno), errno);
    return -1;
  }
  return sock;
}

int setTcpNoDelay(const int sock)
{
  const int one = 1;
  
  if (setsockopt(sock, SOL_SOCKET, TCP_NODELAY, 
		 (char *) &one, sizeof(one)) < 0) {
    error("TCP_NODELAY: %s (%d) sock=%d", strerror(errno), errno,sock);
    return -1;
  }
  return sock;
}

uint16_t getSrcPort(const int sock)
{
  struct sockaddr_in sa;
  socklen_t slen = sizeof(struct sockaddr_in);

  sa.sin_family = AF_INET;
  sa.sin_port = 0;
  sa.sin_addr.s_addr = htonl(INADDR_ANY);
  if (getsockname(sock, (struct sockaddr *) &sa, (socklen_t *) &slen) < 0) {
    error("getSrcPort: %s (%d)", strerror(errno), errno);
    return 0;
  }
  return ntohs(sa.sin_port);
}

/**
 * Control blocking(1)/non blocking(0)
 * return sock if OK, else -1
 */
int handleBlocking(const int sock, const bool block)
{
#if HAVE_FCNTL
  int flags;

  if ((flags = fcntl(sock, F_GETFL)) < 0) {
    error("F_GETFL: %s (%d)", strerror(errno), errno);
    return -1;
  }
  if (block)
    flags &= ~O_NONBLOCK;
  else
    flags |= O_NONBLOCK;
  if (fcntl(sock, F_SETFL, flags) < 0) {
    error("F_SETFL: %s (%d)", strerror(errno), errno);
    return -1;
  }
#endif // HAVE_FCNTL
  return sock;
}

int setBlocking(const int sock)
{
  return handleBlocking(sock, true);
}

int setNoBlocking(const int sock)
{
  return handleBlocking(sock, false);
}

/**
 * Set the ttl
 * return sock if OK, else -1
 */
int setScope(const int sock, const uint8_t _ttl)
{
#if defined(__WIN32__) || defined(_WIN32)
#define TTL_TYPE int
#else
#define TTL_TYPE uint8_t
#endif
  if (_ttl) {
    TTL_TYPE ttl = (TTL_TYPE)_ttl;
    if (setsockopt(sock, IPPROTO_IP, IP_MULTICAST_TTL, (const char*)&ttl, sizeof(ttl)) < 0) {
      error("IP_MULTICAST_TTL: %s (%d)", strerror(errno), errno);    
      return -1;
    }
  }
  return sock;
}

/**
 * Set loopback: active (1) either inactive (0)
 * return sock if OK, else -1
 */
int handleLoopback(const int sock, const uint8_t loop)
{
#ifdef IP_MULTICAST_LOOP // Windoze doesn't handle IP_MULTICAST_LOOP
  if (setsockopt(sock, IPPROTO_IP, IP_MULTICAST_LOOP, 
		 &loop, sizeof(loop)) < 0) {
#if IPMC_ENABLED
    error("IP_MULTICAST_LOOP: %s (%d)", strerror(errno), errno);
#endif
    return -1;
  }
#endif
  return sock;
}

int setLoopback(const int sock)
{
  return handleLoopback(sock, 1);
}

int setNoLoopback(const int sock)
{
  return handleLoopback(sock, 0);
}

int addMembership(const int sock, const void *pmreq)
{
  if (setsockopt(sock, IPPROTO_IP, IP_ADD_MEMBERSHIP,
                 pmreq, sizeof(struct ip_mreq)) < 0) {
#if IPMC_ENABLED
    error("IP_ADD_MEMBERSHIP: %s (%d)", strerror(errno), errno);
#endif
    return -1;
  }
  return 0;
}

int dropMembership(const int sock, const void *pmreq)
{
  if (setsockopt(sock, IPPROTO_IP, IP_DROP_MEMBERSHIP,
                 pmreq, sizeof(struct ip_mreq)) < 0) {
#if IPMC_ENABLED
    error("IP_DROP_MEMBERSHIP: %s (%d)", strerror(errno), errno);
#endif
    return -1;
  }
  return 0;
}

/** Set a Multicast socket */
void setSendSocket(const int sock, const uint8_t ttl)
{
#ifdef PROXY_MULTICAST
  if (mup.active) {
    setNoBlocking(sock);
    return;
  }
#endif

  setScope(sock, ttl);
#if NEEDLOOPBACK
  setLoopback(sock);    // loopback
#else
  setNoLoopback(sock);  // no loopback
#endif
  setNoBlocking(sock);
}

/**
 * Create an UDP socket
 * Prevue pour emettre (uni et mcast) et recevoir (unicast donc)
 * Do setScope (ttl) and setLoopback (off)
 * return sock if OK, else -1
 */
int createSendSocket(const uint8_t ttl)
{
  int sock;

  if ((sock = socketDatagram()) == 0)
    setSendSocket(sock, ttl);
  return sock;
}

/**
 * Create an Unicast socket
 * return fd, -1 if problem
 */
int createUcastSocket(const uint32_t uni_addr, const uint16_t port)
{
  int sock;
  struct sockaddr_in sa;

  if ((sock = socketDatagram()) < 0)
    return -1;
  memset(&sa, 0, sizeof(struct sockaddr_in));
  sa.sin_family = AF_INET; 
  sa.sin_port = htons(port);
  sa.sin_addr.s_addr = htonl(uni_addr);

  setReuseAddr(sock);
  if (bind(sock, (struct sockaddr *) &sa, sizeof(struct sockaddr_in)) < 0) {
    error("receive unicast bind: %s", strerror(errno));
  }
  return sock;
}

bool isMulticastAddress(const uint32_t address)
{
  // Note: We return False for addresses in the range 224.0.0.0
  // through 224.0.0.255, because these are non-routable
  unsigned addressInHostOrder = ntohl(address);
  return addressInHostOrder > 0xE00000FF && addressInHostOrder <= 0xEFFFFFFF;
}
