#include "sysdep.h"

#define VRE_VERSION		5
#define VACS_PORT		4150 + VRE_VERSION
#define VACS_FILE		"worlds.vre"
#define VRENG_MADDR_BASE	"224.255.0.0"
#define VRENG_MPORT_BASE	52656
#define VRENG_TTL		127
#define VRENG_UNIVERSE		"vreng"
#define VRENG_MPORT_INCR	10
#define WORLD_LEN		16
#define URL_LEN			256
#define UNIVERSE_LEN		32
#define CHANNEL_LEN		32
#define DEF_VACS_LOG		"/var/log/vacs.log"

// internal structure
struct _vacs {
  char url[URL_LEN];
  char world[WORLD_LEN];
  char channel[CHANNEL_LEN];
  char universe[UNIVERSE_LEN];
  uint32_t maddr;
  uint16_t mport;
  uint8_t ttl;
  struct _vacs *next;
};


bool debugvacs = 0;
char *log_file = NULL;
FILE *flog = NULL;
uint32_t next_maddr;
uint16_t next_mport;
struct _vacs *vacsList = NULL;


const char *date()
{
  time_t clock;
  struct tm *tm;
  static char datestr[20];

  clock = time(0);
  tm = localtime(&clock);
  sprintf(datestr, "%02d/%02d/%02d-%02d:%02d:%02d", tm->tm_mday, tm->tm_mon + 1, tm->tm_year % 100, tm->tm_hour, tm->tm_min, tm->tm_sec);
  return datestr;
}

int getWacsLine(char *line, FILE *fpvacs)
{
  do {
    if (fgets(line, BUFSIZ, fpvacs) == NULL)
      return -1;
  } while (line[0] == '#');
  line[strlen(line) -1] = 0;
  return strlen(line);
}

void getUniverse(const char *url, char *name)
{
  char *u = strdup(url);
  char *p = strchr(u, '/');
  p++;
  p = strchr(p, '/');
  p++;
  char *p2 = strchr(p, '/');
  *p2 = 0;
  strcpy(name, p);      
  free(u);
}

void getWorld(const char *url, char *name)
{
  char *u = strdup(url);
  char *p = strrchr(u, '.');
  *p = 0;
  p = strrchr(u, '/');
  p++;
  strcpy(name, p);
  free(u);
}

uint32_t getAddr(const char *channel)
{
  char *c = strdup(channel);
  char *p = strtok(c, "/");
#if HAVE_INET_PTON
  uint32_t addr;
  inet_pton(AF_INET, p, &addr);
#else
  uint32_t addr = inet_addr(p);
#endif
  free(c);
  return addr;
}

uint16_t getPort(const char *channel)
{
  char *c = strdup(channel);
  char *p = strtok(c, "/");
  p = strtok(NULL, "/");
  uint16_t port = atoi(p);
  free(c);
  return port;
}

uint8_t getTtl(const char *channel)
{
  uint8_t ttl = VRENG_TTL;

  char *c = strdup(channel);
  char *p = strtok(c, "/");
  p = strtok(NULL, "/");
  p = strtok(NULL, "/");
  if (p) {
    ttl = atoi(p);
    if (ttl == 0)
      ttl = VRENG_TTL;
  }
  free(c);
  return ttl;
}

/** Saves all the cache */
void saveCache()
{
  FILE *fpvacs;

  if ((fpvacs = fopen(VACS_FILE, "w")) == NULL) {
    perror("fopen write");
    return;
  }
  for (struct _vacs *w = vacsList->next; w ; w = w->next)
    fprintf(fpvacs, "%s\t%s\n", w->url, w->channel);
  fclose(fpvacs);
}

void releaseCache(const char *world)
{
}

/** Resolves a world in its universe, returns channel if found */
bool resolveCache(const char *url, char *chanstr)
{
  fprintf(flog, "resolveCache: url=%s\n", url); fflush(flog);
  char world[WORLD_LEN], universe[UNIVERSE_LEN];

  getWorld(url, world);
  getUniverse(url, universe);

  for (struct _vacs *w = vacsList; w ; w = w->next) {
    if ((! strcmp(world, w->world)) && (! strcmp(universe, w->universe))) {
      char maddrstr[CHANNEL_LEN];
#if HAVE_INET_NTOP
      char dst[46];
      strcpy(maddrstr, inet_ntop(AF_INET, &w->maddr, dst, sizeof(dst)));
#else
      strcpy(maddrstr, inet_ntoa(w->maddr));
#endif
      sprintf(chanstr, "%s/%d/%d", maddrstr, w->mport, VRENG_TTL);
      fprintf(flog, "resolveCache: chanstr=%s\n", chanstr); fflush(flog);
      return true;
    }
  }
  return false;
}

/** Adds vacs structure, appends to cache, returns new channel */
void addCache(const char *url, char *channel)
{
  fprintf(flog, "addCache: url=%s\n", url); fflush(flog);
  struct _vacs *vacs = (struct _vacs *) malloc(sizeof(_vacs));

  strcpy(vacs->url, url);		// url
  getWorld(vacs->url, vacs->world);	// world
  getUniverse(vacs->url, vacs->universe);	// universe

  // maddr mport
  vacs->maddr = next_maddr;
  vacs->mport = next_mport;

  // channel
  char maddrstr[CHANNEL_LEN];
  memset(maddrstr, 0, sizeof(maddrstr));
#if HAVE_INET_NTOP
  char dst[46];
  strcpy(maddrstr, inet_ntop(AF_INET, &vacs->maddr, dst, sizeof(dst)));
#else
  strcpy(maddrstr, inet_ntoa(vacs->maddr));
#endif
  memset(vacs->channel, 0, sizeof(vacs->channel));
  sprintf(vacs->channel, "%s/%d/%d", maddrstr, vacs->mport, VRENG_TTL); //vreng
  strcpy(channel, vacs->channel);
  sprintf(vacs->channel, "%s/%d", maddrstr, vacs->mport);	// record

  vacs->next = NULL;

  // add this entry at the end of the vacsList
  struct _vacs *wlast = NULL;
  for (struct _vacs *w = vacsList; w ; w = w->next) {
    wlast = w;
  }
  if (wlast) {
    wlast->next = vacs;
  }
  else
    fprintf(flog, "addCache: wlast NULL\n"); fflush(flog);

  // append this record to the file
  FILE *fpvacs;

  if ((fpvacs = fopen(VACS_FILE, "a")) == NULL) {
    perror("fopen append");
    return;
  }
  fprintf(fpvacs, "%s\t%s\n", vacs->url, vacs->channel);
  fclose(fpvacs);
  fprintf(flog, "append: %s\t%s\n", vacs->url, vacs->channel); fflush(flog);

  // next values
  next_maddr = vacs->maddr + htonl(1);
  next_mport = vacs->mport + VRENG_MPORT_INCR;
}

/** Loads the cache file */
void loadCache()
{
  FILE *fpvacs;
  char line[BUFSIZ];

  if ((fpvacs = fopen(VACS_FILE, "r")) == NULL) {
    perror("fopen read");
    exit(1);
  }

  // first record
  struct _vacs *vacs = (struct _vacs *) malloc(sizeof(_vacs));
  vacsList = vacs;
  strcpy(vacs->world, "Manager");
  *vacs->url = 0;
  sprintf(vacs->channel, "%s/%d/%d", VRENG_MADDR_BASE, VRENG_MPORT_BASE, VRENG_TTL);
#if HAVE_INET_PTON
  inet_pton(AF_INET, VRENG_MADDR_BASE, &vacs->maddr);
#else
  vacs->maddr = inet_addr(VRENG_MADDR_BASE);
#endif
  vacs->mport = VRENG_MPORT_BASE;
  vacs->ttl = VRENG_TTL;
  strcpy(vacs->universe, VRENG_UNIVERSE);
  vacs->next = NULL;
  struct _vacs *prev = vacs;

  next_maddr = vacs->maddr + htonl(1);
  next_mport = vacs->mport + VRENG_MPORT_INCR;

  // other records
  while (getWacsLine(line, fpvacs) != -1) {
    struct _vacs *vacs = (struct _vacs *) malloc(sizeof(_vacs));

    // url
    char *p = strtok(line , " \t");
    strcpy(vacs->url, p);
    // world
    getWorld(vacs->url, vacs->world);
    // universe
    getUniverse(vacs->url, vacs->universe);

    // channel
    p = strtok(NULL, " \t");
    strcpy(vacs->channel, p);
    // maddr
    vacs->maddr = getAddr(p);
    // mport
    vacs->mport = getPort(p);
    vacs->ttl = getTtl(p);
    if (vacs->ttl == 0)
      vacs->ttl = VRENG_TTL;

    // next values
    next_maddr = vacs->maddr + htonl(1);
    next_mport = vacs->mport + VRENG_MPORT_INCR;

    prev->next = vacs;
    vacs->next = NULL;
    prev = vacs;
    //fprintf(flog, "w=%s u=%s c=%s a=%x p=%d t=%d n=%x/%d\n", vacs->world, vacs->universe, vacs->channel, vacs->maddr, vacs->mport, vacs->ttl, next_maddr, next_mport); fflush(flog);
  }
  fclose(fpvacs);
}

void sendCache(const int sdvre)
{
  char bufvacs[URL_LEN + CHANNEL_LEN + 2];
  fprintf(flog, "sendCache:\n"); fflush(flog);

  for (struct _vacs *w = vacsList->next; w ; w = w->next) {
    memset(bufvacs, 0, sizeof(bufvacs));
    sprintf(bufvacs, "%s %s/%d", w->url, w->channel, VRENG_TTL);
    send(sdvre, bufvacs, sizeof(bufvacs), 0);
    //fprintf(flog, "%s %s/%d [%x]\n", w->url, w->channel, VRENG_TTL, w->maddr); fflush(flog);
  }
  close(sdvre);
}

/** Handles requests */
void parseRequest(int sdvre, const struct sockaddr_in *sa)
{
  char request[BUFSIZ], channel[CHANNEL_LEN];
  int len;

  if ((len = recv(sdvre, request, sizeof(request), 0)) < 0) {
    perror("vacs: recv");
    return;
  }
  request[len] = '\0';
  fprintf(flog, "request: %s\n", request); fflush(flog);

  if (! strncmp(request, "list", 4)) {		// list
    sendCache(sdvre);
  }
  else if (! strncmp(request, "resolve", 7)) {	// resolve
    char *url = strtok(request, " ");
    url = strtok(NULL, " ");
    if (resolveCache(url, channel) == 0) {
      addCache(url, channel);
    }
    channel[strlen(channel)] = 0;
    send(sdvre, channel, strlen(channel) + 1, 0);
  }
  else if (! strncmp(request, "release", 7)) {	// release
    char *world = strtok(request, " ");
    world = strtok(NULL, " ");
    releaseCache(world);
  }
  else {					// unknown
    if (len != 0) {
      fprintf(flog, "unknown request: %s len=%d\n", request, len); fflush(flog);
    }
  }
  return;
}

/** Runs in the background */
void background()
{
  if (fork())
    exit(0);
}

/** Inits the server */
void initWacsServer()
{
  if (! debugvacs)
    background();

  loadCache();

  int sdvacs, sdvre;
  struct sockaddr_in sa;
  socklen_t slen = sizeof(struct sockaddr_in);

  if ((sdvacs = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
    perror("socket");
    exit(1);
  }
  memset(&sa, 0, sizeof(sa));
  sa.sin_family = AF_INET;
  sa.sin_port = htons(VACS_PORT);
  sa.sin_addr.s_addr = htonl(INADDR_ANY);
  int one = 1;
  if (setsockopt(sdvacs, SOL_SOCKET, SO_REUSEADDR, (char *) &one, sizeof(one))<0)
    perror("reuseaddr");
  if (bind(sdvacs, (struct sockaddr *) &sa, sizeof(sa)) < 0) {
    perror("binding");
    exit(1);
  }

  listen(sdvacs, 5);

  /* Server Main Loop */
  while (1) {
    if ((sdvre = accept(sdvacs, (struct sockaddr *) &sa, &slen)) < 0) {
      sleep(1);	// wait hoping that will be better next time...
      continue;
    } 
#if HAVE_INET_NTOP
    char dst[46];
    const char *pdst = inet_ntop(AF_INET, &sa.sin_addr, dst, sizeof(dst));
#else
    const char *pdst = inet_ntoa(sa.sin_addr);
#endif
    struct hostent *hp;
    if ((hp = gethostbyaddr((const char *) &sa.sin_addr,
                            sizeof(struct in_addr), AF_INET)) == NULL)
      perror("gethostbyaddr");
    fprintf(flog, "%s %s (%s) connected\n",
                  date(), (hp) ? hp->h_name : "<unresolved>", pdst);
    fflush(flog);

    parseRequest(sdvre, &sa);
    close(sdvre);
  } 
}

void usage(char *argv[])
{
  printf("usage: %s [-?] [-h] [-D] [-v] [-d vacsdir] [-l logfile]\n", argv[0]);
  exit(0);
} 

static char *vacs_dir = NULL;

/* Parse command line args */
void parseArgs(int argc, char *argv[])
{
  int c;

  while ((c = getopt(argc, argv, "?Dhvd:l:")) != -1) {
    switch (c) {
      case '?':
      case 'h': usage(argv); break;
      case 'D': debugvacs = true; break;
      case 'd': vacs_dir = optarg; break;
      case 'l': log_file = optarg; break;
      case 'v': printf("%s\n", PACKAGE_VERSION); exit(0);
    }
  }
  if (optind < argc) {
    usage(argv);
  }
}

int main(int argc, char *argv[])
{
  parseArgs(argc, argv);

  if (log_file) {
    if ((flog = fopen(log_file, "a")) == NULL) {
      fprintf(stderr, "can't open log_file %s\n", log_file);
    }
  }
  else
    flog = stderr;

  if (vacs_dir) {
    if (chdir(vacs_dir) < 0) {
      fprintf(stderr, "vacs: %s is not a directory\n", vacs_dir);
      exit(1);
    }
  }

  initWacsServer();
  return 0;
} 
