// ------------------------------------------------------------ //
// Author   : This file has been written by Yann Renard         //
// Copyright: This file is totaly free and you may distribute   //
//            it to anyone you want, without modifying this     //
//            header. If you use it in a commercial project (?) //
//            or in bigger project (!), I would be glad to know //
//            about it :) Please mail me...                     //
//            be glad to know about it, please mail me          //
//                myself_yr@hotmail.com                         //
// ------------------------------------------------------------ //

#include "global.h"
#include "boneFactory.h"

void Factory::buildSphere(Mesh3D *result,
				float radius,
				int cpt1,
				int cpt2,
				float centerx,
				float centery,
				float centerz)
{
  int i,j;
  if (cpt1 <= 0) return;
  if (cpt2 <= 0) return;

  trace(DBG_MAN, "buildSphere: Creating");
  trace(DBG_MAN, "           Radius        : %f" , radius);
  trace(DBG_MAN, "           Nb edges 1    : %i" , cpt1);
  trace(DBG_MAN, "           Nb edges 2    : %i" , cpt2);

  for (i=0; i<cpt1; i++)
    for (j=0; j<cpt2; j++)
      result->addVertex (
       centerx + radius * sin(i * M_PI / (cpt1-1)) * cos(j * 2. * M_PI / cpt2),
       centery + radius * cos(i * M_PI / (cpt1-1)),
       centerz + radius * sin(i * M_PI / (cpt1-1)) * sin(j * 2. * M_PI / cpt2)
      );

  TRIANGLE *triangle;
  int cpt = 0;
  for (i=0; i<cpt1; i++) {
    for (j=0; j<cpt2; j++) {
      int i1 = (i) * cpt2 + (j);
      int i2 = (i+1)%cpt1 * cpt2 + (j);
      int i3 = (i+1)%cpt1 * cpt2 + (j+1)%cpt2;
      int i4 = (i) * cpt2 + (j+1)%cpt2;

      result->addTriangle(i1,i3,i2);
      triangle = result->triangleList.getElementAt(cpt++);
      triangle->setColor(1,1,1);

      result->addTriangle(i1,i4,i3);
      triangle = result->triangleList.getElementAt(cpt++);
      triangle->setColor(1,1,1);
    }
  }

  result->rebuildNormals();
  result->setName("Factory::buildSphere");
}

void Factory::buildTorus(Mesh3D *result,
			       float bigRadius, float littleRadius,
			       int cpt1, int cpt2,
			       float centerx, float centery, float centerz)
{
  int i,j;
  if (cpt1 <= 0) return;
  if (cpt2 <= 0) return;

  trace(DBG_MAN, "buildTorus:");
  trace(DBG_MAN, "           Big radius    : %f" , bigRadius);
  trace(DBG_MAN, "           Little radius : %f" , littleRadius);
  trace(DBG_MAN, "           Nb edges 1    : %i" , cpt1);
  trace(DBG_MAN, "           Nb edges 2    : %i" , cpt2);

  TRIANGLE *triangle;
  int cpt = 0;
  for (i=0; i<cpt1; i++)
    for (j=0; j<cpt2; j++)
      result->addVertex(
        centerx + (bigRadius + littleRadius * cos(j * 2. * M_PI / cpt2 + M_PI)) * sin(i * 2. * M_PI / cpt1),
        centery + (bigRadius + littleRadius * cos(j * 2. * M_PI / cpt2 + M_PI)) * cos(i * 2. * M_PI / cpt1),
        centerz +              littleRadius * sin(j * 2. * M_PI / cpt2 + M_PI)
	);

  for (i=0; i<cpt1; i++) {
    for (j=0; j<cpt2; j++) {
      int i1 = ((i)) * cpt2 + ((j));
      int i2 = ((i+1)%cpt1) * cpt2 + ((j));
      int i3 = ((i+1)%cpt1) * cpt2 + ((j+1)%cpt2);
      int i4 = ((i)) * cpt2 + ((j+1)%cpt2);
      result->addTriangle(i1,i3,i2);
      triangle = result->triangleList.getElementAt(cpt++);
      triangle->setColor(1,1,1);

      result->addTriangle(i1,i4,i3);
      triangle = result->triangleList.getElementAt(cpt++);
      triangle->setColor (1,1,1);
    }
  }
  result->rebuildNormals();
  result->setName("Factory::buildTorus");
}

void Factory::buildCylinder(Mesh3D *result,
				  float radius, float height,
				  int cpt1, int cpt2,
				  float centerx, float centery, float centerz)
{
  if (cpt1 <= 0) return;
  if (cpt2 <= 0) return;

  trace(DBG_MAN, "buildCylinder:");
  trace(DBG_MAN, "           Radius        : %f" , radius);
  trace(DBG_MAN, "           Nb edges 1    : %i" , cpt1);
  trace(DBG_MAN, "           Nb edges 2    : %i" , cpt2);

  for (int i=0; i<cpt1; i++) {
    for (int j=0; j<cpt2; j++) {
      result->addVertex(
        centerx +   radius                      * sin(j * 2. * M_PI / cpt2),
        centery + (height * i) / (cpt1 - 1),
        centerz +   radius                      * cos(j * 2. * M_PI / cpt2)
	// centerx + (radius * i) / (cpt1 - 1) * sin(j * 2. * M_PI / cpt2),
        // centery + (height * i) / (cpt1 - 1),
        // centerz + (radius * i) / (cpt1 - 1) * cos(j * 2. * M_PI / cpt2)
	);
    }
  }

  TRIANGLE *triangle;
  int cpt = 0;
  for (int i=0; i<cpt1-1; i++) {
    for (int j=0; j<cpt2; j++) {
      int i1 = ((i)) * cpt2 + ((j));
      int i2 = ((i+1)%cpt1) * cpt2 + ((j));
      int i3 = ((i+1)%cpt1) * cpt2 + ((j+1)%cpt2);
      int i4 = ((i)) * cpt2 + ((j+1)%cpt2);
      result->addTriangle(i1,i3,i2);
      triangle = result->triangleList.getElementAt(cpt++);
      triangle->setColor(1,1,1);

      result->addTriangle(i1,i4,i3);
      triangle = result->triangleList.getElementAt(cpt++);
      triangle->setColor(1,1,1);
    }
  }
  result->rebuildNormals();
  result->setName("Factory::buildCylinder");
}

void Factory::readVRMLfile(Mesh3D *result, char *filename, float size, float centerx, float centery, float centerz, int colorMask)
{
  trace(DBG_MAN, "Creating TRI mesh with vrml file");
  trace(DBG_MAN, "           Filename      : [%s]", filename);

  char endLine = '\0';
  char skippedText[1024];
  float x,y,z;
  int index1, index2, index3, indexEnd;
  int vertices=0, faces=0;

  // First look if the file exists
  FILE *fp = fopen(filename, "rt");
  if (fp == NULL) return;
  fclose(fp);

  // Now that we know it exists, we'll look after the vertices part
  fp = fopen(filename, "rt");
  strcpy(skippedText, "");
  while ((strcmp(skippedText, "point") != 0) && (!feof(fp)))
    fscanf(fp, "%s", skippedText);
  if (!feof(fp)) {
    fscanf(fp, "%s\n", skippedText);
    endLine = '\0';
    while (endLine != ']') {
      fscanf(fp, "%f %f %f%c", &x, &y, &z, &endLine);
      result->addVertex(centerx + x * size, centery + y * size, centerz + z * size);
      vertices ++;
    }
    trace(DBG_MAN, "           Vertices added: %i", vertices);
  }
  fclose(fp);

  // We'll look after the faces part
  fp = fopen(filename, "rt");
  strcpy(skippedText, "");
  while ((strcmp(skippedText, "coordIndex") != 0) && (!feof(fp)))
    fscanf(fp, "%s", skippedText);
  if (!feof(fp)) {
    fscanf(fp, "%s", skippedText);
    endLine = '\0';
    while (endLine != ']') {
      fscanf(fp, "%d, %d, %d, %d%c", &index1, &index2, &index3, &indexEnd, &endLine);
      result->addTriangle(index1, index2, index3);
      faces ++;
    }
    trace(DBG_MAN, "           Faces added   : %i", faces);
  }
  fclose(fp);

  // We'll now go on color part
  fp = fopen(filename, "rt");
  strcpy(skippedText, "");
  while ((strcmp(skippedText, "color") != 0) && (!feof(fp)))
    fscanf(fp, "%s", skippedText);
  if (!feof(fp))
    fscanf(fp, "%s", skippedText);
  if (!feof(fp) && strcmp(skippedText, "NULL")) {
    TRIANGLE * triangle;
    float r,g,b;
    int cpt=0;

    printf("[%s]\n", skippedText);
    fscanf(fp, "%s", skippedText);
    fscanf(fp, "%s", skippedText);
    fscanf(fp, "%s", skippedText);

    endLine = '\0';
    while (endLine != ']') {
      fscanf(fp, "%f %f %f%c", &r, &g, &b, &endLine);
      triangle = result->triangleList.getElementAt(cpt++);

      if ((r==g) && (g==b) && (colorMask != 0)) {
        r = (triangle->vertex1->initialPosition.x + 1.) / 2.;
        g = (triangle->vertex1->initialPosition.y + 1.) / 2.;
        b = (triangle->vertex1->initialPosition.z + 1.) / 2.;
      }
      triangle->setColor(r,g,b);
    }
  }
  fclose(fp);

  // And finaly come on texture mapping coordinates
  fp = fopen(filename, "rt");
  strcpy(skippedText, "");
  while ((strcmp(skippedText, "texCoord") != 0) && (!feof(fp)))
    fscanf(fp, "%s", skippedText);
  if (!feof(fp)) {
    Vertex * vertex;
    float u,v;
    int cpt=0;

    fscanf(fp, "%s", skippedText);
    fscanf(fp, "%s", skippedText);
    fscanf(fp, "%s", skippedText);
    fscanf(fp, "%s", skippedText);

    endLine = '\0';
    while (endLine != ']') {
      fscanf(fp, "%f %f%c", &u, &v, &endLine);
      vertex = result->vertexList.getElementAt(cpt++);
      vertex->u = u;
      vertex->v = v;
    }
  }
  fclose(fp);
  result->rebuildNormals();
  result->setName(filename);
}

//-- VRENG 3D intern format
void Factory::readVRE3Dfile(Mesh3D *result, BoneVertex *skeletonRoot, char *filename, float scale)
{
  FILE *fp = fopen(filename, "rb");
  if (fp == NULL) return;

  int i, index1, index2, index3;
  float x,y,z;

  trace(DBG_MAN, "readVRE3Dfile: reading mesh and skeleton from VRE3D file %s", filename);

  // Reading name
  char name[512];
  readString(fp, name);
  result->setName(name);

  // Reading vertices
  int vertices = readInt(fp);
  for (i=0; i<vertices; i++) {
    x = readFloat(fp) * scale;
    y = readFloat(fp) * scale;
    z = readFloat(fp) * scale;
    result->addVertex(x,y,z);
  }
  trace(DBG_MAN, "           Vertices added: %i", vertices);

  int faces = readInt(fp);
  for (i=0; i<faces; i++) {
    // Reading indices
    index1 = readInt(fp);
    index2 = readInt(fp);
    index3 = readInt(fp);
    result->addTriangle(index1, index2, index3);
    TRIANGLE *face = result->triangleList.getElementAt(i);
    // Reading color
    float r = readFloat(fp);
    float g = readFloat(fp);
    float b = readFloat(fp);
    float a = readFloat(fp);
    face->setColor(r,g,b,a);
    // Reading texture coordinantes
    face->u1 = readFloat(fp);
    face->v1 = readFloat(fp);
    face->u2 = readFloat(fp);
    face->v2 = readFloat(fp);
    face->u3 = readFloat(fp);
    face->v3 = readFloat(fp);
  }
  trace(DBG_MAN, "           Faces added   : %i", faces);

  // Reading skeleton
  skeletonRoot->readFromFile(fp, scale);
  trace(DBG_MAN, "           Skeleton added!");

  fclose(fp);
  result->rebuildNormals();
}

void Factory::writeVRE3Dfile(Mesh3D *outMesh, BoneVertex *skeletonRoot, char *filename)
{
  FILE *fp = fopen(filename, "wb");
  if (fp == NULL) return;

  if (! outMesh->vertexListCompiled) outMesh->compileVertexList();
  if (! outMesh->triangleListCompiled) outMesh->compileTriangleList();

  int i, index1, index2, index3;

  trace(DBG_FORCE, "writeVRE3Dfile: writing mesh and skeleton to VRE3D file");

  // Writing name
  writeString(fp, outMesh->getName());

  // Writing vertices coordinates
  writeInt(fp, outMesh->vertices);
  for (i=0; i<outMesh->vertices; i++) {
    writeFloat(fp, outMesh->vertex[i]->initialPosition.x);
    writeFloat(fp, outMesh->vertex[i]->initialPosition.y);
    writeFloat(fp, outMesh->vertex[i]->initialPosition.z);
  }
  trace(DBG_FORCE, "           Vertices added: %i", outMesh->vertices);

  // Writing triangles
  writeInt(fp, outMesh->triangles);
  for (i=0; i<outMesh->triangles; i++) {
    index1 = 0;
    index2 = 0;
    index3 = 0;
    while (outMesh->vertex[index1] != outMesh->triangle[i]->vertex1) index1++;
    while (outMesh->vertex[index2] != outMesh->triangle[i]->vertex2) index2++;
    while (outMesh->vertex[index3] != outMesh->triangle[i]->vertex3) index3++;
    // Writing indices
    writeInt(fp, index1);
    writeInt(fp, index2);
    writeInt(fp, index3);
    // Writing triangle color
    writeFloat(fp, outMesh->triangle[i]->colorRed);
    writeFloat(fp, outMesh->triangle[i]->colorGreen);
    writeFloat(fp, outMesh->triangle[i]->colorBlue);
    writeFloat(fp, outMesh->triangle[i]->colorAlpha);
    // Writing triangle texture coordinates
    writeFloat(fp, outMesh->triangle[i]->u1);
    writeFloat(fp, outMesh->triangle[i]->v1);
    writeFloat(fp, outMesh->triangle[i]->u2);
    writeFloat(fp, outMesh->triangle[i]->v2);
    writeFloat(fp, outMesh->triangle[i]->u3);
    writeFloat(fp, outMesh->triangle[i]->v3);
  }
  trace(DBG_FORCE, "           Faces added   : %i", outMesh->triangles);

  // Writing skeleton
  skeletonRoot->writeToFile(fp);
  trace(DBG_FORCE, "           Skeleton added!");

  fclose(fp);
}

void Factory::readTGA(char *filename, int **buffer, int *width, int *height)
{
  trace(DBG_MAN, "readTGA: reading TGA file: [%s]", filename);

  FILE *fp = fopen(filename, "rb");
  if (fp == NULL) {
    trace(DBG_FORCE, "readTGA: open TGA: %s", filename);
    return;
  }

  int sizex, sizey, data;
  data = fgetc(fp); if (data != 0) return; // ID Length = 0
  data = fgetc(fp); if (data != 0) return; // Colormap type = No colormap included
  data = fgetc(fp); if (data != 2) return; // Image type = uncompressed true color
  data = fgetc(fp); if (data != 0) return; // Colormap spec: 1st entry index
  data = fgetc(fp); if (data != 0) return;
  data = fgetc(fp); if (data != 0) return; // Colormap spec: colormap length
  data = fgetc(fp); if (data != 0) return;
  data = fgetc(fp); if (data != 0) return; // Colormap spec: color map entry size
  data = fgetc(fp); if (data != 0) return; // X origin
  data = fgetc(fp); if (data != 0) return;
  data = fgetc(fp); if (data != 0) return; // Y origin
  data = fgetc(fp); if (data != 0) return;
  sizex = fgetc(fp); sizex += fgetc(fp) << 8;
  sizey = fgetc(fp); sizey += fgetc(fp) << 8;
  data = fgetc(fp); if (data != 24) return; // Pixel depth(24 bits)
  data = fgetc(fp); if (data != 0) return; // Image descriptor

  *width  = sizex;
  *height = sizey;
  smartFree(*buffer);
  *buffer = (int *) malloc(sizex * sizey * sizeof(int));

  printf("           Width         : %i\n", sizex);
  printf("           Height        : %i\n", sizey);

  for (int y=0; y<sizey; y++) {
    for (int x=0; x<sizex; x++) {
      data  = fgetc(fp);
      data += fgetc(fp) << 8;
      data += fgetc(fp) << 16;
      (*buffer) [(sizey - y - 1) * sizex + x] = data << 8;
    }
  }
  printf("           Bits per pixel: 24\n");

  fclose(fp);
}
