// ------------------------------------------------------------ //
// Author   : This file has been written by Yann Renard         //
// Copyright: This file is totaly free and you may distribute   //
//            it to anyone you want, without modifying this     //
//            header. If you use it in a commercial project (?) //
//            or in bigger project (!), I would be glad to know //
//            about it :) Please mail me...                     //
//            be glad to know about it, please mail me          //
//                myself_yr@hotmail.com                         //
// ------------------------------------------------------------ //

#ifndef BONELIST_H
#define BONELIST_H

//======================================================
// La classe LIST_NODE represente un noeud de la liste
// chainee. Elle contient un pointeur sur l'element stocke
// dans ce noeud, un pointeur sur le noeud suivant et un
// pointeur sur le noeud precedent... Cette classe ne fait
// que gerer ces champs (lecture/ecriture)
//
template <class ELEMENT> class LIST_NODE {
 private:
       LIST_NODE * nextNode;  // Noeud precedent this dans la liste
       LIST_NODE * prevNode;  // Noeud suivant this dans la liste
       ELEMENT * element;     // Element de la liste contenu dans this

 public:

   //------------------------------------------------------------
   // Constructeur de noeud, initialise tous les pointeurs a NULL
   //
   LIST_NODE() {
     nextNode = NULL;
     prevNode = NULL;
     element  = NULL;
   }

   //------------------------------------------------------------
   // Constructeur de noeud par recopie, recopie tous les champs
   // du noeud passe en parametre
   //
   LIST_NODE(LIST_NODE & node) {
     nextNode = node.nextNode;
     prevNode = node.prevNode;
     element  = node.element;
   }

   //------------------------------------------------------------
   // Le destructeur ne fait rien
   //
   virtual ~LIST_NODE() {}

   //------------------------------------------------------------
   // Methodes de remplissage des champs d'un noeud
   //
   inline void setPrevious (LIST_NODE * zePrev) { prevNode = zePrev; }
   inline void setNext     (LIST_NODE * zeNext) { nextNode = zeNext; }
   inline void setElement (ELEMENT * zeElement) { element = zeElement; }
   inline void setValue (LIST_NODE * zePrev, LIST_NODE * zeNext, ELEMENT * zeElement)
   {
     setNext(zeNext);
     setPrevious(zePrev);
     setElement(zeElement);
   }

   //------------------------------------------------------------
   // Methodes de lecture des champs d'un noeud
   //
   inline LIST_NODE * getNext     (void) { return nextNode; }
   inline LIST_NODE * getPrevious (void) { return prevNode; }
   inline ELEMENT * getElement    (void) { return element;  }
 };
 

//============================================================
// Class LIST : la classe liste est la classe principale de
// gestion de la liste chainnee. Il stocke les noeuds
// contenant les pointeurs sur elements a la queu leu leu
// en commencant par 'listHead'
// Methodes : void  empty (void)
//            int   count (void)
//            void  addElement (ELEMENT * zeElement)
//            void  addList (LIST * zeList)
//            void  removeElement (int index)
//            void  removeElement (ELEMENT * ptr)
//            ELEMENT * getElementAt (int index)
//            ELEMENT ** getNiceTable (int * count)
//
template <class ELEMENT> class LIST {
 private:
       LIST_NODE < ELEMENT > * listHead; // Tete de la liste chainee
       LIST_NODE < ELEMENT > * listEnd;  // Queue de la liste chainee
       ELEMENT              ** element;  // Table contenant les pointeur sur element lors de la compilation de la liste en tableau
       int                     elements; // Nombre d'elements dans la table lors de la compilation de la liste en tableau
       int                     built;    // Masque disant si la liste est compilee ou non

   //------------------------------------------------------------
   // Recherche du premier noeud de la liste
   //
   inline LIST_NODE < ELEMENT > * getFirstNode()
   {
     return listHead;
   }

   //------------------------------------------------------------
   // Recherche du dernier noeud de la liste
   //
   inline LIST_NODE < ELEMENT > * getLastNode()
   {
     return listEnd;
   }

   //------------------------------------------------------------
   // Compilation de la liste chainee en tableau
   // Cette methode ne s'execute que si la liste a ete modifiee
   // dans la passe (addElement ou removeElement).
   //
   inline void rebuild()
   {
     if (built) return;

     LIST_NODE < ELEMENT > * currentNode = listHead;

     elements = count();
     smartFree(element);
     element = (ELEMENT **) malloc (elements * sizeof (ELEMENT *));
     for (int i=0; i<elements; i++, currentNode = currentNode->getNext())
       element[i] = currentNode->getElement();
        
     built = 1;
   }

   //-----------------------------------------------
   // Fonction recursive de vidage de la liste, il
   // s'agit de d'appeler le vidage du fils puis de
   // liberer le fils
   //
   inline void recursiveEmpty(LIST_NODE < ELEMENT > * currentNode)
   {
     if (currentNode != NULL) {
       recursiveEmpty(currentNode->getNext());
       delete currentNode;
     }
   }

 public:

   //------------------------------------------------------------
   // Constructeur par defaut. Il initialise la liste comme une
   // liste vide
   //
   LIST() {
     listHead = NULL;
     listEnd  = NULL;
     element  = NULL;
     elements = 0;
     built    = 0;
   }

   //------------------------------------------------------------
   // Le destructeur vide la liste de son contenu
   //
   virtual ~LIST()
   {
     empty();
   }

   //------------------------------------------------------------
   // La fonction de vidage de la liste enleve les noeuds un par
   // un et efface le tableau compile
   //
   inline void empty()
   {
     recursiveEmpty(listHead);
     listHead = NULL;
     listEnd  = NULL;

     smartFree(element);
     element = 0;
     built = 0;
   }

   //------------------------------------------------------------
   // Fonction de comptage du nombre d'elements de la liste
   //
   inline int count()
   {
     LIST_NODE < ELEMENT > * currentNode = listHead;
     int result = 0;
     while (currentNode != NULL) {
       currentNode = currentNode->getNext();
       result++;
     }
     return result;
   }

   //------------------------------------------------------------
   // Fonction d'acces a un element de la liste retourne NULL si
   // l'index est inferieur a 0 ou superieur au nombre
   // d'elements
   //
   inline ELEMENT * getElementAt(int index)
   {
     LIST_NODE < ELEMENT > * currentNode = listHead;
     ELEMENT * result = NULL;
     while ((currentNode != NULL) && (index > 0)) {
       currentNode = currentNode->getNext();
       index--;
     }
     if ((currentNode != NULL) && (index == 0))
       result = currentNode->getElement();
     return result;
   }

   //------------------------------------------------------------
   // Ajouter un element dans la liste, cet element est ajoute
   // en queue de liste
   //
   inline void addElement(ELEMENT * zeElement)
   {
     LIST_NODE < ELEMENT > * newNode = new LIST_NODE < ELEMENT > ();
     newNode->setElement(zeElement);

     if (listHead == NULL) {
       listHead = newNode;
       listEnd  = newNode;
     }
     else {
       listEnd->setNext(newNode);
       newNode->setPrevious(listEnd);
       listEnd = newNode;
     }
     built = 0;
   }

   //------------------------------------------------------------
   // Ajouter une liste dans cette liste
   //
   inline void addList(LIST * zeList)
   {
     LIST_NODE < ELEMENT > * newNode = zeList->getFirstNode ();

     if (listHead == NULL) {
       listHead = newNode;
       listEnd  = newNode;
     }
     else {
       listEnd->setNext(newNode);
       newNode->setPrevious(listEnd);
       listEnd = newNode;
     }
     built = 0;
   }

   //------------------------------------------------------------
   // Enlever un element dans la liste, tous les elements
   // suivants remontent d'un rang
   //
   inline void removeElement(int index)
   {
     if (listHead == NULL) return;

     LIST_NODE < ELEMENT > * prevNode;
     LIST_NODE < ELEMENT > * currentNode;
     LIST_NODE < ELEMENT > * nextNode;

     if (index == 0) { // Cas particulier de la tete
       prevNode = NULL;
       nextNode = listHead->getNext();
       delete listHead;
       listHead = nextNode;
     }
     else { // Autre cas
       currentNode = listHead;
       while ((index > 0) && (currentNode != NULL)) {
         currentNode = currentNode->getNext();
         index--;
       }
       if (currentNode != NULL) {
         nextNode = currentNode->getNext();
         prevNode = currentNode->getPrevious();
         delete currentNode;
         if (nextNode != NULL) nextNode->setPrevious(prevNode);
         if (prevNode != NULL) prevNode->setNext(nextNode);
       }
     }
     built = 0;
   }

   //------------------------------------------------------------
   // Enlever un element dans la liste, tous les elements
   // suivants remontent d'un rang
   //
   inline void removeElement(ELEMENT *ptr)
   {
     if (listHead == NULL) return;

     LIST_NODE < ELEMENT > * prevNode;
     LIST_NODE < ELEMENT > * currentNode;
     LIST_NODE < ELEMENT > * nextNode;

     if (ptr == listHead -> getElement ()) { // Cas particulier de la tete
       prevNode = NULL;
       nextNode = listHead->getNext();
       delete listHead;
       listHead = nextNode;
     }
     else { // Autre cas
       currentNode = listHead;
       while ((currentNode != NULL) && (currentNode -> getElement() != ptr))
         currentNode = currentNode->getNext();
       if (currentNode != NULL) {
         nextNode = currentNode->getNext();
         prevNode = currentNode->getPrevious();
         delete currentNode;
         if (nextNode != NULL) nextNode->setPrevious(prevNode);
         if (prevNode != NULL) prevNode->setNext(nextNode);
       }
     }
     built = 0;
   }

   //------------------------------------------------------------
   // Compilation de la liste en un tableau. Cette methode
   // retourne le nombre d'elements dans la variable count
   //
   inline ELEMENT ** getNiceTable(int *count)
   {
     if (!built) rebuild();
     if (count != NULL) *count = elements;
     return element;
   }
};

#endif // BONELIST_H
