#include "global.h"
#include "format.h"	// imgs
#include "images.h"


/**
 * Image constructor
 */
Image::Image(const uint16_t _width, const uint16_t _height, const uint8_t _format, const uint8_t _type)
{
  new_image++;
  xsize = _width;
  ysize = _height;
  format = _format;
  type = _type;
  if ((pixmap = new GLubyte[_width * _height * _format]) == NULL) {
    warning("can't new pixmap");
    return;
  }
}

Image::~Image()
{
  del_image++;
  delete[] pixmap;
  pixmap = NULL;
}


/* Resampling */

/**
 * bilinear interpolation with xf, yf normalized to 2^16
 */
static inline
int interpol(const int v00, const int v01, const int v10, const int v11, const int xf, const int yf)
{
  int v0 = v00 + (((v01-v00) * xf) >> NORM8_BITS);
  int v1 = v10 + (((v11-v10) * xf) >> NORM8_BITS);

  return (v0 + (((v1-v0) * yf) >> NORM8_BITS));
}

/**
 * called by updateTextures
 * TODO: more accurate resampling
 */
Image * Image::resizeImage(const uint16_t width_new, const uint16_t height_new, const uint8_t format, uint8_t type, const Image *image_old)
{
  if (type > 1)
    type = IMAGE_FIX;

  Image *image_new = new Image(width_new, height_new, format, type);

  uint16_t width_old, height_old;
  GLubyte *pix_new, *pix_old;

  width_old  = image_old->xsize;
  height_old = image_old->ysize;
  pix_new = (GLubyte *) image_new->pixmap;
  pix_old = (GLubyte *) image_old->pixmap;
 
  int x1inc = ((width_old - 1)  * NORM_8) / (width_new - 1);
  int y1inc = ((height_old - 1) * NORM_8) / (height_new - 1);

  int x, y, x1, y1;

  for (y1=0, y=0; y < height_new; y++) {
    int xi, yi, xf, yf;
    for (x1=0, x=0; x < width_new; x++) {
      xi = x1 >> (NORM16_BITS - NORM8_BITS);
      yi = y1 >> (NORM16_BITS - NORM8_BITS);
      xf = x1 & ((1 << (NORM16_BITS - NORM8_BITS))-1);
      yf = y1 & ((1 << (NORM16_BITS - NORM8_BITS))-1);

      if (xi < (width_old-1) && yi < (height_old-1)) {
        for (int k=0; k<format; k++)
          pix_new[k] = interpol(pix_old[(yi     * width_old+xi)   * format+k],
                                pix_old[(yi     * width_old+xi+1) * format+k],
                                pix_old[((yi+1) * width_old+xi)   * format+k],
                                pix_old[((yi+1) * width_old+xi+1) * format+k],
                                xf, yf);
      }
      else {
        for (int k=0; k<format; k++)
          pix_new[k] = pix_old[(yi * width_old + xi) * format+k];
      }
      pix_new += format;
      x1 += x1inc;
    }
    y1 += y1inc;
  }
  return image_new;
}

ImgReader::ImgReader()
{ }

ImgReader::ImgReader(void *_texc, ImageReader _read_func)
{
  img_hdl = _texc;
  read_func = _read_func;
}

FILE * ImgReader::downloadInTmp()
{
  char *tmpfile;

  if ((tmpfile = tempnam("/tmp", "vre")) == NULL) {
    perror("tempnam");
    return NULL;
  }

  FILE *fp;
  if ((fp = fopen(tmpfile, "wb")) == NULL) {
    perror("open create tmpfile");
    free(tmpfile);
    return NULL;
  }

  int len;
  char buf[BUFSIZ];
  while ((len = this->read_func(img_hdl, buf, sizeof(buf))) > 0)
    fwrite(buf, 1, len, fp);
  fclose(fp);
  if ((fp = fopen(tmpfile, "rb")) == NULL) {
    perror("open read tmpfile");
    free(tmpfile);
    return NULL;
  }
  unlink(tmpfile);
  free(tmpfile);
  return fp;
}

FILE * ImgReader::downloadInCache(const char *url, char *filename, bool flagclose)
{
  char *p = strrchr(url, '/');
  sprintf(filename, "%s%s", vrengcache, p);

  FILE *fp;
  if ((fp = fopen(filename, "rb")) == NULL) {
    if ((fp = fopen(filename, "wb")) == NULL) {
      error("downloadInCache: can't create %s", filename);
      return NULL;
    }

    int len;
    char buf[BUFSIZ];
    while ((len = this->read_func(img_hdl, buf, sizeof(buf))) > 0)
      fwrite(buf, 1, len, fp);
  }
  if (flagclose)
    fclose(fp);
  return fp;
}

void ImgReader::unlinkInTmp(char *filename)
{
  //pd unlink(filename);
}

void ImgReader::get_ch()
{
  uint8_t data;

  if (this->read_func(img_hdl, (char *) &data, 1) < 1)
    data = 0;
  ch = data;
}

void ImgReader::get_ch1()
{
  get_ch();
  while (ch == '#') {
    do
      get_ch();
    while (ch != 0 && ch != '\n' && ch != '\r');
    get_ch();
  }
}

void ImgReader::skip_spaces()
{
  while (isspace(ch))
    get_ch1();
}

int ImgReader::get_int()
{
  int n = 0;

  if (!isdigit(ch))
    return -1; // returns -1 if error
  do {
    n = n*10 + ch-'0';
    get_ch1();
  } while (isdigit(ch));
  return n;
}

static uint8_t imgbuf[BUFSIZ];
static int32_t imgbuf_pos = 0;
static int32_t imgbuf_len = 0;

void imgClearBuf(void)
{
  imgbuf_pos = imgbuf_len = 0;
}

uint8_t imgGetC(void *texhdl, ImageReader read_func)
{
  if (imgbuf_pos >= imgbuf_len) {
    imgbuf_pos = 0;
    imgbuf_len = read_func(texhdl, (char *) imgbuf, sizeof(imgbuf));
  }
  return imgbuf[imgbuf_pos++];
}
