#include "global.h"
#include "images.h"

#if defined(HAVE_LIBJPEG) && defined(HAVE_JPEGLIB_H)
extern "C" {
#include <jpeglib.h>
#include <setjmp.h>
}

static void readJpegHeader(FILE* infile, struct jpeg_decompress_struct& cinfo) {
  // Step 1: allocate and initialize JPEG decompression object
  jpeg_create_decompress(&cinfo);

  // Step 2: specify data source (eg, a file)
  jpeg_stdio_src(&cinfo, infile);

  // Step 3: read file parameters with jpeg_read_header()
  (void) jpeg_read_header(&cinfo, TRUE);
}

static void readJpegData(Image *ima, struct jpeg_decompress_struct& cinfo) {
  JSAMPARRAY buffer;	// Output row buffer
  int row_stride;	// physical row width in output buffer
  
  // Step 4: set parameters for decompression
  
  // Step 5: Start decompressor
  (void) jpeg_start_decompress(&cinfo);

  row_stride = cinfo.output_width * cinfo.output_components;

  // Make a one-row-high sample array that will go away when done with image
  buffer = (*cinfo.mem->alloc_sarray)
                ((j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1);

  // Step 6: while (scan lines remain to be read)
  //           jpeg_read_scanlines(...);
  trace(DBG_VGL, "readJpegData: width=%d height=%d depth=%d",
                 cinfo.output_width, cinfo.output_height, cinfo.output_components);
  int k = 0;
  int y = 0;
  while (cinfo.output_scanline < cinfo.output_height) {
    (void) jpeg_read_scanlines(&cinfo, buffer, 1);
    JSAMPROW prow = buffer[0];
  
    if (cinfo.output_components == 3) {      // RGB
      for (unsigned int x = 0; x < cinfo.output_width; x++) {
        ima->pixmap[k +0] = prow[0];
        ima->pixmap[k +1] = prow[1];
        ima->pixmap[k +2] = prow[2];
        prow += 3; 
        k += 3;
      }
    } 
    else if (cinfo.output_components == 1) {  // GREYSCALE
      for (unsigned int x = 0; x < cinfo.output_width; x++) {
        ima->pixmap[k] = *prow;
        prow++;
        k++;
      }
    }
    y++;
  } 

  // Step 7: Finish decompression
  (void) jpeg_finish_decompress(&cinfo);

  // Step 8: Release JPEG decompression object
  jpeg_destroy_decompress(&cinfo);
}

struct my_error_mgr {
  struct jpeg_error_mgr pub;	// "public" fields
  jmp_buf setjmp_buffer;	// for return to caller
};

static void my_error_exit(j_common_ptr cinfo)
{
  my_error_mgr* myerr = (my_error_mgr*) cinfo->err;
  // (*cinfo->err->output_message) (cinfo);
  longjmp(myerr->setjmp_buffer, 1);
}
#endif //!HAVE_LIBJPEG


Image * loadJPG(void *texc, ImageReader read_func)
{
#if HAVE_LIBJPEG
  ImgReader *ir = new ImgReader(texc, read_func);
  TextureCacheEntry *tc = (TextureCacheEntry *) texc;

  char filename[64];
  FILE *fp;

  if ((fp = ir->downloadInCache(tc->texurl, filename, 0)) == NULL)
    return NULL;
  delete ir;

  struct jpeg_decompress_struct cinfo;

  /* error handling */
  struct my_error_mgr jerr;
  cinfo.err = jpeg_std_error(&jerr.pub);
  jerr.pub.error_exit = my_error_exit;
  if (setjmp(jerr.setjmp_buffer)) {
    error("loadJPG: invalidData");
    jpeg_destroy_decompress(&cinfo);
    fclose(fp);
    return NULL;
  }

  /* we read the header */
  readJpegHeader(fp, cinfo);

  /* we read the data */
  if (cinfo.image_width == 0 || cinfo.image_height == 0
      // we can only decode RGB (num=3) or greayscale (num=1)
      || (cinfo.num_components != 3 && cinfo.num_components != 1)) {
    error("loadJPG: num_components=%d", cinfo.num_components);
    jpeg_destroy_decompress(&cinfo);
    fclose(fp);
    return NULL;
  }

  Image *image = new Image(cinfo.image_width, cinfo.image_height, cinfo.num_components, IMAGE_FIX);
  if (!image) {
    error("loadJPG: can't new image");
    jpeg_destroy_decompress(&cinfo);
    fclose(fp);
    return NULL;
  }

  readJpegData(image, cinfo);
  fclose(fp);

  return image;
#else
  return NULL;
#endif //!HAVE_LIBJPEG
}
