#include <ubit/ubit.hpp>

#include "global.h"
#include "landmark.h"


// landmark instance
static Landmark landmark;
static int counter_visible;


Landmark * Landmark::getLandmark()
{
  return &landmark;
}

Landmark::Landmark()
{
  landmark_lists = -1;
  counter_visible = true;
  axis_visible = false;
  grid_visible = false;
  overlap = false;
  grid_3d = false;
  behavior = BEHAVIOR_STICK;
}

void Landmark::defaultValue()
{
  posx = posy = posz = 0;
  rotx = roty = rotz = 0;
  grid_width = grid_depth = grid_height = LM_GRID_SLICE;
  grid_red = grid_green = grid_blue = grid_alpha = 1;

// FIXME see handle functions
//  s_x->setValue(posx + 50);
//  s_y->setValue(posy + 50);
//  s_z->setValue(posz + 50);
}

void Landmark::init()
{
  defaultValue();

  // the default matrix transformation
  // FIXME are these values always valid ? if not get them from a render class 
  glmat[0]  = 0;
  glmat[1]  = 0;
  glmat[2]  = -1;
  glmat[3]  = 0;
  
  glmat[4]  = -1;
  glmat[5]  = 0;
  glmat[6]  = 0;
  glmat[7]  = 0;
  
  glmat[8]  = 0;
  glmat[9]  = 1;
  glmat[10] = 0;
  glmat[11] = 0;
  
  glmat[12] = 0;
  glmat[13] = -1.85;
  glmat[14] = 0;
  glmat[15] = 1;
  
  draw();
}

/*
 * Render functions
 */

void Landmark::drawGrid()
{
  int grid_3d_height = (grid_3d) ? grid_height : 0;

  glColor4f(grid_red, grid_green, grid_blue, grid_alpha);

  glBegin(GL_LINES);
  for (int j = -grid_3d_height; j <= grid_3d_height; j++) {
    float x, y, z;
    z = (float) (j * LM_GRID_HEIGHT) / grid_height;
    for (int i = -grid_width; i < grid_width; i++) { 
      y = (float) (i * LM_GRID_WIDTH) / grid_width;
      glVertex3f(-LM_GRID_DEPTH, y, z);
      glVertex3f( LM_GRID_DEPTH, y, z);
    }
    for (int i = -grid_depth; i < grid_depth; i++) { 
      x = (float) (i * LM_GRID_DEPTH) / grid_depth;
      glVertex3f(x, -LM_GRID_WIDTH, z);
      glVertex3f(x,  LM_GRID_WIDTH, z);
    }
  }
  
  if (grid_3d) {
    for (int j = -grid_width; j < grid_width; j++) {
      float x, y;
      y = (float) (j * LM_GRID_WIDTH) / grid_width;
      for (int i = -grid_depth; i < grid_depth; i++) {
	x = (float) (i * LM_GRID_DEPTH) / grid_depth;
	glVertex3f(x, y, -LM_GRID_HEIGHT);
	glVertex3f(x, y,  LM_GRID_HEIGHT);
      }
    }
  }
  glEnd();
}

void Landmark::drawAxis()
{
  glBegin(GL_LINES);
   glColor3f(1, .5, .5);
   glVertex3f(0, 0, 0);
   glColor3f(1, 0, 0);
   glVertex3f(LM_AXIS_LENGTH, 0, 0);

   glColor3f(.5, .1, .1);
   glVertex3f(0, 0, 0);
   glColor3f(0, 0, 0);
   glVertex3f(-LM_AXIS_LENGTH, 0, 0);

   glColor3f(.5, 1, .5);
   glVertex3f(0, 0, 0);
   glColor3f(0, 1, 0);
   glVertex3f(0, LM_AXIS_LENGTH, 0);

   glColor3f(.1, .5, .1);
   glVertex3f(0, 0, 0);
   glColor3f(0, 0, 0);
   glVertex3f(0, -LM_AXIS_LENGTH, 0);

   glColor3f(.5, .5, 1);
   glVertex3f(0, 0, 0);
   glColor3f(0, 0, 1);
   glVertex3f(0, 0, LM_AXIS_LENGTH);
 
   glColor3f(.1, .1, .5);
   glVertex3f(0, 0, 0);
   glColor3f(0, 0, 0);
   glVertex3f(0, 0, -LM_AXIS_LENGTH);
  glEnd();
}

void Landmark::draw()
{
  // TODO ? use two different lists for grid and axis ?
  if (landmark_lists != -1)
    glDeleteLists(landmark_lists, 2);
  landmark_lists = glGenLists(2);
  glNewList(landmark_lists, GL_COMPILE);
  drawGrid();
  glEndList();
  glNewList(landmark_lists + 1, GL_COMPILE);
  drawAxis();
  glEndList();
}

void Landmark::displayGrid()
{
  if (! grid_visible)
    return;

//  float grid_diffuse[] = {1.0, 1.0, 1.0, 1};
//  glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, grid_diffuse);

  glPushMatrix();
   glTranslatef(posx, posy, posz);
   glRotatef(rotz, 0, 0, 1);
   glRotatef(rotx, 0, 1, 0);
   glRotatef(roty, 1, 0, 0);
   glCallList(landmark_lists);
  glPopMatrix();
}

void Landmark::displayAxis()
{
  if (! axis_visible)
    return;

  glCallList(landmark_lists + 1);
}

void Landmark::render()
{
  if (! grid_visible && ! axis_visible)
    return;
  
  glPushMatrix();

  if (overlap)
    glDisable(GL_DEPTH_TEST); // no overlap by other objects
  glDisable(GL_LIGHTING);
  
  // get rid of translation movement:
  float projection[16];
  glGetFloatv(GL_MODELVIEW_MATRIX, projection);
  projection[12] = 0;
  projection[13] = 0;
  projection[14] = 0;
  glLoadMatrixf(projection);

  glLineWidth(3);
  displayAxis();
  glLineWidth(1);

  if (behavior == BEHAVIOR_SFOLLOW) // load default view matrix
    glLoadMatrixf(glmat);
  else if (behavior == BEHAVIOR_STICK) {
    glPopMatrix();
    glPushMatrix();
  }

  displayGrid();

  glEnable(GL_LIGHTING);
  if (overlap)
    glEnable(GL_DEPTH_TEST);

  glPopMatrix();
}

/*
 * Dialog functions
 */

void Landmark::genScrollbar()
{
  s_width = &uhscrollbar(UOn::change / ucall(this, &Landmark::setWidth));
  s_width->setValue(LM_GRID_SLICE);
  s_width->setScrollerIncrement(1);
  s_width->setBackgroundIncrement(1);

  s_height = &uhscrollbar(UOn::change / ucall(this, &Landmark::setHeight));
  s_height->setValue(LM_GRID_SLICE);
  s_height->setScrollerIncrement(1);
  s_height->setBackgroundIncrement(1);

  s_depth = &uhscrollbar(UOn::change / ucall(this, &Landmark::setDepth));
  s_depth->setValue(LM_GRID_SLICE);
  s_depth->setScrollerIncrement(1);
  s_depth->setBackgroundIncrement(1);

  s_red = &uhscrollbar(UOn::change / ucall(this, &Landmark::setRed));
  s_red->setValue(LM_SCROLL_MAX);
  s_red->setScrollerIncrement(10);
  s_red->setBackgroundIncrement(10);

  s_green = &uhscrollbar(UOn::change / ucall(this, &Landmark::setGreen));
  s_green->setValue(LM_SCROLL_MAX);
  s_green->setScrollerIncrement(10);
  s_green->setBackgroundIncrement(10);

  s_blue = &uhscrollbar(UOn::change / ucall(this, &Landmark::setBlue));
  s_blue->setValue(LM_SCROLL_MAX);
  s_blue->setScrollerIncrement(10);
  s_blue->setBackgroundIncrement(10);

  s_x = &uhscrollbar(UOn::change / ucall(this, &Landmark::setPosX));
  s_x->setScrollerIncrement(1);
  s_x->setBackgroundIncrement(1);
  
  s_y = &uhscrollbar(UOn::change / ucall(this, &Landmark::setPosY));
  s_y->setScrollerIncrement(1);
  s_y->setBackgroundIncrement(1);
  
  s_z = &uhscrollbar(UOn::change / ucall(this, &Landmark::setPosZ));
  s_z->setScrollerIncrement(1);
  s_z->setBackgroundIncrement(1);

  s_rotx = &uhscrollbar(UOn::change / ucall(this, &Landmark::setRotX));
  s_rotx->setScrollerIncrement(1);
  s_rotx->setBackgroundIncrement(1);
  
  s_roty = &uhscrollbar(UOn::change / ucall(this, &Landmark::setRotY));
  s_roty->setScrollerIncrement(1);
  s_roty->setBackgroundIncrement(1);
  
  s_rotz = &uhscrollbar(UOn::change / ucall(this, &Landmark::setRotZ));
  s_rotz->setScrollerIncrement(1);
  s_rotz->setBackgroundIncrement(1);
}

UDialog * Landmark::buildDialog()
{
  return getLandmark()->dialog();
}

UDialog * Landmark::dialog()
{
  URadioSelect &behavior = uradioSelect();

  genScrollbar();

  UBox& part1 = uvbox
    (
     uvbox(UBorder::etchedIn
	   + uhbox(UColor::blue + UFont::bold + "View")
	   + ucheckbox("Visible Counters" + UMode::selected + ucall(this, &Landmark::toggleCounterVisible))
	   + ucheckbox("Visible Axis" + ucall(this, &Landmark::toggleAxisVisible))
	   + ucheckbox("Visible Grid" + ucall(this, &Landmark::toggleGridVisible))
	   + ucheckbox("3D Grid" + ucall(this, &Landmark::toggleGrid3d))
	   + ucheckbox("Overlap" + ucall(this, &Landmark::toggleOverlap))
	   )
     +
     uvbox(UBorder::etchedIn
	   + uhbox(UColor::blue + UFont::bold + "Behavior")
	   + ucheckbox(behavior + UMode::selected + "Stick to the world" +
			UOn::select / ucall(this, BEHAVIOR_STICK,
					    &Landmark::toggleBehavior))
	   + ucheckbox(behavior + "Follow" 
			+ UOn::select / ucall(this, BEHAVIOR_FOLLOW, &Landmark::toggleBehavior))
	   + ucheckbox(behavior + "Strict follow" + UOn::select / ucall
			(this, BEHAVIOR_SFOLLOW, &Landmark::toggleBehavior))
	   )
     +
     uvbox(UBorder::etchedIn
	   + ubutton("Reset" + ucall(this, &Landmark::defaultValue))
	   )
     );

  UBox& part2 = uvbox
    (
     uwidth(160)
     +
     uvbox(UBorder::etchedIn
	   + uhbox(UColor::blue + UFont::bold + "Slice ")
	   + uhbox("  Width:  " + uhflex() + *s_width)
	   + uhbox("  Height: " + uhflex() + *s_height)
	   + uhbox("  Depth:  " + uhflex() + *s_depth)
	   )
     +
     uvbox(UBorder::etchedIn
	   + uhbox(UColor::blue + UFont::bold + "Orientation")
	   + uhbox("  Rot X: " + uhflex() + *s_rotx)
	   + uhbox("  Rot Y: " + uhflex() + *s_roty)
	   + uhbox("  Rot Z: " + uhflex() + *s_rotz)
     )
     +
     uvbox(UBorder::etchedIn
	   + uhbox(UColor::blue + UFont::bold + "Position")
	   + uhbox("  Pos X: " + uhflex() + *s_x)
	   + uhbox("  Pos Y: " + uhflex() + *s_y)
	   + uhbox("  Pos Z: " + uhflex() + *s_z)
	   )
     +
     uvbox(UBorder::etchedIn
	   + uhbox(UColor::blue + UFont::bold + "Color ")
	   + uhbox("  R: " + uhflex() + *s_red)
	   + uhbox("  G: " + uhflex() + *s_green)
	   + uhbox("  B: " + uhflex() + *s_blue)
	   )
     );

  return &udialog 
    (
     utitle("Axis/Grid") 
     + ulabel("Axis/Grid") 
     + UBgcolor::white
     + uhbox(uhmargin(8) + uvmargin(8) + part1 + " " + uhflex() + part2)
     + ubutton(UFont::bold + uhcenter() + " Close " + ucloseWin())
     );
}

/*
 * Callback functions
 */

void Landmark::toggleOverlap()
{ overlap = 1 - overlap; }
void Landmark::toggleBehavior(const int new_behavior)
{ behavior = new_behavior; }

void Landmark::toggleCounterVisible()
{ counter_visible = 1 - counter_visible; }

void Landmark::toggleAxisVisible()
{ axis_visible = 1 - axis_visible; }

void Landmark::toggleGridVisible()
{ grid_visible = 1 - grid_visible; }

void Landmark::toggleGrid3d()
{ 
  grid_3d = 1 - grid_3d;
  draw();
}

bool Landmark::isCounterVisible()
{
  return counter_visible;
}

/* changing width and depth of the grid */
void Landmark::setWidth(UEvent &e)
{
  grid_width = (int)((UScrollbar *)e.getSource())->getValue();
  // TODO: don't use getValue: find another way to read the value (argument ?)
  if (grid_width <= 0)
    grid_width = 1;
  draw();
}
void Landmark::setHeight(UEvent &e)
{
  grid_height = (int)((UScrollbar *)e.getSource())->getValue();
  if (grid_height <= 0)
    grid_height = 1;
  draw();
}
void Landmark::setDepth(UEvent &e)
{
  grid_depth = (int)((UScrollbar *)e.getSource())->getValue();
  if (grid_depth <= 0)
    grid_depth= 1;
  draw();
}
void Landmark::setRed(UEvent &e)
{
  grid_red = ((UScrollbar *)e.getSource())->getValue() / LM_SCROLL_MAX;
  draw();
}
void Landmark::setGreen(UEvent &e)
{
  grid_green = ((UScrollbar *)e.getSource())->getValue() / LM_SCROLL_MAX;
  draw();
}
void Landmark::setBlue(UEvent &e)
{
  grid_blue = ((UScrollbar *)e.getSource())->getValue() / LM_SCROLL_MAX;
  draw();
}

void Landmark::setPosX(UEvent &e)
{
  posx = ((UScrollbar *)e.getSource())->getValue() - 50;
}
void Landmark::setPosY(UEvent &e)
{
  posy = ((UScrollbar *)e.getSource())->getValue() - 50;
}
void Landmark::setPosZ(UEvent &e)
{
  posz = ((UScrollbar *)e.getSource())->getValue() - 50;
}

void Landmark::setRotX(UEvent &e)
{
  rotx = ((UScrollbar *)e.getSource())->getValue();
}
void Landmark::setRotY(UEvent &e)
{
  roty = ((UScrollbar *)e.getSource())->getValue();
}
void Landmark::setRotZ(UEvent &e)
{
  rotz = ((UScrollbar *)e.getSource())->getValue();
}
