/*
 * Copyright (C) 1998 Janne Lf <jlof@mail.student.oulu.fi>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "global.h"
#include "http.h"	// Http::Skip
#include "lwo.h"

#define MK_ID(a,b,c,d) ((((uint32_t)(a))<<24)| \
			(((uint32_t)(b))<<16)| \
			(((uint32_t)(c))<< 8)| \
			(((uint32_t)(d))    ))

#define ID_FORM MK_ID('F','O','R','M')
#define ID_LWOB MK_ID('L','W','O','B')
#define ID_PNTS MK_ID('P','N','T','S')
#define ID_SRFS MK_ID('S','R','F','S')
#define ID_SURF MK_ID('S','U','R','F')
#define ID_POLS MK_ID('P','O','L','S')
#define ID_COLR MK_ID('C','O','L','R')

static int32_t read_char(Http *http)
{
  return http->GetByte();
}

static int32_t read_short(Http *f)
{
  return (read_char(f)<<8) | read_char(f);
}

static int32_t read_long(Http *f)
{
  return (read_char(f)<<24) | (read_char(f)<<16) | (read_char(f)<<8) | read_char(f);
}

static GLfloat read_float(Http *f)
{
  int32_t x = read_long(f);
  return *(GLfloat*)&x;
}

static int read_string(Http *f, char *s)
{
  int c;
  int cnt = 0;
  do {
    c = read_char(f);
    if (cnt < LW_MAX_NAME_LEN)
      s[cnt] = c;
    else
      s[LW_MAX_NAME_LEN-1] = 0;
    cnt++;
  } while (c != 0);
  /* if length of string (including \0) is odd skip another byte */
  if (cnt%2) {
    read_char(f);
    cnt++;
  }
  return cnt;
}

static void read_srfs(Http *f, int nbytes, lwObject *lwo)
{
  int guess_cnt = lwo->material_cnt;

  while (nbytes > 0) {
    lwMaterial *material;

    /* allocate more memory for materials if needed */
    if (guess_cnt <= lwo->material_cnt) {
      guess_cnt += guess_cnt/2 + 4;
      lwo->material = (lwMaterial*) realloc(lwo->material, sizeof(lwMaterial)*guess_cnt);
    }
    material = lwo->material + lwo->material_cnt++;

    /* read name */
    nbytes -= read_string(f,material->name);

    /* defaults */
    material->r = 0.7;
    material->g = 0.7;
    material->b = 0.7;
  }
  lwo->material = (lwMaterial*) realloc(lwo->material, sizeof(lwMaterial)*lwo->material_cnt);
}

static void read_surf(Http *f, int nbytes, lwObject *lwo)
{
  char name[LW_MAX_NAME_LEN];
  lwMaterial *material = NULL;

  Http *http = (Http *) f;

  /* read surface name */
  nbytes -= read_string(f,name);

  /* find material */
  for (int i=0; i < lwo->material_cnt; i++) {
    if (strcmp(lwo->material[i].name,name) == 0) {
      material = &lwo->material[i];
      break;
    }
  }

  /* read values */
  while (nbytes > 0) {
    int id = read_long(f);
    int len = read_short(f);
    nbytes -= 6 + len + (len%2);

    switch (id) {
    case ID_COLR:
      material->r = read_char(f) / 255.0;
      material->g = read_char(f) / 255.0;
      material->b = read_char(f) / 255.0;
      material->alpha=1;
      read_char(f); // dummy
      break;
    default:
      http->Skip(len+(len%2));
    }
  }
}

static void read_pols(Http *f, int nbytes, lwObject *lwo)
{
  int guess_cnt = lwo->face_cnt;

  Http *http = (Http *) f;
  
  while (nbytes > 0) {
    lwFace *face;

    /* allocate more memory for polygons if necessary */
    if (guess_cnt <= lwo->face_cnt) {
      guess_cnt += guess_cnt + 4;
      lwo->face = (lwFace*) realloc((void*) lwo->face, sizeof(lwFace)*guess_cnt);
    }
    face = lwo->face + lwo->face_cnt++;

    /* number of points in this face */
    face->index_cnt = read_short(f);
    nbytes -= 2;

    /* allocate space for points */
    face->index = (int*) calloc(sizeof(int)*face->index_cnt,1);
    
    /* read points in */
    for (int i=0; i < face->index_cnt; i++) {
      face->index[i] = read_short(f);
      nbytes -= 2;
    }
    
    /* read surface material */
    face->material = read_short(f);
    nbytes -= 2;
    
    /* skip over detail  polygons */
    if (face->material < 0) {
      int det_cnt;
      face->material = -face->material;
      det_cnt = read_short(f);
      nbytes -= 2;
      while (det_cnt-- > 0) {
	int cnt = read_short(f);
	http->Skip(cnt*2+2);
	nbytes -= cnt*2+2;
      }
    }
    face->material -= 1;
  }
  /* readjust to true size */
  lwo->face = (lwFace*) realloc(lwo->face, sizeof(lwFace)*lwo->face_cnt);
}

static void read_pnts(Http *f, int nbytes, lwObject *lwo)
{
  lwo->vertex_cnt = nbytes / 12;
  lwo->vertex = (float*) calloc(sizeof(GLfloat)*lwo->vertex_cnt*3, 1);
  for (int i=0; i < lwo->vertex_cnt; i++) {
    lwo->vertex[i*3+0] = read_float(f);
    lwo->vertex[i*3+1] = read_float(f);
    lwo->vertex[i*3+2] = read_float(f);
  }
}

void lwoHttpReader(void *alwo, Http *http)
{
  lwObject **lwo = (lwObject **) alwo;

  if (! http) {
    *lwo = NULL;
    error("lwoHttpReader: http NULL");
    return;
  }

  httpClearBuf();

  int32_t form_bytes = 0;
  int32_t read_bytes = 0;

  /* check for headers */
  if (read_long(http) != ID_FORM) {
    error("lwoHttpReader: error ID_FORM");
    return;
  }
  form_bytes = read_long(http);
  read_bytes += 4;

  if (read_long(http) != ID_LWOB) {
    error("lwoHttpReader: not a LWOB file");
    return;
  }

  /* create new lwObject */
  if ((*lwo = (lwObject*) calloc(sizeof(lwObject), 1)) == NULL) {
    warning("lwoHttpReader: can't alloc");
    return;
  }

  /* read chunks */
  while (read_bytes < form_bytes) {
    int32_t  id     = read_long(http);
    int32_t  nbytes = read_long(http);
    read_bytes += 8 + nbytes + (nbytes%2);

    switch (id) {
    case ID_PNTS:
      read_pnts(http, nbytes, *lwo);
      break;
    case ID_POLS:
      read_pols(http, nbytes, *lwo);
      break;
    case ID_SRFS:
      read_srfs(http, nbytes, *lwo);
      break;
    case ID_SURF:
      read_surf(http, nbytes, *lwo);
      break;
    default:
      http->Skip(nbytes + (nbytes%2));
    }
  }
  return;
}

void lw_object_free(lwObject *lw_object)
{
  if (lw_object->face) {
    for (int i=0; i < lw_object->face_cnt; i++) {
      if (lw_object->face[i].index)
        free(lw_object->face[i].index);
    }
    free(lw_object->face);
    lw_object->face = NULL;
  }
  free(lw_object->material);
  free(lw_object->vertex);
  free(lw_object);
}


#define PX(i) (lw_object->vertex[face->index[i]*3+0])
#define PY(i) (lw_object->vertex[face->index[i]*3+1])
#define PZ(i) (lw_object->vertex[face->index[i]*3+2])

void lw_object_show(const lwObject *lw_object)
{
  int prev_index_cnt = -1;
  int prev_material  = -1;
  GLfloat prev_nx = 0;
  GLfloat prev_ny = 0;
  GLfloat prev_nz = 0;

  for (int i=0; i < lw_object->face_cnt; i++) {
    GLfloat ax,ay,az,bx,by,bz,nx,ny,nz,r;
    const lwFace *face = lw_object->face+i;

    /* ignore faces with less than 3 points */
    if (face->index_cnt < 3)
      continue;

    /* calculate normal */
    ax = PX(1) - PX(0);
    ay = PY(1) - PY(0);
    az = PZ(1) - PZ(0);

    bx = PX(face->index_cnt-1) - PX(0);
    by = PY(face->index_cnt-1) - PY(0);
    bz = PZ(face->index_cnt-1) - PZ(0);

    nx = ay * bz - az * by;
    ny = az * bx - ax * bz;
    nz = ax * by - ay * bx;

    r = sqrt(nx*nx + ny*ny + nz*nz);
    if (r < 0.000001) /* avoid division by zero */
      continue;
    nx /= r;
    ny /= r;
    nz /= r;

    /* glBegin/glEnd */
    if (prev_index_cnt != face->index_cnt || prev_index_cnt > 4) {
      if (prev_index_cnt > 0) glEnd();
      prev_index_cnt = face->index_cnt;
      switch (face->index_cnt) {
      case 3:
	glBegin(GL_TRIANGLES);
	break;
      case 4:
	glBegin(GL_QUADS);
	break;
      default:
	glBegin(GL_POLYGON);
      }
    }

    /* update material if necessary */
    if (prev_material != face->material) {
      prev_material = face->material;
      GLfloat rgb[4];
      rgb[0]=lw_object->material[face->material].r;
      rgb[1]=lw_object->material[face->material].g;
      rgb[2]=lw_object->material[face->material].b;
      rgb[3]=lw_object->material[face->material].alpha;
      glTexEnvfv(GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, rgb);
      glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT,rgb);

      /*glColor3f(lw_object->material[face->material].r,
		  lw_object->material[face->material].g,
		  lw_object->material[face->material].b);*/
    }

    /* update normal if necessary */
    if (nx != prev_nx || ny != prev_ny || nz != prev_nz) {
      prev_nx = nx;
      prev_ny = ny;
      prev_nz = nz;
      glNormal3f(nx,ny,nz);
    }

    /* draw polygon/triangle/quad */
    for (int j=0; j < face->index_cnt; j++)
      glVertex3f(PX(j),PY(j),PZ(j));
  }

  /* if glBegin was called call glEnd */
  if (prev_index_cnt > 0)
    glEnd();

}

GLfloat lw_object_radius(const lwObject *lwo)
{
  double max_radius = 0.0;

  for (int i=0; i < lwo->vertex_cnt; i++) {
    GLfloat *v = &lwo->vertex[i*3];
    double r = v[0]*v[0] + v[1]*v[1] + v[2]*v[2];
    if (r > max_radius)
      max_radius = r;
  }
  return sqrt(max_radius);
}

void lw_object_scale(lwObject *lwo, GLfloat scale)
{
  for (int i=0; i < lwo->vertex_cnt; i++) {
    lwo->vertex[i*3+0] *= scale;
    lwo->vertex[i*3+1] *= scale;
    lwo->vertex[i*3+2] *= scale;
  }
}
