/* GiMd2Viewer - Quake2 model viewer
 * Copyright (C) 1998  Lionel ULMER <bbrox@mygale.org>
 *
 * Based on code Copyright (C) 1997 Trey Harrison <trey@crack.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "global.h"
#include "http.h"	// httpOpen
#include "vgl.h"
#include "md2.h"


Md2::Md2()
{
 numframes = 0;
 skinwidth = 0;
 skinheight = 0;
 texinfo = NULL;
 glcmds = NULL;
 frames = NULL;
}

Md2::~Md2()
{
  for (int f=0; f < numframes; f++)
    delete[] frames[f].vert_table;
  delete[] frames;
  delete[] glcmds;
  delete[] texinfo;
}

void Md2::drawMd2Model(int32_t frame, int32_t nextfrm, const float ratio, const float scale)
{
  /* keeps the frame value valid */
  if (numframes == 0) {
    error("drawMd2Model: numframes null");
    return;
  }

  frame %= numframes;
  nextfrm %= numframes;
 
  /* get the frames information */
  Frame *frame1 = frames + frame;
  Frame *frame2 = frames + nextfrm;
 
  /* do the gl commands */
  int32_t *glcmd = glcmds;
  int32_t nb_vert = 0;

  while (*glcmd != 0) {
    int32_t num_verts;

    /* determine the command to draw the triangles */
    if (*glcmd > 0) { // triangle strip
      num_verts = *glcmd;
      glBegin(GL_TRIANGLE_STRIP);
    }
    else { // triangle fan
      num_verts = -(*glcmd);
      glBegin(GL_TRIANGLE_FAN);
    }
    glcmd++;

    for (int i = 0; i < num_verts; i++) {
      Vec3 p;  /* Interpolated point */
      int32_t vert_index;

      /* Grab the vertex index */
      vert_index = *glcmd; glcmd++;

      /* Interpolate */
      p.x = frame1->vert_table[vert_index].x +
	(frame2->vert_table[vert_index].x -
	 frame1->vert_table[vert_index].x) * ratio;
      p.y = frame1->vert_table[vert_index].y +
	(frame2->vert_table[vert_index].y -
	 frame1->vert_table[vert_index].y) * ratio;
      p.z = frame1->vert_table[vert_index].z +
	(frame2->vert_table[vert_index].z -
	 frame1->vert_table[vert_index].z) * ratio;

      glTexCoord2f(texinfo[nb_vert].s, texinfo[nb_vert].t);
      Draw::vertex3f(p.x * scale, p.y * scale, p.z * scale);
      nb_vert++;
    }
    glEnd();
  }
}

int Md2::getDlistMd2Model(const int frame, const int nextfrm, const float inter, const float scale)
{
  int num = glGenLists(1);
  glNewList(num, GL_COMPILE);
  glEnable(GL_CULL_FACE);
  glCullFace(GL_FRONT);
  drawMd2Model(frame, nextfrm, inter, scale);
  glDisable(GL_CULL_FACE);
  glEndList();
  return num;
}

int Md2::getFrames(Md2Header *md2_hdr, Http *http)
{
  trace(DBG_WO, "getFrames: numframes=%d", numframes);

  /* converts the FrameInfos to Frames */
  frames = new Frame[numframes];

  int32_t offset = 0;
  for (int frame = 0; frame < numframes; frame++) {
    Vec3 scale, origin;

    scale.x = http->GetFloat(); offset += 4;
    scale.y = http->GetFloat(); offset += 4;
    scale.z = http->GetFloat(); offset += 4;
    origin.x = http->GetFloat(); offset += 4;
    origin.y = http->GetFloat(); offset += 4;
    origin.z = http->GetFloat(); offset += 4;

    http->GetBuf((char *) &(frames[frame].name), 16);
    offset += 16;
    frames[frame].vert_table = new Vertex[md2_hdr->num_xyz];

    /* loads the vertices */
    for (int i=0; i < md2_hdr->num_xyz; i++) {
      Trivertex cur_vert;
      Vertex *p = (frames[frame].vert_table) + i;
 
      cur_vert.x = http->GetByte(); offset += 1;
      cur_vert.y = http->GetByte(); offset += 1;
      cur_vert.z = http->GetByte(); offset += 1;
      cur_vert.lightnormalindex = http->GetByte(); offset += 1;
      p->x = ((cur_vert.x * scale.x) + origin.x);
      p->y = ((cur_vert.y * scale.y) + origin.y);
      p->z = ((cur_vert.z * scale.z) + origin.z);
    }
  }
  return offset;
}

/**
 * we keep only the commands and the index in the glcommands
 * we 'pre-parse' the texture coordinates
 * do not ask me how I found this formula :-))
 */
int32_t Md2::getGLCmds(Md2Header *md2_hdr, Http *http)
{
  trace(DBG_VGL, "getGLCmds");

  int32_t num_vertices = ((md2_hdr->num_tris + 2 * md2_hdr->num_glcmds - 2) / 7);
  texinfo = new TexInfo[num_vertices];
  glcmds = new int32_t[md2_hdr->num_glcmds - 2 * num_vertices];
  //trace(DBG_VGL, "glcmds size = %d", sizeof(int32_t) * (md2_hdr->num_glcmds - 2 * num_vertices));

  /* now transform the GL commands */
  int32_t *glcmds_copy = glcmds;
  TexInfo *_texinfo = texinfo;

  int32_t glcmd, offset;

  for (offset = 0, glcmd = 0; (glcmd = http->GetInt()) != 0; ) {
    int32_t nb_verts;

    offset += 4;
    //error("glcmd=%d offset=%d", glcmd, offset);
  
    /* determine the command to draw the triangles */
    if (glcmd > 0)	// triangle strip
      nb_verts = glcmd;
    else		// triangle fan
      nb_verts = -glcmd;
    *(glcmds_copy++) = glcmd;	// copy the command
 
    /* gets the texture information */
    for (int i=0; i < nb_verts; i++) {
      _texinfo->s = http->GetFloat(); offset += 4;
      _texinfo->t = http->GetFloat(); offset += 4;
      _texinfo++;
 
      /* we keep the vertex index */
      glcmd = http->GetInt(); offset += 4;
      *(glcmds_copy++) = glcmd;	// copy the vertex index
    }
  }
  /* do not forget to copy the zero :-) */
  *(glcmds_copy++) = glcmd;
  return offset + 4;	// or offset + 4 ?
}

void md2ModelHttpReader(void *_md2, Http *http)
{
  Md2 **md2 = (Md2 **) _md2;
 
  if (! http)
    return;

  httpClearBuf();

  /* Read the header */
  Md2Header md2_hdr;
  int32_t offset = 0;

  http->GetBuf((char *) &(md2_hdr.ident), 4); offset += 4;
  md2_hdr.version = http->GetInt(); offset += 4;
  md2_hdr.skinwidth = http->GetInt(); offset += 4;
  md2_hdr.skinheight = http->GetInt(); offset += 4;
  md2_hdr.framesize = http->GetInt(); offset += 4;
  md2_hdr.num_skins = http->GetInt(); offset += 4;
  md2_hdr.num_xyz = http->GetInt(); offset += 4;
  md2_hdr.num_st = http->GetInt(); offset += 4;
  md2_hdr.num_tris = http->GetInt(); offset += 4;
  md2_hdr.num_glcmds = http->GetInt(); offset += 4;
  md2_hdr.num_frames = http->GetInt(); offset += 4;
  md2_hdr.ofs_skins = http->GetInt(); offset += 4;
  md2_hdr.ofs_st = http->GetInt(); offset += 4;
  md2_hdr.ofs_tris = http->GetInt(); offset += 4;
  md2_hdr.ofs_frames = http->GetInt(); offset += 4;
  md2_hdr.ofs_glcmds = http->GetInt(); offset += 4;
  md2_hdr.ofs_end = http->GetInt(); offset += 4;

  /* check if this is really a .MD2 file :-) */
  if (strncmp(md2_hdr.ident, "IDP2", 4)) {
    *md2 = NULL;
    return;
  }
 
  /* create the model */
  *md2 = new Md2();
 
  /* we do not need all the info from the header, just some of it */
  (*md2)->numframes = md2_hdr.num_frames;
  (*md2)->skinwidth = md2_hdr.skinwidth;
  (*md2)->skinheight = md2_hdr.skinheight;

  /* check which info is first */
  if (md2_hdr.ofs_frames > md2_hdr.ofs_glcmds) {
    http->Skip(md2_hdr.ofs_glcmds - offset);
    offset = md2_hdr.ofs_glcmds;
    offset += (*md2)->getGLCmds(&md2_hdr, http);
    http->Skip(md2_hdr.ofs_frames - offset);
    offset = md2_hdr.ofs_frames;
    offset += (*md2)->getFrames(&md2_hdr, http);
  }
  else {
    http->Skip(md2_hdr.ofs_frames - offset);
    offset = md2_hdr.ofs_frames;
    offset += (*md2)->getFrames(&md2_hdr, http);
    http->Skip(md2_hdr.ofs_glcmds - offset);
    offset = md2_hdr.ofs_glcmds;
    offset += (*md2)->getGLCmds(&md2_hdr, http);
  }
  return;
}

Md2 * Md2::loadMd2Model(const char *url)
{
  Md2 *md2 = NULL;	// filled by md2ModelHttpReader
 
  httpOpen(url, md2ModelHttpReader, &md2, THREAD_NO_BLOCK);
  return md2;
}
