/* GiMd2Viewer - Quake2 model viewer
 * Copyright (C) 1998  Lionel ULMER <bbrox@mygale.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "global.h"
#include "images.h"


typedef struct {
  uint8_t Manufacturer;
  uint8_t Version;
  uint8_t Encoding;
  uint8_t BitsPerPixel;
  uint8_t Xmin_l; uint8_t Xmin_h;
  uint8_t Ymin_l; uint8_t Ymin_h;
  uint8_t Xmax_l; uint8_t Xmax_h;
  uint8_t Ymax_l; uint8_t Ymax_h;
  uint8_t HDPI_l; uint8_t HDPI_h;
  uint8_t VDPI_l; uint8_t VDPI_h;
  uint8_t Colormap[48];
  uint8_t Reserved;
  uint8_t NPlanes;
  uint8_t BytesPerLine_l; uint8_t BytesPerLine_h;
  uint8_t PaletteInfo_l; uint8_t PaletteInfo_h;
  uint8_t HScreenSize_l; uint8_t HScreeSize_h;
  uint8_t VScreenSize_l; uint8_t VScreeSize_h;
  uint8_t Filler[54];
} PcxInfo;

typedef struct {
  uint8_t r;
  uint8_t g;
  uint8_t b;
} PcxColor;


Image * loadPCX(void *texc, ImageReader read_func)
{
  PcxInfo pcxInfo;
  PcxColor palette[256];

  /* fool proofing */
  if (! texc)
    return NULL;

  imgClearBuf();

  /* loads the Header */
  read_func(texc, (char *) &pcxInfo, sizeof(pcxInfo));

  /* test if this is really a PCX file */
  if (pcxInfo.Manufacturer != 0x0A) {
    error("loadPCX: not a PCX format: %x", pcxInfo.Manufacturer);
    return NULL;
  }

  /* only decode one type of PCX */
  if ((pcxInfo.Version != 0x05) || (pcxInfo.BitsPerPixel != 8) ||
      (pcxInfo.NPlanes != 0x01))
    return NULL;
 
  /* gets the image dimensions */
  uint16_t xsize = ((pcxInfo.Xmax_l | (pcxInfo.Xmax_h << 8)) - (pcxInfo.Xmin_l | (pcxInfo.Xmin_h << 8))) + 1;
  uint16_t ysize = ((pcxInfo.Ymax_l | (pcxInfo.Ymax_h << 8)) - (pcxInfo.Ymin_l | (pcxInfo.Ymin_h << 8))) + 1;
  uint16_t bpl = (pcxInfo.BytesPerLine_l | (pcxInfo.BytesPerLine_h << 8));
 
  /* temporary indexed buffer */
  uint8_t *img = new uint8_t[xsize * ysize];
  
  /* start decoding the file */
  uint8_t *tmp = img;
  for (int y=0; y < ysize; y++) {
    int count = 0;
    uint8_t data = 0x00;
  
    /* decodes one line */
    int x = 0;
    while (x < bpl) {
      if (count > 0) {
	if (x < xsize)
	  *(tmp++) = data;
	x++;
	count--;
      } else {
	data = imgGetC(texc, read_func);
	if ((data & 0xC0) == 0xC0) {
	  count = data & 0x3F;
	  data = imgGetC(texc, read_func);
	} else {
	  count = 1;
	}
      }
    }
  }

  /* decodes the palette */
  if (imgGetC(texc, read_func) != 0x0C) {
    delete[] img; img = NULL;
    return NULL;
  }
  for (int i=0; i < 256; i++) {
    palette[i].r = imgGetC(texc, read_func);
    palette[i].g = imgGetC(texc, read_func);
    palette[i].b = imgGetC(texc, read_func);
  }

  /* allocates image */
  Image *image = new Image(xsize, ysize, IMAGE_RGB, IMAGE_FIX);
  
  /* converts the indexed buffer to RGB */
  tmp = img;
  uint8_t *pix = image->pixmap;
  for (int x=0; x < xsize * ysize; x++) {
    *(pix++) = palette[*tmp].r;
    *(pix++) = palette[*tmp].g;
    *(pix++) = palette[*tmp].b;
    tmp++;
  }
  delete[] img; img = NULL;
  
  return image;
}
