#include "global.h"
#include "vgl.h"
#include "gcontext.h"	// gcontext
#include "texture.h"	// initCache
#include "txf.h"	// initTexfontCache
#include "wo.h"		// WObject
#include "world.h"	// worlds
#include "bgcolor.h"	// Bgcolor
#include "user.h"	// USER_TYPE
#include "landmark.h"	// Landmark
#include "glu.h"	// glu alternative


/*
 * Camera handling
 */

static struct _camera {
  float fovy, near, far;
} cam_user;

/**
 * Object viewed, called by update3D (all objects)
 */
void VGContext::setCameraPosition(M4 *vr_mat)
{
  camera_pos = *vr_mat;
}

/**
 * Observer view, called by updateCameraFromObject: world.c user.c
 */
void cameraProjection(float fovy, float near, float far)
{
  GLint viewport[4];
  glGetIntegerv(GL_VIEWPORT, viewport);
  int width = viewport[2];
  int height = viewport[3];
  float ratio = (float) width / (float) height;
  float top = near * tan(fovy * M_PI_180);
  float bottom = -top;
  float right = top * ratio;
  float left = -right;
  cam_user.fovy = fovy;
  cam_user.near = near;
  cam_user.far = far;

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glFrustum(left, right, bottom, top, near, far);
  glTranslatef(0.0, 0.0, -near); /* orig -0.4 */
  glMatrixMode(GL_MODELVIEW);
}


/*
 * Rendering
 */

void VGContext::eyePosition(float *gl_mat)
{
  glMatrixMode(GL_MODELVIEW);
  M4_to_GL(gl_mat, &camera_pos);
  glLoadMatrixf(gl_mat); 
}

/**
 * VGContext constructor
 */
VGContext::VGContext()
{
  first_solid = NULL;
  last_solid = NULL;
  first_bbox = 0;
  quality = false;
}

void initHints(const GLenum hint)
{
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, hint);
  glHint(GL_POINT_SMOOTH_HINT, hint);
  glHint(GL_LINE_SMOOTH_HINT, hint);
  glHint(GL_POLYGON_SMOOTH_HINT, hint);
  glHint(GL_FOG_HINT, hint);
}

void VglInit(const bool _quality)
{
  gcontext = new VGContext();

  gcontext->quality = _quality;

  glFrontFace(GL_CCW);		// trigo
  glEnable(GL_DEPTH_TEST);
  glEnable(GL_ALPHA_TEST);
  glEnable(GL_LIGHTING);
  glEnable(GL_LIGHT0);
  glEnable(GL_LIGHT1);
  glEnable(GL_CULL_FACE);
  glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  if (_quality)
    initHints(GL_NICEST);
  else
    initHints(GL_FASTEST);
   
  /* texture cache */
  TextureCacheEntry::initCache(_quality);
  /* texfont cache */
  initTexfontCache();

  Landmark::getLandmark()->init();

#if 0 //info
  GLint modelview, projection;
  glGetIntegerv(GL_MAX_MODELVIEW_STACK_DEPTH, &modelview);
  glGetIntegerv(GL_MAX_PROJECTION_STACK_DEPTH, &projection);
  trace(DBG_VGL, "matrices: modelview=%d projection=%d", modelview, projection);
#endif //info
}

void setMaterials(void)
{
  const float material_black[4] = {0, 0, 0, 1};

  glMaterialfv(GL_FRONT, GL_AMBIENT, material_black);
  glMaterialfv(GL_FRONT, GL_DIFFUSE, material_black);
  glMaterialfv(GL_FRONT, GL_SPECULAR, material_black);
  glMaterialfv(GL_FRONT, GL_EMISSION, material_black);
}

void specialRendering(int num)
{
#if 0 //pd
  ObjectList *metaList[] = { mobileList, stillList, invisibleList, NULL };

  for (ObjectList **meta = metaList; meta; meta++) {
    for (ObjectList *pl = *meta; pl; pl = pl->next) {
      if (pl->pobject && (pl->pobject->num == (uint32_t) num)) {  
        error("specialRendering: %s num=%d meta=%p pl=%p", pl->pobject->name.class_name, num, meta, pl);
        setMaterials();
        if (! pl->pobject->isBehavior(NO_SELECTABLE)) {
          glPopName();
          glPushName((int32_t)pl);
        }
        pl->pobject->render();
      }
    }
  }
#else //PG
  for (ObjectList *pl = mobileList; pl; pl = pl->next) {
    if (pl->pobject && (pl->pobject->num == (uint32_t) num)) {  
      //error("specialRendering: mobile %s num=%d", pl->pobject->name.class_name, num);
      setMaterials();
      if (! pl->pobject->isBehavior(NO_SELECTABLE)) {
        glPopName();
        glPushName((int32_t)pl);
      }
      pl->pobject->render();
    }
  }
  for (ObjectList *pl = stillList; pl; pl = pl->next) {
    if (pl->pobject && (pl->pobject->num == (uint32_t) num)) {  
      //error("specialRendering: still %s num=%d", pl->pobject->name.class_name, num);
      if (! pl->pobject->isBehavior(NO_SELECTABLE)) {
        glPopName();
        glPushName((int32_t)pl);
      }
      pl->pobject->render();
    }
  }
  for (ObjectList *pl = invisibleList; pl; pl = pl->next) {
    if (pl->pobject && (pl->pobject->num == (uint32_t) num)) {  
      //error("specialRendering: invisible %s", pl->pobject->name.class_name);
      if (! pl->pobject->isBehavior(NO_SELECTABLE)) {
        glPopName();
        glPushName((int32_t)pl);
      }
      pl->pobject->render();
    }
  }
#endif //PG
}

/** Display user */
//void displayMirroredUser(Pos &pos)
void displayMirroredUser(float az)
{
  int id = 0;

  glRotatef(RADIAN2DEGREE(az), 0, 0, 1);
  for (Solid *ps = gcontext->last_solid; ps != gcontext->first_solid ; ps = ps->prev) {
    if (ps->object && ps->object->type == USER_TYPE) {
      if (ps->id == id)
        continue;	// already seen
      id = ps->id;
      //trace(DBG_VGL,"displayMirroredUser: id=%d name=%s", ps->id, ps->object->name.class_name);
      glPushMatrix();
       //pd glTranslatef(0, -ps->object->pos.y, ps->object->pos.z - ps->object->pos.bbsize.v[2] /*0.5 * USER_DEFAULTHEIGHT*/);
       glRotatef(RADIAN2DEGREE(-(ps->object->pos.az)), 0, 0, 1);
       glRotatef(RADIAN2DEGREE(-(ps->object->pos.ax)), 1, 0, 0);
       glRotatef(RADIAN2DEGREE(-(ps->object->pos.ay)), 0, 1, 0);
       glCallList(ps->displaylist[0]);
      glPopMatrix();
    }
  }
}

void displayMirroredScene(float az)
{
  static Solid *ps = gcontext->first_solid;

  if (!ps || !ps->displaylist)
    ps = gcontext->first_solid;

  glRotatef(RADIAN2DEGREE(az), 0, 0, 1);
  if (ps->object) {
    error("displayMirroredScene: id=%d type=%d name=%s", ps->id, ps->object->type, ps->object->name.class_name);
    glRotatef(RADIAN2DEGREE(-(ps->object->pos.az)), 0, 0, 1);
    glRotatef(RADIAN2DEGREE(-(ps->object->pos.ax)), 1, 0, 0);
    glRotatef(RADIAN2DEGREE(-(ps->object->pos.ay)), 0, 1, 0);
    glCallList(ps->displaylist[0]);
  }
  ps = ps->next;
  if (ps == NULL)
    ps = gcontext->first_solid;
}

void Solid::displayObject()
{
  //glDisable(GL_LIGHTING);
  glCallList(displaylist[0]);
  //glEnable(GL_LIGHTING);
}

/**
 * Render displaylist
 */
int Solid::renderDisplayList(const int renderflag)
{
  float gl_mat[16];
#if 0
  struct _camera cam_mirror;
#endif

  if (! displaylist) {
    error("displaylist NULL, sid=%d", id);
    return 0;
  }
  glPopName();
  glPushName((int32_t)this);
  glPushMatrix();
   M4_to_GL(gl_mat, &posmat);
   glMultMatrixf(gl_mat);

   switch (renderflag) {
   case RENDER_NORMAL:
     if (object && object->type == USER_TYPE && id != 1) {
       glRotatef(RADIAN2DEGREE(-object->pos.ax), 1, 0, 0);
       glRotatef(RADIAN2DEGREE(-object->pos.ay), 0, 1, 0);
     }
     glCallList(displaylist[curframe]);
     break;

   case RENDER_FLASHY_BEGIN:
     glEnable(GL_POLYGON_OFFSET_FILL);
     glPolygonOffset(1., 1.);
     glCallList(displaylist[curframe]);
     glDisable(GL_POLYGON_OFFSET_FILL);
     break;
   case RENDER_FLASHY_END:
     glDisable(GL_LIGHTING);
#if 1
     glColor3fv(flashcolor);
#else
     glColor3f(0, 1, 0);
#endif
     glLineWidth(3);
     glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
     glCallList(displaylist[curframe]);
     glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
     glLineWidth(1);
     glEnable(GL_LIGHTING);
     break;

   case RENDER_REFLEXIVE:
#if 0 //projmirror
     cam_mirror.fovy = (float) USER_FOVY;
     cam_mirror.near = sqrt((bbcenter.v[0]-bbsize.v[0]) * (bbcenter.v[0]-bbsize.v[0]) + (bbcenter.v[1]-bbsize.v[1]) * (bbcenter.v[1]-bbsize.v[1]));
     cam_mirror.far = sqrt((bbcenter.v[0]+bbsize.v[0]) * (bbcenter.v[0]+bbsize.v[0]) + (bbcenter.v[1]+bbsize.v[1]) * (bbcenter.v[1]+bbsize.v[1]));
     glMatrixMode(GL_PROJECTION);
     glPushMatrix();
     glLoadIdentity();
     float top = cam_mirror.near * tan(cam_mirror.fovy * M_PI_180);
     float bottom = -top;
     float ratio = bbsize.v[0] / bbsize.v[2];
     float right = top * ratio;
     float left = -right;
     glFrustum(left, right, bottom, top, cam_mirror.near, cam_mirror.far);
     glTranslatef(0.0, 0.0, -cam_mirror.near);
     glPopMatrix();
     glMatrixMode(GL_MODELVIEW);
#endif //projmirror
     glDisable(GL_DEPTH_TEST);
     glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);
     glEnable(GL_STENCIL_TEST);
     glStencilFunc(GL_ALWAYS, 1, 0xffffffff);
     glStencilOp(GL_REPLACE, GL_REPLACE, GL_REPLACE);

     displayObject();	//draw_mirror

     glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
     glEnable(GL_DEPTH_TEST);
     glStencilFunc(GL_EQUAL, 1, 0xffffffff);
     glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
     glPushMatrix();
     //glScalef(-1, 1, 1);
     glEnable(GL_NORMALIZE);
     glCullFace(GL_FRONT);

     if (object) {
       displayMirroredUser(object->pos.az);
       //displayMirroredScene(object->pos.az);
     }

     glDisable(GL_NORMALIZE);
     glCullFace(GL_BACK);
     glPopMatrix();
     glDisable(GL_STENCIL_TEST);
     glEnable(GL_BLEND);
     glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

     displayObject();	//draw_mirror

     glDisable(GL_BLEND);
     break;
   }
  glPopMatrix();
  return 1;
}

void Solid::renderReflexive()
{
  renderDisplayList(RENDER_REFLEXIVE);
}

bool Solid::renderBlinking()
{
  if (blink) {
    blink = false;
    return true;
  }
  else {
    blink = true;
    return false;
  }
}

/**
 * Render solids
 */
void VGContext::renderSolids()
{
  uint8_t rendered[1024];

  int objectsnumber = WObject::getObjectsNumber();
  //pd int solidsnumber = Solid::getSolidsNumber();
  for (int i=0; i<objectsnumber && i < (int) sizeof(rendered) ; i++)
    rendered[i] = 0;

  // Display opaque solids first
  for (Solid *ps = first_solid; ps ; ps = ps->next) {
    if (ps->isvisible && !ps->istransparent) {
      if (ps->object) {
        //trace(DBG_VGL, "opaque: sid=%d name=%s", ps->id, ps->object->name.instance_name);
        if (ps->isblinking) {
          if (! ps->renderBlinking())
            continue;
        }
        if (ps->isreflexive)
          ps->renderReflexive();
        else
          ps->renderDisplayList(RENDER_NORMAL);
        //trace(DBG_VGL, "rendered: sid=%d num=%d name=%s", ps->id, ps->object->num, ps->object->name.instance_name);
        //pd rendered[ps->object->num] = 1;
      }

#if 0 //pg
      if (ps->next && ps->object && ps->next->object && ps->object->num != ps->next->object->num) {
        for (uint32_t i = ps->object->num; i < ps->next->object->num-1; i++) {
          //error("specialrender2 n=%i %s", i+1, ps->object->name.class_name);
          specialRendering(i+1);
          rendered[i+1] = 1;
        }
      }
#endif //pg
    }
  }

  // Specific rendering done by object itself
  for (int i=1; i < objectsnumber; i++) {
    if (! rendered[i])
      specialRendering(i);
  }

  // Display transparent solids last
  for (Solid *ps = first_solid; ps ; ps = ps->next) {
    if (ps->isblinking) {
      if (! ps->renderBlinking())
        continue;
    }
    if (ps->istransparent)
      ps->renderDisplayList(0);
    if (ps->isreflexive)
      ps->renderReflexive();
  }

  // Display flash contour
  for (Solid *ps = first_solid; ps ; ps = ps->next) {
    if (ps->isflashable && ps->isflashy) {
      ps->renderDisplayList(RENDER_FLASHY_BEGIN);
      ps->renderDisplayList(RENDER_FLASHY_END);
      //ps->setSolidFlashy(false);
    }
  }
}

static
void renderColorBackground(void)
{
  glClearColor(worlds->bgcolor->r, worlds->bgcolor->g, worlds->bgcolor->b, worlds->bgcolor->a);
  //pd glClear(GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT);
  glClear(GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT | GL_STENCIL_BUFFER_BIT | GL_ACCUM_BUFFER_BIT);
}

static
void renderLights(void)
{
  //PD static const GLfloat light_ambient0[4] = {0.4, 0.4, 0.4, 1};
  static const GLfloat light_ambient0[4] = {0, 0, 0, 1};
  //PD static const GLfloat light_diffuse0[4] = {0.8, 0.8, 0.8, 1};
  static const GLfloat light_diffuse0[4] = {1, 1, 1, 1};
  static const GLfloat light_specular0[4] = {1, 1, 1, 1};
  //PD static const GLfloat light_position0[4] = {1, 0, -1, 0};
  static const GLfloat light_position0[4] = {1, 1, 2, 0};

  static const GLfloat light_ambient1[] =  {0.2, 0.2, 0.2, 1};
  static const GLfloat light_diffuse1[] =  {1.0, 1.0, 1.0, 1};
  static const GLfloat light_specular1[] = {1.0, 1.0, 1.0, 1};
  //pd static const GLfloat light_position1[] = {50, 50, 20, 0};
  static const GLfloat light_position1[4] = {-0.5, 0.7, -1, 0};

  glLightfv(GL_LIGHT0, GL_AMBIENT, light_ambient0);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse0);
  glLightfv(GL_LIGHT0, GL_SPECULAR, light_specular0);
  glLightfv(GL_LIGHT0, GL_POSITION, light_position0);
  glLightfv(GL_LIGHT1, GL_AMBIENT, light_ambient1);
  glLightfv(GL_LIGHT1, GL_DIFFUSE, light_diffuse1);
  glLightfv(GL_LIGHT1, GL_SPECULAR, light_specular1);
  glLightfv(GL_LIGHT1, GL_POSITION, light_position1);

  glEnable(GL_LIGHT0);
  glEnable(GL_LIGHT1);
}


/**
 * General rendering
 */
void VglRender(void)
{
  float gl_mat[16];

  gcontext->eyePosition(gl_mat);	// eye position
  TextureCacheEntry::updateTextures();	// textures
  renderColorBackground();		// background
  renderLights();			// lights
  gcontext->renderSolids();		// all other solids
  Landmark::getLandmark()->render();	// grid and axis
}

void VglQuit(void)
{
  TextureCacheEntry::closeCache();
}

/**
 * called by init3D (gui)
 */
void VglSetViewPort(int width, int height)
{
  trace(DBG_VGL, "setViewPort: w=%d h=%d", width, height);
  glViewport(0, 0, width, height);
  cameraProjection(cam_user.fovy, cam_user.near, cam_user.far);
}

/**
 * get solid handle by its number
 */  
Solid * VGContext::getSolidByNumber(int number)
{
  /* search for solids in solidlist */
  for (Solid *ps = first_solid; ps ; ps = ps->next) {
    if (number == (int32_t)ps) {
      return ps;
    }
  }
  /* search for solids handled by specialRendering */
  for (ObjectList *pl = mobileList; pl ; pl = pl->next) {
    if (pl->pobject && number == (int32_t)pl) {
      trace(DBG_VGL, "getSolidByNumber: number=%d type=%s", number, pl->pobject->name.class_name);
      return pl->pobject->soh;
    }
  }
#if 1 //pd
  for (ObjectList *pl = stillList; pl ; pl = pl->next) {
    if (pl->pobject && number == (int32_t)pl) {
      trace(DBG_VGL, "getSolidByNumber: number=%d type=%s", number, pl->pobject->name.class_name);
      return pl->pobject->soh;
    }
  }
#endif //pd
  error("getSolidByNumber: number=%d not found", number);
  return NULL;	// no solid found in selection
}

/**
 * 3D Selection
 */

#define SELECT_BUF_SIZE (4*1024)

Solid * VglGetBufferSelection(int x, int y)
{
  /* selection mode */
  GLuint selectbuf[SELECT_BUF_SIZE], *psel = selectbuf;
  glSelectBuffer(SELECT_BUF_SIZE, selectbuf);
  glRenderMode(GL_SELECT);
  glInitNames();
  glPushName(0);

  /* cadrage */
  GLint viewport[4];
  glGetIntegerv(GL_VIEWPORT, viewport);
  int width = viewport[2];
  int height = viewport[3];
  float top = cam_user.near * tan(cam_user.fovy * M_PI_180);
  float bottom = -top;
  float ratio = (float) width / (float) height;
  float right = top * ratio;
  float left = -right;
  float gl_mat[16];

  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
   glLoadIdentity();
   glFrustum(left + (x-0.1) * (right-left) / (width-1),
	     left + (x+0.1) * (right-left) / (width-1),
	     top -  (y+0.1) * (top-bottom) / (height-1),
	     top -  (y-0.1) * (top-bottom) / (height-1),
	     cam_user.near, cam_user.far);
   glTranslatef(0, 0, -cam_user.near); // orig=-0.50

   /* eye position */
   gcontext->eyePosition(gl_mat);

   /* draw the solids in the selection buffer */
   gcontext->renderSolids();

   /* we put the normal mode back */
   glMatrixMode(GL_PROJECTION);
  glPopMatrix();

  int hits = glRenderMode(GL_RENDER);

  /* scan the selection buffer, find the nearest hit */
  GLuint zmin = 0xFFFFFFFF; //PB original: zmin=-1 (bad)
  GLint name = -1;
  for (GLint i=0; i < hits; i++) {
    if (gcontext->getSolidByNumber(psel[3]))
      trace(DBG_VGL, "selection1 i=%d zmin=%d psel1=%.3f psel2=%.3f psel3=%d name=%s", i, zmin, (float) psel[1]/0x7fffffff, (float) psel[2]/0x7fffffff, psel[3],  gcontext->getSolidByNumber(psel[3])->object->name.instance_name);
    else
      error("selection2 i=%d zmin=%d psel1=%.2f psel2=%.2f psel3=%d ", i, zmin, (float) psel[1]/0x7fffffff, (float) psel[2]/0x7fffffff, psel[3]);
    if (psel[1] < zmin) { //PB orginal: if (psel[2] < zmin)
      name = psel[3];	// name
      zmin = psel[1]; //PB orginal: zmin = psel[2];  
    }
    psel += 3 + psel[0];
  }
  trace(DBG_VGL, "selection hits=%d zmin=%d name=%d", hits, zmin, name);

  if (name <= 0)
    return NULL;
  return gcontext->getSolidByNumber(name);
}

/**
 * Converts (x,y) screen coordinates into a vector representing
 * the direction of the line from the user's eyes to the point he clicked on
 * vector dir is returned
 */
void VglClickToVector(int x, int y, V3 *dir)
{
  GLdouble x1, y1, z1, x2, y2, z2, modelM[16], projM[16];
  GLint viewport[4];

  glGetIntegerv(GL_VIEWPORT, viewport);
  glGetDoublev(GL_MODELVIEW_MATRIX, modelM);
  glGetDoublev(GL_PROJECTION_MATRIX, projM);
  gluUnProject(x, viewport[3]-y, 0., modelM, projM, viewport, &x1, &y1, &z1);
  gluUnProject(x, viewport[3]-y, 1., modelM, projM, viewport, &x2, &y2, &z2);
  //error("VglClickToVector: x=%d y=%d x1=%.2f y1=%.2f z1=%.2f x2=%.2f y2=%.2f z2=%.2f", x, y, x1, y1, z1, x2, y2, z2);

  dir->v[0] = x1 - x2;
  dir->v[1] = y1 - y2;
  dir->v[2] = z1 - z2;
}
