#include "global.h"
#include "http.h"	// httpOpen
#include "images.h"	// Image
#include "format.h"	// getLoaderByMime
#include "texture.h"


/* local variables */
static TextureCacheEntry *textureList;
static Image *default_image;
static int texture_quality;
static Vpthread_mutex_t texture_mutex, *texc_pmutex = &texture_mutex;
static int texture_number;


void TextureCacheEntry::initCache(const int quality)
{
  textureList = NULL;
  texture_quality = quality;
#if 1	// RGB
  default_image = new Image(TEXTURE_SIZE, TEXTURE_SIZE, IMAGE_RGB, IMAGE_FIX);
#else	// RGBA
  default_image = new Image(TEXTURE_SIZE, TEXTURE_SIZE, IMAGE_RGBA, IMAGE_FIX);
#endif

  uint8_t *pixmap = default_image->pixmap;
  for (int i=0; i < TEXTURE_SIZE*TEXTURE_SIZE; i++) {
#if 1	// RGB
    pixmap[3*i+0] = pixmap[3*i+1] = pixmap[3*i+2] = 0x80;	// grey
#else	// RGBA
    pixmap[4*i+0] = pixmap[4*i+1] = pixmap[4*i+2] = 0x80;	// grey
    pixmap[4*i+3] = 0x00;
#endif
  }
  initMutex(texc_pmutex);
  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
}

void TextureCacheEntry::closeCache()
{
  /* TODO: free textures */
  delete default_image;
}

/**
 * called par Render()
 */
void TextureCacheEntry::updateTextures()
{
  lockMutex(texc_pmutex);

  TextureCacheEntry *texc;
  for (texc = textureList; texc; texc = texc->next) {
    if (texc->image) {
      glBindTexture(GL_TEXTURE_2D, texc->texnum);

      if (texc->image->type != IMAGE_ANIM &&
         (texc->image->xsize != TEXTURE_SIZE ||
          texc->image->ysize != TEXTURE_SIZE)) {
        /* image to resize */
        if (texc->image->format != IMAGE_RGB && texc->image->format != IMAGE_RGBA) {
          error("updateTextures: n=%d f=%d w=%d h=%d t=%d p=%p u=%s", texc->texnum, texc->image->format, texc->image->xsize, texc->image->ysize, texc->image->type, texc->image, texc->texurl);
          continue;
        }
        if (! texc->image->pixmap) {
          error("updateTextures pixmap=null: n=%d w=%d h=%d u=%s", texc->texnum, texc->image->xsize, texc->image->ysize, texc->texurl);
          continue;
        }

        Image *image1 = NULL;
#if 1	// RGB
        if ((image1 = Image::resizeImage(TEXTURE_SIZE, TEXTURE_SIZE, texc->image->format, texc->image->type, texc->image)) == NULL)
          continue;
        glTexImage2D(GL_TEXTURE_2D, 0, TEXTURE_INTERNAL_FORMAT,
                     image1->xsize, image1->ysize, 0,
                     GL_RGB, GL_UNSIGNED_BYTE, image1->pixmap);
#else	// RGBA
        if ((image1 = Image::resizeImage(TEXTURE_SIZE, TEXTURE_SIZE, 4, texc->image->type, texc->image)) == NULL)
          continue;
        glTexImage2D(GL_TEXTURE_2D, 0, 4,
                     image1->xsize, image1->ysize, 0,
                     GL_RGBA, GL_UNSIGNED_BYTE, image1->pixmap); // crash
#endif
        delete image1;
        image1 = NULL;
      }
      else {
        /* image well sized */
#if 1	// RGB
        glTexImage2D(GL_TEXTURE_2D, 0, TEXTURE_INTERNAL_FORMAT,
                     texc->image->xsize, texc->image->ysize, 0,
                     GL_RGB, GL_UNSIGNED_BYTE, texc->image->pixmap);
#else	// RGBA
        glTexImage2D(GL_TEXTURE_2D, 0, 4,
                     texc->image->xsize, texc->image->ysize, 0,
                     GL_RGBA, GL_UNSIGNED_BYTE, texc->image->pixmap); //trouble
#endif
      }

      if (texc->loaded) {
        delete texc->image;
        texc->image = NULL;
      }
    }
  }
  unlockMutex(texc_pmutex);
}

#if 0 //pdimg
int TextureCacheEntry::readImage(char *buf, const int len)
{
  return Read(http, buf, len);	// return read length
}
#else
int readImage(void *_texc, char *buf, const int len)
{
  TextureCacheEntry *texc = (TextureCacheEntry *) _texc;

  return texc->http->Read(buf, len);	// return read length
}
#endif //pdimg

#if 0 //unused
void displayImage(void *_texc, Image *image)
{
  TextureCacheEntry *texc = (TextureCacheEntry *) _texc;

  lockMutex(texc_pmutex);
  texc->image = image;
  unlockMutex(texc_pmutex);
}
#endif //unused

#if 0 //pdhttp
void TextureCacheEntry::textureHttpReader(Http *_http)
#else
void TextureCacheEntry::textureHttpReader(void *_texc, Http *_http) //pd
#endif //pdhttp
{
  TextureCacheEntry *texc = (TextureCacheEntry *) _texc; //pd

  if (! _http) {
    //error("textureHttpReader: http null, url=%s", texc->texurl);
    return; // error while opening
  }
  //pdhttp http = this; //pd
  texc->http = _http; //pd

  trace(DBG_HTTP, "texture: mime=%s texurl=%s", texc->texmime, texc->texurl);

  Image *image = NULL;

#if 0	// EXT
  switch (getLoaderByExt(texc->texurl)) {
#else	// MIME
  switch (getLoaderByMime(texc->texmime)) {
#endif
  case LOADER_GIF:
    image = loadGIF(texc, readImage); break;
  case LOADER_PNG:
    image = loadPNG(texc, readImage); break;
  case LOADER_JPG:
    image = loadJPG(texc, readImage); break;
  case LOADER_PPM:
    image = loadPPM(texc, readImage); break;
  case LOADER_PGM:
    image = loadPGM(texc, readImage); break;
  case LOADER_BMP:
    image = loadBMP(texc, readImage); break;
  case LOADER_SGI:
    image = loadSGI(texc, readImage); break;
  case LOADER_PCX:
    image = loadPCX(texc, readImage); break;
  case LOADER_XPM:
    image = loadXPM(texc, readImage); break;
  case LOADER_NULL:
    break;
  default:
    warning("texture: unknown type mime=%s texurl=%s", texc->texmime, texc->texurl);
    image = NULL;
    break;
  }
  if (! image)
    return; // 0

  lockMutex(texc_pmutex);
  texc->image = image;
  texc->loaded = true;
  unlockMutex(texc_pmutex);
  return; // 1
}

TextureCacheEntry::TextureCacheEntry()
{
  new_texture++;
  texnum = 0;
  http = NULL;
  image = NULL;
  loaded = false;

  next = textureList;
  textureList = this;
}

TextureCacheEntry::~TextureCacheEntry()
{
  del_texture++;
}

int TextureCacheEntry::getFromCache(const char *url)
{
  if (! isprint(*url))
    return 0;

  lockMutex(texc_pmutex);

  TextureCacheEntry *texc;
  for (texc = textureList; texc ; texc = texc->next) {
    if (!strcmp(texc->texurl, url)) {
      /* texture is in the cache */
      unlockMutex(texc_pmutex);
      return texture_number = texc->texnum;
    }
  }

  /* we must download the texture now */

  texc = new TextureCacheEntry(); // new entry in cache

  strcpy(texc->texurl, url);

  glGenTextures(1, (GLuint *) &texc->texnum);
  glBindTexture(GL_TEXTURE_2D, texc->texnum);
#if 1	// RGB
  glTexImage2D(GL_TEXTURE_2D, 0, TEXTURE_INTERNAL_FORMAT,
	       default_image->xsize, default_image->ysize, 0,
	       GL_RGB, GL_UNSIGNED_BYTE, default_image->pixmap);
#else	// RGBA
  glTexImage2D(GL_TEXTURE_2D, 0, 4,
	       default_image->xsize, default_image->ysize, 0,
	       GL_RGBA, GL_UNSIGNED_BYTE, default_image->pixmap);
#endif
 
  if (texture_quality) {
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  }
  else {
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  }
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

  unlockMutex(texc_pmutex);

  trace(DBG_HTTP, "texture: num=%d %s", texc->texnum, url);

  httpOpen(texc->texurl, textureHttpReader, texc, THREAD_BLOCK); 

  texture_number = texc->texnum;	// keep it
  return texc->texnum;
}

TextureCacheEntry * TextureCacheEntry::getEntryByUrl(const char *url)
{
  for (TextureCacheEntry *texc = textureList; texc ; texc = texc->next) {
    if (!strcmp(texc->texurl, url))
      return texc;	// texture is in the cache
  }
  /* we will download the texture latter */
  return NULL;
}

TextureCacheEntry * TextureCacheEntry::getEntryByNumber(const int num)
{
  for (TextureCacheEntry *texc = textureList; texc ; texc = texc->next) {
    if (texc->texnum == num)
      return texc;	// texture is in the cache
  }
  return NULL;
}

int TextureCacheEntry::getCurrentNumber()
{
  return texture_number;
}
