#include "global.h"
#include "wo.h"
#include "world.h"
#include "book.h"

#include "net.h"	// NetObject
#include "vgl.h"	// bbsize
#include "app.h"	// startmp3

#include "sheet.h"      // class Sheet
#include "user.h"	// User (voir bookPull)
#include "http.h"	// httpOpen


const WClass Book::wclass(BOOK_TYPE, "Book", Book::creator);
const uint8_t Book::props = BOOK_PROPS;

static uint16_t oid = 0;


//////////////////////////////////////
// fonctions relatives aux feuilles //
//////////////////////////////////////

/* turnRight */
void sheetTurnRight(Sheet *po, void *d, time_t s, time_t u)
{
  if (po->status == SHEET_RIGHT)
    return;

  po->move.lspeed.v[0] = 0;
  po->move.lspeed.v[1] = 0;
  po->move.lspeed.v[2] = 0;
  po->move.aspeed.v[0] = po->aspeed;
  po->move.aspeed.v[1] = 0;
  po->move.aspeed.v[2] = 0;
  po->initImposedMovement(ABSF(deltaAngle(po->pos.az, po->aright) / po->move.aspeed.v[0]));
  App::startmp3(SHEET_MP3);
  po->status = SHEET_RIGHT;
}

/* turnRight for a heap */
void heapTurnRight(Sheet *po, void *d, time_t s, time_t u)
{
  if (po->status == SHEET_RIGHT)
    return;

  float ttl = ABSF(deltaAngle(po->aleft, po->aright) / po->aspeed);
  float dx = po->soh->bbsize.v[1] * (Cos(po->aleft) + Cos(po->aright));
  float dy = -po->soh->bbsize.v[1] * (Sin(po->aleft) + Sin(po->aright));

  po->move.lspeed.v[0] = dx / ttl;
  po->move.lspeed.v[1] = dy / ttl;
  po->move.lspeed.v[2] = 0;
  po->move.aspeed.v[0] = po->aspeed;
  po->move.aspeed.v[1] = 0;
  po->move.aspeed.v[2] = 0;
  po->initImposedMovement(ttl);
  po->status = SHEET_RIGHT;
}

/* pousser un tas droit (pendant que le tas gauche tourne) */
void heapPushRight(Sheet* po, float dist, void* d, time_t s, time_t u)
{
   if (po->status == SHEET_LEFT)
     return;

   float ttl = ABSF(deltaAngle(po->aleft, po->aright) / po->aspeed);

   po->move.lspeed.v[0] = -dist * Sin(po->aright) / ttl;
   po->move.lspeed.v[1] = dist * Cos(po->aright) / ttl;
   po->move.lspeed.v[2] = 0;
   po->move.aspeed.v[0] = 0;
   po->move.aspeed.v[1] = 0;
   po->move.aspeed.v[2] = 0;

   po->initImposedMovement(ttl);
}

/* turnLeft */
void sheetTurnLeft(Sheet *po, void *d, time_t s, time_t u)
{
  if (po->status == SHEET_LEFT)
    return;

  po->move.lspeed.v[0] = 0;
  po->move.lspeed.v[1] = 0;
  po->move.lspeed.v[2] = 0;
  po->move.aspeed.v[0] = -po->aspeed;
  po->move.aspeed.v[1] = 0;
  po->move.aspeed.v[2] = 0;

  po->initImposedMovement(ABSF(deltaAngle(po->pos.az, po->aleft) / po->move.aspeed.v[0]));
  App::startmp3(SHEET_MP3);
  po->status = SHEET_LEFT;
}

/* turnLeft for a heap */
void heapTurnLeft(Sheet *po, void *d, time_t s, time_t u)
{
  if (po->status == SHEET_LEFT)
    return;

  float ttl = ABSF(deltaAngle(po->pos.az, po->aleft) / po->aspeed);
  float dx = -po->soh->bbsize.v[1] * (Cos(po->aleft) + Cos(po->aright));
  float dy = po->soh->bbsize.v[1] * (Sin(po->aleft) + Sin(po->aright));

  po->move.lspeed.v[0] = dx / ttl;
  po->move.lspeed.v[1] = dy / ttl;
  po->move.lspeed.v[2] = 0;
  po->move.aspeed.v[0] = -po->aspeed;
  po->move.aspeed.v[1] = 0;
  po->move.aspeed.v[2] = 0;
  po->initImposedMovement(ttl);
  po->status = SHEET_LEFT;
}

/* pousser un tas gauche (pendant que le tas droit tourne) */
void heapPushLeft(Sheet* po, float dist, void* d, time_t s, time_t u)
{
   if (po->status == SHEET_RIGHT)
     return;

   float ttl = ABSF(deltaAngle(po->aleft, po->aright) / po->aspeed);

   po->move.lspeed.v[0] = dist * Sin(po->aleft) / ttl;
   po->move.lspeed.v[1] = -dist * Cos(po->aleft) / ttl;
   po->move.lspeed.v[2] = 0;
   po->move.aspeed.v[0] = 0;
   po->move.aspeed.v[1] = 0;
   po->move.aspeed.v[2] = 0;

   po->initImposedMovement(ttl);
}

void sheetPull(Sheet *po, void *d, time_t s, time_t u)
{ }

void sheetRotate(Sheet *po, void *d, time_t s, time_t u)
{ }

/* destroy */
void sheetDestroy(Sheet* po, void *d, time_t s, time_t u)
{
  po->toDelete();	// delete Wobject
}


////////////////////////////////////
// fonctions relatives aux livres //
////////////////////////////////////

/* Downloads urls: filling of the two arrays */
void urlSheetsHttpReader(void *_po, Http *http)
{
  Book *po = (Book *) _po;

  if (! http) {
    error("urlSheetsHttpReader: unable to open http connection");
    return;
  }

  int i;
  char *p;
  char line[URL_LEN];

  httpClearBuf();

  // get the number of pages (first line)
  if (! http->GetLine(line))
    return;
  po->nbSheets = atoi(line) / 2;
  if (po->nbSheets <= 0)
    return;

  po->texTab = (char**) malloc((2 * po->nbSheets + 3) * sizeof(char*)); // + 3 pour les textures de tranche
  po->htmlTab = (char**) malloc((2 * po->nbSheets) * sizeof(char*));

  for (i=0 ; i < 2 * po->nbSheets ; i++) {
    if (! http->GetLine(line))
      return;
    if ((p = strtok(line, " \t")) == NULL) {
      error("urlSheetsHttpReader: bad line %s", line);
      return;
    }
    // on recopie l'url de la texture correspondante
    char* sheet_tex = strdup(p);
    trace(DBG_MAN, "urlSheetsHttpReader: url_tex=%s i=%d", sheet_tex, i);
    po->texTab[i] = strdup(p);
    if ((p = strtok(NULL, " \t"))) {
      // on recopie l'url de la page html correspondante
      char* sheet_html = strdup(p);
      trace(DBG_MAN, "urlSheetsHttpReader: url_html=%s i=%d", sheet_html, i);
      po->htmlTab[i] = strdup(p);
      free(sheet_html);
    }
    else {
      char* sheet_html = strdup(sheet_tex);
      trace(DBG_MAN, "urlSheetsHttpReader: url_html=%s i=%d", sheet_html, i);
      po->htmlTab[i] = strdup(sheet_tex);
      free(sheet_html);
    }
    free(sheet_tex);
  }

  // a ce stade, il doit rester trois urls : celles des textures de tranche
  if (! http->GetLine(line))
    p = NULL;
  if (p == NULL)
    strcpy(line, BOOK_DEF_TEX);
  if ((p = strtok(line, " \t")) == NULL)
    p = BOOK_DEF_TEX;
  trace(DBG_MAN, "urlSheetsHttpReader: tranche i=%d", i);

  po->texTab[i] = strdup(p);
  char *tranche = strdup(p);
  i++;
  if ((p = strtok(NULL, " \t")))
    po->texTab[i] = strdup(p);
  else
    po->texTab[i] = strdup(tranche);
  i++;
  if ((p = strtok(NULL, " \t")))
    po->texTab[i] = strdup(p);
  else
    po->texTab[i] = strdup(tranche);
  free(tranche);
}


////////////////////////////////
// methodes de la classe Book //
////////////////////////////////

/* creation from a file */
WObject * Book::creator(char *l)
{
  return new Book(l);
}

Book::Book(char *l)
{
  enableBehavior(PERSISTENT);

  currentSheet = 0;
  nbSheets = 0;
  stillToTurn = 0;
  stillToTurn2 = 0;
  v = 1;

  l = parseObject(l);
  l = parsePosition(l);
  l = parseGeometry(l);

  while (l) {
    if (!strncmp(l, "aright", 6))
      l = parseFloat(l, &aright, "aright");
    else if (!strncmp(l, "aleft", 5))
      l = parseFloat(l, &aleft, "aleft");
    else if (!strncmp(l, "nbsheets", 8))
      l = parseUInt8(l, &nbSheets, "nbsheets");
    else if (!strncmp(l, "sheetthickness", 14))
      l = parseFloat(l, &sheetThickness, "sheetthickness");
    else if (!strncmp(l, "currentsheet", 12))
      l = parseUInt8(l, &currentSheet, "currentsheet");
    else if (!strncmp(l, "sheets", 6))
      l = parseString(l, sheetsUrl, "sheets");
  }

  sheetWidth = soh->bbsize.v[0] / 2.0;

//////// partie avec url

  if (*sheetsUrl) {
    trace(DBG_MAN, "Book: sheetsUrl=%s", sheetsUrl);

    httpOpen(sheetsUrl, urlSheetsHttpReader, this, THREAD_NO_BLOCK);
  }
  else
    return;

  if (generalActionList[SHEET_CREATE][SHEET_TYPE].method) {
    /////////////////////////////////////////////////////
    // cas d'un livre ferme, rabattu sur le cote droit //
    /***************************************************/
    if (currentSheet == 0) {
      status = BOOK_CLOSED_R;

      // creation du tas droit
      char lr[BUFSIZ];
      char textures[BUFSIZ];
      float dr = 0.5 * sheetThickness * nbSheets; // demi-epaisseur du tas droit

      sprintf(lr, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", pos.x - dr*Sin(aright), pos.y + dr*Cos(aright), pos.z, aright, pos.ax);
      sprintf(lr, "%s solid=\"box,size=%.2f,%.2f,%.2f", lr, sheetWidth, dr, soh->bbsize.v[2]);
      sprintf(textures, ",tex_yn=%s,tex_yp=%s", texTab[0], texTab[2 * nbSheets - 1]); // couverture avant  + couverture arrire
      sprintf(textures, "%s,tex_xp=%s,tex_xn=%s,tex_zp=%s\"", textures, texTab[2*nbSheets], texTab[2*nbSheets+2], texTab[2*nbSheets+1]); // tranche
      strcat(lr, textures);

      generalActionList[SHEET_CREATE][SHEET_TYPE].method(this, lr, SHEET_RIGHT, 0);

      // creation du tas gauche (epaisseur : 1 feuille)
      char lf[BUFSIZ];

      sprintf(lf, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", pos.x, pos.y, pos.z, aright, pos.ax);
      sprintf(lf, "%s solid=\"box,size=%.2f,%.2f,%.2f", lf, sheetWidth, 0.5 * sheetThickness, soh->bbsize.v[2]);
      sprintf(textures, ",tex_yn=%s,tex_yp=%s", texTab[0], texTab[1]); // couverture avant + p 1
      sprintf(textures, "%s,tex_xp=%s,tex_xn=%s,tex_zp=%s\"", textures, texTab[2*nbSheets], texTab[2*nbSheets+2], texTab[2*nbSheets+1]); // tranche
      strcat(lf, textures);

      generalActionList[SHEET_CREATE][SHEET_TYPE].method(this, lf, SHEET_RIGHT, 1);

      // on ne cree pas de feuille volante
      mobileSheet = NULL;
    }

    else {
      //////////////////////////////////////////////////////
      // cas d'un livre ferme, rabattu sur le cote gauche //
      /****************************************************/
      if (currentSheet == nbSheets) {
	status = BOOK_CLOSED_L;

	// creation du tas de droite (epaisseur : 1 feuille)
	char lr[BUFSIZ];
	char textures[BUFSIZ];

	sprintf(lr, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", pos.x, pos.y, pos.z, aleft, pos.ax);
	sprintf(lr, "%s solid=\"box,size=%.2f,%.2f,%.2f", lr, sheetWidth, 0.5 * sheetThickness, soh->bbsize.v[2]);
	sprintf(textures, ",tex_yn=%s,tex_yp=%s", texTab[2 * nbSheets - 2], texTab[2 * nbSheets - 1]);
	sprintf(textures, "%s,tex_xp=%s,tex_xn=%s,tex_zp=%s\"", textures, texTab[2*nbSheets], texTab[2*nbSheets+2], texTab[2*nbSheets+1]); // tranche
	strcat(lr, textures);

	generalActionList[SHEET_CREATE][SHEET_TYPE].method(this, lr, SHEET_LEFT, 0);

	// creation du tas de gauche
	char lf[BUFSIZ];
	float df = 0.5 * sheetThickness * nbSheets; // demi-epaisseur du tas gauche

	sprintf(lf, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", pos.x + df*Sin(aleft), pos.y - df*Cos(aleft), pos.z, aleft, pos.ax);
	sprintf(lf, "%s solid=\"box,size=%.2f,%.2f,%.2f", lf, sheetWidth, df, soh->bbsize.v[2]);
	sprintf(textures, ",tex_yn=%s,tex_yp=%s", texTab[0], texTab[2 * nbSheets - 2]); // couverture avant + couverture arriere
	sprintf(textures, "%s,tex_xp=%s,tex_xn=%s,tex_zp=%s\"", textures, texTab[2*nbSheets], texTab[2*nbSheets+2], texTab[2*nbSheets+1]); // tranche
	strcat(lf, textures);

	generalActionList[SHEET_CREATE][SHEET_TYPE].method(this, lf, SHEET_LEFT, 1);

	// on ne cree pas de feuille volante
	mobileSheet = NULL;
      }

      else {
	///////////////////////////
	// cas d'un livre ouvert //
	/*************************/
	status = BOOK_OPENED;

	// creation du tas droit
	char lr[BUFSIZ];
	char textures[BUFSIZ];
    	float dr = 0.5 * sheetThickness * (nbSheets - currentSheet); // demi-epaisseur du tas droit
	float dr2 = dr + sheetThickness ; // valeur augmentee pour pb texture

	sprintf(lr, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", pos.x - dr2*Sin(aright), pos.y + dr2*Cos(aright), pos.z, aright, pos.ax);
	sprintf(lr, "%s solid=\"box,size=%.2f,%.2f,%.2f", lr, sheetWidth, dr, soh->bbsize.v[2]);
	sprintf(textures, ",tex_yn=%s,tex_yp=%s", texTab[2 * currentSheet], texTab[2 * nbSheets - 1]);
	sprintf(textures, "%s,tex_xp=%s,tex_xn=%s,tex_zp=%s\"", textures, texTab[2*nbSheets], texTab[2*nbSheets+2], texTab[2*nbSheets+1]);
	strcat(lr, textures);

	generalActionList[SHEET_CREATE][SHEET_TYPE].method(this, lr, SHEET_RIGHT, 0);

	// creation du tas gauche
	char lf[BUFSIZ];
	float df = 0.5 * sheetThickness * currentSheet; // demi-epaisseur du tas gauche

	sprintf(lf, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", pos.x + df*Sin(aleft), pos.y - df*Cos(aleft), pos.z, aleft, pos.ax);/***/
	sprintf(lf, "%s solid=\"box,size=%.2f,%.2f,%.2f", lf, sheetWidth, df, soh->bbsize.v[2]);
	sprintf(textures, ",tex_yn=%s,tex_yp=%s", texTab[0], texTab[2 * currentSheet - 1]);
	sprintf(textures, "%s,tex_xp=%s,tex_xn=%s,tex_zp=%s\"", textures, texTab[2*nbSheets], texTab[2*nbSheets+2], texTab[2*nbSheets+1]);
	strcat(lf, textures);

	generalActionList[SHEET_CREATE][SHEET_TYPE].method(this, lf, SHEET_LEFT, 1);

	// creation de la feuille libre
	char lm[BUFSIZ];

	sprintf(lm, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", pos.x, pos.y, pos.z, aright, pos.ax);
	sprintf(lm, "%s solid=\"box,size=%.2f,%.2f,%.2f", lm, sheetWidth, 0.5 * sheetThickness, soh->bbsize.v[2]);
	sprintf(textures, ",tex_yn=%s,tex_yp=%s\"", texTab[2 * currentSheet], texTab[2 * currentSheet + 1]);
	strcat(lm, textures);

	generalActionList[SHEET_CREATE][SHEET_TYPE].method(this, lm, SHEET_RIGHT, 2);
      }
    }
  }

  enableBehavior(COLLIDE_NEVER); // pas de collisions
  initializeObject(LIST_MOBILE);
  setMaxLasting(BOOK_LASTING);
  createPermanentNetObject(props, ++oid);
}

bool Book::isMoving()
{
  return (move.ttl > 0.0005);
}

void Book::changePosition(float lasting)
{
  pos.x  += lasting * move.lspeed.v[0];
  pos.y  += lasting * move.lspeed.v[1];
  pos.z  += lasting * move.lspeed.v[2];
  pos.az += lasting * move.aspeed.v[0];
}

void Book::updateTime(time_t sec, time_t usec, float *lasting)
{
  *lasting = diffTime(sec, usec);
  if (*lasting < move.ttl) {
    move.ttl -= *lasting;
    move.sec = sec;
    move.usec = usec;
  }
  else {
    *lasting = move.ttl;
    stopImposedMovement();
  }
}

bool Book::updateToNetwork(const Pos &oldpos)
{
  bool change = false;

  if ((pos.x != oldpos.x) || (pos.y != oldpos.y)) {
    noh->declareObjDelta(BOOK_PROPXY);
    change = true;
  }
  if (pos.z != oldpos.z) {
    noh->declareObjDelta(BOOK_PROPZ);
    change = true;
  }
  if (pos.az != oldpos.az) {
    noh->declareObjDelta(BOOK_PROPAZ);
    change = true;
  }
  return change;
}

void Book::whenIntersect(WObject *pcur, WObject *pold)
{
  if (status == BOOK_OPENED && projectMovementOnObject(pcur->pos, pold->pos, pos))
    pcur->updateObject(pold->pos);
  else
    pold->copyPositionAndBB(pcur);
}

void Book::quit()
{
  oid = 0;
}

/* next page */
void bookNextPage(Book *po, void *d, time_t s, time_t u)
{
  if (po->status == BOOK_CLOSED_L)
    return;

  if  (po->currentSheet == po->nbSheets - 1) {
    // cas particulier 1 : il ne reste qu'une feuille a droite
    //////////////////////////////////////////////////////////

    // d'abord augmenter le tas de gauche (+ leger recul pour pb texture)
    char lf[BUFSIZ];
    char textures[BUFSIZ];
    float df = 0.5 * po->sheetThickness * po->nbSheets; // demi-epaisseur du nouveau tas gauche
    float df2 = df + 2.5*po->sheetThickness; // valeur augmentee pour pb texture

    sprintf(lf, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", po->pos.x + df2*Sin(po->aleft), po->pos.y - df2*Cos(po->aleft), po->pos.z, po->aleft, po->pos.ax);
    sprintf(lf, "%s solid=\"box,size=%.2f,%.2f,%.2f", lf, po->sheetWidth, df, po->soh->bbsize.v[2]);
    sprintf(textures, ",tex_yn=%s,tex_yn=%s", po->texTab[0], po->texTab[2 * po->nbSheets - 3]);
    sprintf(textures, "%s,tex_yp=%s,tex_xp=%s,tex_zp=%s\"", textures, po->texTab[2*po->nbSheets], po->texTab[2*po->nbSheets+2], po->texTab[2*po->nbSheets+1]);
    strcat(lf, textures);

    sheetDestroy(po->leftSide, d, s, u);
    generalActionList[SHEET_CREATE][SHEET_TYPE].method(po, lf, SHEET_LEFT, 1);

    // destruction de la mobileSheet
    sheetDestroy(po->mobileSheet, d, s, u);
    po->mobileSheet = NULL;

    // rotation de la partie droite
    sheetTurnLeft(po->rightSide, d, s, u);

    po->currentSheet++;
    po->status = BOOK_CLOSED_L;
  }

  else {
    if  (po->currentSheet == 0) {
      // BOOK_CLOSED_R
      // cas particulier 2: il n'y a pas encore de feuile a gauche, et la mobileSheet est a NULL
      //////////////////////////////////////////////////////////
      // on cree la mobileSheet mais on fait tourner la partie gauche (resumee a 1 feuille)

      /* create mobile page first */
      char lm[BUFSIZ];
      char textures[BUFSIZ];

      sprintf(lm, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", po->pos.x, po->pos.y, po->pos.z, po->aright, po->pos.ax);
      sprintf(lm, "%s solid=\"box,size=%.2f,%.2f,%.2f", lm, po->sheetWidth, 0.5 * po->sheetThickness, po->soh->bbsize.v[2]);
      sprintf(textures, ",tex_yn=%s,tex_xn=%s\"", po->texTab[2], po->texTab[3]);
      strcat(lm, textures);

      generalActionList[SHEET_CREATE][SHEET_TYPE].method(po, lm, SHEET_LEFT, 2);

      // remplacer le tas droit //
      char lr[BUFSIZ];
      float dr = 0.5 * po->sheetThickness * (po->nbSheets - 1); // demi-epaisseur du tas droit
      float dr2 = dr + po->sheetThickness ; // valeur augmentee pour pb texture

      sprintf(lr, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", po->pos.x - dr2*Sin(po->aright), po->pos.y + dr2*Cos(po->aright), po->pos.z, po->aright, po->pos.ax);
      sprintf(lr, "%s solid=\"box,size=%.2f,%.2f,%.2f", lr, po->sheetWidth, dr, po->soh->bbsize.v[2]);
      sprintf(textures, ",tex_yn=%s,tex_xn=%s", po->texTab[2], po->texTab[2 * po->nbSheets - 1]); //  p 2 + couverture arrire
      sprintf(textures, "%s,tex_yp=%s,tex_xp=%s,tex_zp=%s\"", textures, po->texTab[2*po->nbSheets], po->texTab[2*po->nbSheets+2], po->texTab[2*po->nbSheets+1]);
      strcat(lr, textures);

      sheetDestroy(po->rightSide, d, s, u);
      generalActionList[SHEET_CREATE][SHEET_TYPE].method(po, lr, SHEET_RIGHT, 0);

      // remplacer le tas gauche s'il est trop gros (si on sort d'un bookClose)
      if (po->leftSide->soh->bbsize.v[1] > 0.7 * po->sheetThickness) {
	char lf[BUFSIZ];

	sprintf(lf, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", po->pos.x, po->pos.y, po->pos.z, po->aright, po->pos.ax);
	sprintf(lf, "%s solid=\"box,size=%.2f,%.2f,%.2f", lf, po->sheetWidth, 0.5 * po->sheetThickness, po->soh->bbsize.v[2]);
	sprintf(textures, ",tex_yn=%s,tex_xn=%s", po->texTab[0], po->texTab[1]); // couverture avant + p 1
	sprintf(textures, "%s,tex_yp=%s,tex_xp=%s,tex_zp=%s\"", textures, po->texTab[2*po->nbSheets], po->texTab[2*po->nbSheets+2], po->texTab[2*po->nbSheets+1]);
	strcat(lf, textures);

	sheetDestroy(po->leftSide, d, s, u);
	generalActionList[SHEET_CREATE][SHEET_TYPE].method(po, lf, SHEET_RIGHT, 1);
      }

      // et enfin faire tourner le "tas" (1 feuille) gauche //
      sheetTurnLeft(po->leftSide, d, s, u);

      po->currentSheet++;
      po->status = BOOK_OPENED;
    }

    else {
      //////////////// cas general ////////////////////

      ///////////////////////////////////////////////
      // remplacement du tas de feuilles de gauche //

      char lf[BUFSIZ];
      char textures[BUFSIZ];
      float df = 0.5 * po->sheetThickness * (po->currentSheet+1) ; // demi-epaisseur du nouveau tas gauche
      float df2 = df + po->sheetThickness ; // valeur augmentee pour pb texture

      sprintf(lf, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", po->pos.x + df2*Sin(po->aleft), po->pos.y - df2*Cos(po->aleft), po->pos.z, po->aleft, po->pos.ax);
      sprintf(lf, "%s solid=\"box,size=%.2f,%.2f,%.2f", lf, po->sheetWidth, df, po->soh->bbsize.v[2]);
      sprintf(textures, ",tex_yn=%s,tex_xn=%s", po->texTab[0], po->texTab[2 * po->currentSheet - 1]);
      sprintf(textures, "%s,tex_yp=%s,tex_xp=%s,tex_zp=%s\"", textures, po->texTab[2*po->nbSheets], po->texTab[2*po->nbSheets+2], po->texTab[2*po->nbSheets+1]);
      strcat(lf, textures);

      sheetDestroy(po->leftSide, d, s, u);
      generalActionList[SHEET_CREATE][SHEET_TYPE].method(po, lf, SHEET_LEFT, 1);

      //////////////////////////////////////////////////
      // remplacement de la page mobile si besoin est //

      if (po->mobileSheet->status == SHEET_LEFT) {
	char lm[BUFSIZ];
	char textures[BUFSIZ];

	sprintf(lm, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", po->pos.x, po->pos.y, po->pos.z, po->aright, po->pos.ax);
	sprintf(lm, "%s solid=\"box,size=%.2f,%.2f,%.2f", lm, po->sheetWidth, 0.5 * po->sheetThickness, po->soh->bbsize.v[2]);
	sprintf(textures, ",tex_yn=%s,tex_xn=%s\"", po->texTab[2 * po->currentSheet], po->texTab[2 * po->currentSheet + 1]);
	strcat(lm, textures);

	sheetDestroy(po->mobileSheet, d, s, u);
	generalActionList[SHEET_CREATE][SHEET_TYPE].method(po, lm, SHEET_RIGHT, 2);
      }

      ///////////////////////////////////////////////
      // remplacement du tas de feuilles de droite //

      char lr[BUFSIZ];
      float dr = 0.5 * po->sheetThickness * (po->nbSheets - (po->currentSheet + 1)); // demi-epaisseur du tas droit
      float dr2 = dr + po->sheetThickness ; // valeur augmentee pour pb texture

      sprintf(lr, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", po->pos.x - dr2*Sin(po->aright), po->pos.y + dr2*Cos(po->aright), po->pos.z, po->aright, po->pos.ax);
      sprintf(lr, "%s solid=\"box,size=%.2f,%.2f,%.2f", lr, po->sheetWidth, dr, po->soh->bbsize.v[2]);
      sprintf(textures, ",tex_yn=%s,tex_xn=%s", po->texTab[2 * (po->currentSheet +1)], po->texTab[2 * po->nbSheets - 1]);
      sprintf(textures, "%s,tex_yp=%s,tex_xp=%s,tex_zp=%s\"", textures, po->texTab[2*po->nbSheets], po->texTab[2*po->nbSheets+2], po->texTab[2*po->nbSheets+1]);
      strcat(lr, textures);

      sheetDestroy(po->rightSide, d, s, u);
      generalActionList[SHEET_CREATE][SHEET_TYPE].method(po, lr, SHEET_RIGHT, 0);

      ///////////////////////////////////
      // rotation de la feuille mobile //

      sheetTurnLeft(po->mobileSheet, d, s, u);

      po->currentSheet++; // le tas de gauche compte maintenant une page de plus
    }
  }
}

/* previous page */
void bookPrevPage(Book *po, void *d, time_t s, time_t u)
{
  if (po->status == BOOK_CLOSED_R)
    return;

  if  (po->currentSheet == 1) {
    // cas particulier 1 : il ne reste qu'une page a gauche
    ///////////////////////////////////////////////////////

    // d'abord augmenter le tas de droite (+ leger recul pour pb texture)
    char lr[BUFSIZ];
    char textures[BUFSIZ];
    float dr = 0.5 * po->sheetThickness * po->nbSheets ; // demi-epaisseur du tas droit
    float dr2 = dr + 2.5*po->sheetThickness ; // valeur augmentee pour pb texture

    sprintf(lr, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", po->pos.x - dr2*Sin(po->aright), po->pos.y + dr2*Cos(po->aright), po->pos.z, po->aright, po->pos.ax);
    sprintf(lr, "%s solid=\"box,size=%.2f,%.2f,%.2f", lr, po->sheetWidth, dr, po->soh->bbsize.v[2]);
    sprintf(textures, ",tex_yn=%s,tex_xn=%s", po->texTab[2], po->texTab[2 * po->nbSheets - 1]);
    sprintf(textures, "%s,tex_yp=%s,tex_xp=%s,tex_zp=%s\"", textures, po->texTab[2*po->nbSheets], po->texTab[2*po->nbSheets+2], po->texTab[2*po->nbSheets+1]);
    strcat(lr, textures);

    sheetDestroy(po->rightSide, d, s, u);
    generalActionList[SHEET_CREATE][SHEET_TYPE].method(po, lr, SHEET_RIGHT, 0);

    // destruction de la mobileSheet
    sheetDestroy(po->mobileSheet, d, s, u);
    po->mobileSheet = NULL;

    // rotation de la partie gauche
    sheetTurnRight(po->leftSide, d, s, u);

    po->currentSheet--;
    po->status = BOOK_CLOSED_R;
  }

  else {
    if  (po->currentSheet == po->nbSheets) {
      // BOOK_CLOSED_L
      // cas particulier 2 : il n'y a pas encore de feuile a droite
      /////////////////////////////////////////////////////////////
      // => on cree la mobileSheet mais on fait tourner la partie droite (resumee a 1 feuille)

      // d'abord creer la page mobile //
      char lm[BUFSIZ];
      char textures[BUFSIZ];

      sprintf(lm, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", po->pos.x, po->pos.y, po->pos.z, po->aleft, po->pos.ax);
      sprintf(lm, "%s solid=\"box,size=%.2f,%.2f,%.2f", lm, po->sheetWidth, 0.5 * po->sheetThickness, po->soh->bbsize.v[2]);
      sprintf(textures, ",tex_yn=%s,tex_xn=%s\"", po->texTab[2 * po->nbSheets - 4], po->texTab[2 * po->nbSheets - 3]); //  p 2N-4 + p 2N-3
      strcat(lm, textures);

      generalActionList[SHEET_CREATE][SHEET_TYPE].method(po, lm, SHEET_LEFT, 2);

      // remplacer le tas gauche //
      char lf[BUFSIZ];
      float df = 0.5 * po->sheetThickness * (po->nbSheets - 1); // epaisseur du nouveau tas gauche
      float df2 = df + po->sheetThickness ; // valeur augmentee pour pb texture

      sprintf(lf, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", po->pos.x + df2*Sin(po->aleft), po->pos.y - df2*Cos(po->aleft), po->pos.z, po->aleft, po->pos.ax);
      sprintf(lf, "%s solid=\"box,size=%.2f,%.2f,%.2f", lf, po->sheetWidth, df, po->soh->bbsize.v[2]);
      sprintf(textures, ",tex_yn=%s,tex_xn=%s", po->texTab[0], po->texTab[2 * po->nbSheets - 3]); //  p 0 + p 2N-3
      sprintf(textures, "%s,tex_yp=%s,tex_xp=%s,tex_zp=%s\"", textures, po->texTab[2*po->nbSheets], po->texTab[2*po->nbSheets+2], po->texTab[2*po->nbSheets+1]);
      strcat(lf, textures);

      sheetDestroy(po->leftSide, d, s, u);
      generalActionList[SHEET_CREATE][SHEET_TYPE].method(po, lf, SHEET_LEFT, 1);

      // remplacer le tas droit s'il est trop gros  //
      if (po->rightSide->soh->bbsize.v[1] > 0.7 * po->sheetThickness) {
	char lr[BUFSIZ];

	sprintf(lr, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", po->pos.x, po->pos.y, po->pos.z, po->aright, po->pos.ax);
	sprintf(lr, "%s solid=\"box,size=%.2f,%.2f,%.2f", lr, po->sheetWidth, 0.5 * po->sheetThickness, po->soh->bbsize.v[2]);
	sprintf(textures, ",tex_yn=%s,tex_xn=%s", po->texTab[2 * po->nbSheets - 2], po->texTab[2 * po->nbSheets - 1]);
	sprintf(textures, "%s,tex_yp=%s,tex_xp=%s,tex_zp=%s\"", textures, po->texTab[2*po->nbSheets], po->texTab[2*po->nbSheets+2], po->texTab[2*po->nbSheets+1]);
	strcat(lr, textures);

	sheetDestroy(po->rightSide, d, s, u);
	generalActionList[SHEET_CREATE][SHEET_TYPE].method(po, lr, SHEET_LEFT, 0);
      }

      // et enfin faire tourner le "tas" (1 feuille) droit //
      sheetTurnRight(po->rightSide, d, s, u);

      po->currentSheet--;
      po->status = BOOK_OPENED;
    }

    else {
      //////////////// cas general ////////////////////

      ///////////////////////////////////////////////
      // remplacement du tas de feuilles de droite //

      char lr[BUFSIZ];
      char textures[BUFSIZ];

      float dr = 0.5 * po->sheetThickness * (po->nbSheets - po->currentSheet + 1); // demi-epaisseur du tas droit
      float dr2 = dr + po->sheetThickness ; // valeur augmentee pour pb texture

      sprintf(lr, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", po->pos.x - dr2*Sin(po->aright), po->pos.y + dr2*Cos(po->aright), po->pos.z, po->aright, po->pos.ax);
      sprintf(lr, "%s solid=\"box,size=%.2f,%.2f,%.2f", lr, po->sheetWidth, dr, po->soh->bbsize.v[2]);
      sprintf(textures, ",tex_yn=%s,tex_xn=%s", po->texTab[2 * po->currentSheet], po->texTab[2 * po->nbSheets - 1]);
      sprintf(textures, "%s,tex_yp=%s,tex_xp=%s,tex_zp=%s\"", textures, po->texTab[2*po->nbSheets], po->texTab[2*po->nbSheets+2], po->texTab[2*po->nbSheets+1]);
      strcat(lr, textures);

      sheetDestroy(po->rightSide, d, s, u);
      generalActionList[SHEET_CREATE][SHEET_TYPE].method(po, lr, SHEET_RIGHT, 0);

      //////////////////////////////////////////////////
      // remplacement de la page mobile si besoin est //

      if (po->mobileSheet->status == SHEET_RIGHT) {
	char lm[BUFSIZ];
	char textures[BUFSIZ];

	sprintf(lm, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", po->pos.x, po->pos.y, po->pos.z, po->aleft, po->pos.ax);
	sprintf(lm, "%s solid=\"box,size=%.2f,%.2f,%.2f", lm, po->sheetWidth, 0.5 * po->sheetThickness, po->soh->bbsize.v[2]);
	sprintf(textures, ",tex_yn=%s,tex_xn=%s\"", po->texTab[2 * (po->currentSheet-1)], po->texTab[2 * po->currentSheet - 1]);
	strcat(lm, textures);

	sheetDestroy(po->mobileSheet, d, s, u);
	generalActionList[SHEET_CREATE][SHEET_TYPE].method(po, lm, SHEET_LEFT, 2);
      }

      ////////////////////////////////////////////////
      // remplacement du tas de feuilles de gauche //

      char lf[BUFSIZ];
      float df = 0.5 * po->sheetThickness * (po->currentSheet -1) ; // epaisseur du nouveau tas gauche
      float df2 = df + po->sheetThickness ; // valeur augmentee pour pb texture

      sprintf(lf, "pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\"", po->pos.x + df2*Sin(po->aleft), po->pos.y - df2*Cos(po->aleft), po->pos.z, po->aleft, po->pos.ax);
      sprintf(lf, "%s solid=\"box,size=%.2f,%.2f,%.2f", lf, po->sheetWidth, df, po->soh->bbsize.v[2]);
      sprintf(textures, ",tex_yn=%s,tex_xn=%s", po->texTab[0], po->texTab[2 * po->currentSheet - 3]);
      sprintf(textures, "%s,tex_yp=%s,tex_xp=%s,tex_zp=%s\"", textures, po->texTab[2*po->nbSheets], po->texTab[2*po->nbSheets+2], po->texTab[2*po->nbSheets+1]);
      strcat(lf, textures);

      sheetDestroy(po->leftSide, d, s, u);
      generalActionList[SHEET_CREATE][SHEET_TYPE].method(po, lf, SHEET_LEFT, 1);

      ///////////////////////////////////
      // rotation de la feuille mobile //

      sheetTurnRight(po->mobileSheet, d, s, u);

      po->currentSheet--;  // le tas de gauche compte maintenant une page de moins
    }
  }
}

/* open */
void bookOpen(Book *po, void *d, time_t s, time_t u)
{
  if (po->status == BOOK_OPENED)
    return;

  if (po->status == BOOK_CLOSED_R) {
    po->v = 1;
    bookNextPage(po, d, s, u);
  }
  if (po->status == BOOK_CLOSED_L) {
    po->v = 1;
    bookPrevPage(po, d, s, u);
  }
}

/* close */
void bookClose(Book *po, void *d, time_t s, time_t u)
{
   if ( (po->status == BOOK_CLOSED_R) || (po->status == BOOK_CLOSED_L) )
     return;

   // destruction de la mobileSheet
   sheetDestroy(po->mobileSheet, d, s, u);
   po->mobileSheet = NULL;

   // on ferme toujours le livre sur son cote droit //
   ///////////////////////////////////////////////////

   heapTurnRight(po->leftSide, d, s, u);
   float dist = po->sheetThickness * po->currentSheet;
   heapPushRight(po->rightSide, dist, d, s, u);

   po->currentSheet = 0;
   po->status = BOOK_CLOSED_R;
}

/* methode auxiliaire pour touner 10 pages d'un coup */
void  Book::forward(void *d, time_t s, time_t u)
{
  v = 10;
  //pd v = 8.0 / 9.0 * stillToTurn + 10.0 / 9.0;  // vitesse variable
  if (mobileSheet != NULL)
    mobileSheet->aspeed = v * SHEET_ASPEED;

  bookNextPage(this, d, s, u);

  v=1;
  if (mobileSheet != NULL)
    mobileSheet->aspeed = v * SHEET_ASPEED;
}

/* methode auxiliaire pour touner 10 pages d'un coup */
void  Book::backward(void *d, time_t s, time_t u)
{
  v = 10;
  //pd v = 8.0 / 9.0 * stillToTurn2 + 10.0 / 9.0; // vitesse variable
  if (mobileSheet != NULL)
    mobileSheet->aspeed = v * SHEET_ASPEED;

  bookPrevPage(this, d, s, u);

  v=1;
  if (mobileSheet != NULL)
    mobileSheet->aspeed = v * SHEET_ASPEED;
}

/* forward : 10 pages en avant */
void bookForward(Book *po, void *d, time_t s, time_t u)
{
  if (po->status != BOOK_OPENED)
    return;

  if (po->currentSheet + 10 > po->nbSheets) {
    po->stillToTurn = po->nbSheets - po->currentSheet - 1;
    po->forward(d, s, u);
  }
  else {
    po->stillToTurn=9;
    po->forward(d, s, u);
  }
}

/* backward : 10 pages en arriere */
void bookBackward(Book *po, void *d, time_t s, time_t u)
{
  if (po->status != BOOK_OPENED)
    return;

  if (po->currentSheet - 10 < 0) {
    po->stillToTurn2 = po->currentSheet - 1;
    po->backward(d, s, u);
  }
  else {
    po->stillToTurn2=9;
    po->backward(d, s, u);
  }
}

/* look right : afficher la page de droite dans un navigateur */
void bookLookRight(Book *po, void *d, time_t s, time_t u)
{
  if (po->currentSheet < po->nbSheets)
    App::startbrowser(po->htmlTab[2*po->currentSheet]);
}

/* look left : afficher la page de gauche dans un navigateur */
void bookLookLeft(Book *po, void *d, time_t s, time_t u)
{
  if (po->currentSheet > 0)
    App::startbrowser(po->htmlTab[2*po->currentSheet-1]);
}

/* approach : approcher le livre de l'avatar */
void bookApproach(Book *po, void *d, time_t s, time_t u)
{
  float dx = 0.7 * (worlds->plocaluser->pos.x - po->pos.x);	// pas trop pres
  float dy = 0.7 * (worlds->plocaluser->pos.y - po->pos.y);	// pas trop pres
  float dz = worlds->plocaluser->pos.z + 0.5 * USER_DEFAULTHEIGHT - po->pos.z;

  float aright0 = po->aright;
  float aleft0 = po->aleft;

  float aend = worlds->plocaluser->pos.az - 1.57; // remarque : aend = "aright_fin"
  float da = deltaAngle(aend, aright0);
  if (da < - 3.14)
    da += 6.28;
  if (da > 3.14)
    da -= 6.28;

  float ttl = ABSF(dx / SHEET_LSPEED);
  if (ttl == 0)
    ttl = ABSF(dy / SHEET_LSPEED);

  if (ttl > 0.001) {
    float vz = dz / ttl;
    float va = da / ttl;

    po->move.lspeed.v[0] = dx / ttl;
    po->move.lspeed.v[1] = dy / ttl;
    po->move.lspeed.v[2] = vz;
    po->move.aspeed.v[0] = va;
    po->move.aspeed.v[1] = 0;
    po->move.aspeed.v[2] = 0;

    po->aright = aend;
    po->aleft += da;

    float dxright = dx + po->rightSide->soh->bbsize.v[1] * (Sin(aright0) - Sin(aend));
    float dyright = dy + po->rightSide->soh->bbsize.v[1] * (Cos(aend) - Cos(aright0));

    po->rightSide->move.lspeed.v[0] = dxright / ttl;
    po->rightSide->move.lspeed.v[1] = dyright / ttl;
    po->rightSide->move.lspeed.v[2] = vz;
    po->rightSide->move.aspeed.v[0] = va;
    po->rightSide->move.aspeed.v[1] = 0;
    po->rightSide->move.aspeed.v[2] = 0;

    po->rightSide->aright = aend;
    po->rightSide->aleft += da;

    float dxleft = dx + po->leftSide->soh->bbsize.v[1] * (Sin(aleft0 + da) - Sin(aleft0));
    float dyleft = dy + po->leftSide->soh->bbsize.v[1] * (Cos(aleft0) - Cos(aleft0 + da));

    po->leftSide->move.lspeed.v[0] = dxleft / ttl;
    po->leftSide->move.lspeed.v[1] = dyleft / ttl;
    po->leftSide->move.lspeed.v[2] = vz;
    po->leftSide->move.aspeed.v[0] = va;
    po->leftSide->move.aspeed.v[1] = 0;
    po->leftSide->move.aspeed.v[2] = 0;

    po->leftSide->aright = aend;
    po->leftSide->aleft += da;

    if (po->mobileSheet != NULL) {
      po->mobileSheet->move.lspeed.v[0] = dx / ttl;
      po->mobileSheet->move.lspeed.v[1] = dy / ttl;
      po->mobileSheet->move.lspeed.v[2] = vz;
      po->mobileSheet->move.aspeed.v[0] = va;
      po->mobileSheet->move.aspeed.v[1] = 0;
      po->mobileSheet->move.aspeed.v[2] = 0;

      po->mobileSheet->aright = aend;
      po->mobileSheet->aleft += da;

      po->mobileSheet->initImposedMovement(ttl);
    }

    po->leftSide->initImposedMovement(ttl);
    po->rightSide->initImposedMovement(ttl);
    po->initImposedMovement(ttl);
  }
}

void bookInitFuncList(void)
{
  getPropertyFunc[BOOK_PROPXY][BOOK_TYPE].pf = WO_PAYLOAD get_xy;
  getPropertyFunc[BOOK_PROPZ][BOOK_TYPE].pf = WO_PAYLOAD get_z;
  getPropertyFunc[BOOK_PROPAZ][BOOK_TYPE].pf = WO_PAYLOAD get_az;
  getPropertyFunc[BOOK_PROPHNAME][BOOK_TYPE].pf = WO_PAYLOAD get_hname;

  putPropertyFunc[BOOK_PROPXY][BOOK_TYPE].pf = WO_PAYLOAD put_xy;
  putPropertyFunc[BOOK_PROPZ][BOOK_TYPE].pf = WO_PAYLOAD put_z;
  putPropertyFunc[BOOK_PROPAZ][BOOK_TYPE].pf = WO_PAYLOAD put_az;
  putPropertyFunc[BOOK_PROPHNAME][BOOK_TYPE].pf = WO_PAYLOAD put_hname;

  setActionFunc(BOOK_TYPE, BOOK_OPEN, WO_ACTION bookOpen, "Open");
  setActionFunc(BOOK_TYPE, BOOK_CLOSE, WO_ACTION bookClose, "Close");
  setActionFunc(BOOK_TYPE, BOOK_PULL, WO_ACTION bookApproach, "Approach");
  setActionFunc(BOOK_TYPE, BOOK_NEXT, WO_ACTION bookNextPage, "NextPage");
  setActionFunc(BOOK_TYPE, BOOK_PREVIOUS, WO_ACTION bookPrevPage, "PrevPage");
  setActionFunc(BOOK_TYPE, BOOK_FFWD, WO_ACTION bookForward, "p+10");
  setActionFunc(BOOK_TYPE, BOOK_REWIND, WO_ACTION bookBackward, "p-10");
  setActionFunc(BOOK_TYPE, BOOK_LOOKL, WO_ACTION bookLookLeft, "LookLeft");
  setActionFunc(BOOK_TYPE, BOOK_LOOKR, WO_ACTION bookLookRight, "LookRight");
}
