#include "ubit/ubit.hpp"
#include "global.h"
#include "wo.h"
#include "world.h"	// worlds
#include "user.h"	// User
#include "move.h"	// GRAVITY
#include "wall.h"	// Wall
#include "format.h"	// icons
#include "icon.h"

#include "http.h"	// url2filename
#include "texture.h"	// TextureCacheEntry

#include "net.h"	// NetObject
#include "vgl.h"	// Solid
#include "gui.h"	// clearObjectBar
#include "app.h"	// startbrowser


const WClass Icon::wclass(ICON_TYPE, "Icon", NULL, Icon::replicator);
const uint8_t Icon::props = ICON_PROPS;


void Icon::updateTime(time_t sec, time_t usec, float *lasting)
{
  *lasting = diffTime(sec, usec);
  if (*lasting < move.ttl) {
    move.ttl -= *lasting;
    move.sec = sec;
    move.usec = usec;
  }
  else {
    *lasting = move.ttl;
    stopImposedMovement();
    ttl = ICON_TTL;
  }
}

bool Icon::isMoving()
{
  if (taken)
    return true;
  return (move.ttl > 0.0005);
}

/* sollicited movements */
void Icon::changePosition(float lasting)
{
  if (! taken) {
    float ratio = MAX(getRate() / 40., 1);
    pos.x  += lasting * move.lspeed.v[0] * ratio;
    pos.y  += lasting * move.lspeed.v[1] * ratio;
    pos.z  += lasting * move.lspeed.v[2] * ratio;
    pos.az += lasting * move.aspeed.v[0] * ratio;
    pos.ax += lasting * move.aspeed.v[2] * ratio;

#if 0
    // reduce speed
    move.lspeed.v[0] /= 1.02;
    move.lspeed.v[1] /= 1.02;
    move.lspeed.v[2] /= 1.02;
#endif
  }
  else {
    float offx = 0.4;
    pos.x = worlds->plocaluser->pos.x + offx * Cos(worlds->plocaluser->pos.az);
    pos.y = worlds->plocaluser->pos.y + offx * Sin(worlds->plocaluser->pos.az);
    pos.z = worlds->plocaluser->pos.z + 0.6;
    pos.az = worlds->plocaluser->pos.az + M_PI_2;
  }
}

/** permanent movements */
void Icon::changePermanent(float lasting)
{
  pos.z -= lasting * GRAVITY / 2;
}

/* creation: this method is invisible: called by GUI::putIcon */
void iconCreate(User *pu, void *data, time_t sec, time_t usec)
{
  new Icon(pu, data);
}

void Icon::setDefaults()
{
  lspeed = ICON_LSPEED;
  zspeed = ICON_ZSPEED;
  aspeed = ICON_ASPEED;
  ttl = ICON_TTL;
  remove = false;
}

char * Icon::getParam(char *ptok)
{
  char *p, *p2;
  p = strchr(ptok, '=');
  ptok++;
  if ((p = strchr(p, '"'))) {
    ptok = ++p;
    if ((p2 = strchr(p, '"')))
      *p2 = 0;
  }
  else {
    if ((p2 = strchr(ptok, '>')))
      *p2 = 0;
  }
  return ptok;
}

/* called by GUI */
Icon::Icon(User *puser, void *d)
{
  char *infos = (char *) d;
  ifile = NULL;
  ofile = NULL;
  char *action = NULL;
  vref = NULL;
  char icon[URL_LEN];
  *icon = 0;

  /* parameters transmission */
  for (char *ptok = strtok(infos, "&"); ptok ; ptok = strtok(NULL, "&")) {
    if (! strncmp(ptok, "<url=", 5)) {
      ptok = getParam(ptok);
      strcpy(name.url, ptok);
      taken = true;
    }
    else if (! strncmp(ptok, "<file=", 6)) {
      ptok = getParam(ptok);
      ifile = strdup(ptok);
    }
    else if (! strncmp(ptok, "<ofile=", 7)) {
      ptok = getParam(ptok);
      strcpy(name.url, ptok);
    }
    else if (! strncmp(ptok, "<name=", 6)) {
      ptok = getParam(ptok);
      strcpy(name.given_name, ptok);
    }
    else if (! strncmp(ptok, "<icon=", 6)) {
      ptok = getParam(ptok);
      strcpy(icon, ptok);
    }
    else if (! strncmp(ptok, "<action=", 8)) {
      ptok = getParam(ptok);
      action = strdup(ptok);
    }
    else if (! strncmp(ptok, "<vref=", 5)) {
      ptok = strchr(ptok, '=');
      ptok++;
      char *p2;
      if ((p2 = strchr(ptok, '>')))
        *p2 = 0;
      vref = strdup(ptok);
    }
  }

  tex = new char[128];

  if (vref) {	// old document
    char *l;
    l = parseObject(vref);
    l = parsePosition(l);
    l = parseGeometry(l);
    while (l) {
      if (!strncmp(l, "owner", 5)) {
        l = parseString(l, name.owner, "owner");
      }
    }
    int texn = TextureCacheEntry::getCurrentNumber();
    TextureCacheEntry *tc = TextureCacheEntry::getEntryByNumber(texn);
    if (! tc)
      strcpy(tex, IMG_DEF);
    else
      strcpy(tex, tc->texurl);
    taken = false;
  }

  else {	// new document
    /* position */
    pos.x = puser->pos.x + 0.1;
    pos.y = puser->pos.y;
    pos.z = puser->pos.z + 0.6;
    pos.az = puser->pos.az + M_PI_2;

    /* texture */
    if (*icon)
      strcpy(tex, icon);
    else {
      // default binding icon to document
      char pext[8] = "";
      if (getExt(name.url, pext))
        getImgByExt(pext, tex);
      else
        strcpy(tex, IMG_DEF);
    }

    if (ifile) {	// private file
      if (*name.url) {
        // public url given by user
        App::startwget(ifile, ofile, "-iO");	//TODO in wget.cc
      }
      else {
        // build ofile in ~/public_html/vreng/
        ofile = new char[URL_LEN];
        sprintf(ofile, "%s/public_html", getenv("HOME"));
        if (access(ofile, R_OK|W_OK|X_OK) == 0) {
          strcat(ofile, "/vreng/");
          if (access(ofile, R_OK|W_OK|X_OK) == -1)
            mkdir(ofile, 0755);
          strcat(ofile, ifile);
          FILE *fin, *fout;
          if ((fin = fopen(ifile, "r")) != NULL && (fout = fopen(ofile, "w")) != NULL) {
            char buf[2];
            while (fread(buf, 1, 1, fin))
              fwrite(buf, 1, 1, fout);
            fclose(fin);
            fclose(fout);
            chmod(ofile, 0644);
            //FIXME: define http_server
            sprintf(name.url, "http://%s/~%s/vreng/%s", DEF_HTTP_SERVER, getenv("USER"), ifile);
          }
          else {
            error("can't open %s or %s: %s (%d)", ifile, ofile, strerror(errno), errno);
            free(ifile);
            delete[] ofile;
            ifile = ofile = NULL;
          }
        }
        else {
          error("can't access %s", ofile);
          free(ifile);
          delete[] ofile;
          ifile = ofile = NULL;
        }
      }
    }

    strcpy(name.owner, puser->name.instance_name);

    /* geometry */
    char geom[128];
    sprintf(geom, "solid=\"box,dim=%f,%f,%f,dif=%s,xn=%s\"",
            ICON_WIDTH/2, ICON_DEPTH/2, ICON_HEIGHT/2, ICON_COLOR, tex);
    parseGeometry(geom);
  }

  /* local creation */
  type = ICON_TYPE;
  enableBehavior(REMOVABLE);
  enableBehavior(NO_ELEMENTARY_MOVE);
  //pd enableBehavior(COLLIDE_ONCE);
  initializeObject(LIST_MOBILE);
  setMaxLasting(ICON_LASTING);

  /* network creation */
  createVolatileNetObject(props);
  //pd worlds->plocaluser->noh->declareObjCreation();

  setDefaults();
  ttl = (taken) ? MAXFLOAT : 0;
  initImposedMovement(ttl);
  disablePermanentMovement();

  trace(DBG_WO, "Icon:: url=%s icon=%s name=%s owner=%s", name.url, tex, name.instance_name, name.owner);

  if (action) {
    if (! strncmp(action, "pin", 3))
      iconPin(this, NULL, 0L, 0L);
    else if (! strncmp(action, "push", 4))
      iconPush(this, NULL, 0L, 0L);
    else if (! strncmp(action, "carry", 5))
      iconCarry(this, NULL, 0L, 0L);
  }
}

/* Replication from the network */
WObject * Icon::replicator(uint8_t type_id, NetObjectId noid, Payload *pp)
{
  return new Icon(type_id, noid, pp);
}

Icon::Icon(uint8_t type_id, NetObjectId _noid, Payload *pp)
{
  setType(type_id);

  replicateVolatileNetObject(props, _noid);
  noh->getAllProperties(pp);
  copyNetObjectId(_noid);

  parseName(name.given_name);

  char geom[128];
  sprintf(geom, "solid=\"box,dim=%f,%f,%f,dif=%s,xn=%s\"",
          ICON_WIDTH/2, ICON_DEPTH/2, ICON_HEIGHT/2, ICON_COLOR, tex);
  parseGeometry(geom);

  initializeObject(LIST_MOBILE);
  initBB(pos);

  setDefaults();
  taken = false;
} 

/** Updates to the network */
bool Icon::updateToNetwork(const Pos &oldpos)
{
  bool change = false;
  
  if ((pos.x != oldpos.x) || (pos.y != oldpos.y)) {
    noh->declareObjDelta(ICON_PROPXY);
    change = true;
  }
  if (pos.z != oldpos.z) {
    noh->declareObjDelta(ICON_PROPZ);
    change = true;
  }
  if (pos.az != oldpos.az) {
    noh->declareObjDelta(ICON_PROPAZ);
    change = true;
  }
  if (pos.ax != oldpos.ax) {
    noh->declareObjDelta(ICON_PROPAX);
    change = true;
  }
  return change;
}

void Icon::whenIntersect(WObject *pcur, WObject *pold)
{
  switch (pcur->type) {
  case USER_TYPE:
  case ICON_TYPE:
    pcur->pos.x += 0.1;
    pcur->pos.y += 0.1;
    pcur->update3D();
    pcur->getBB();
    break;
  default:
    pold->copyPositionAndBB(pcur);
  }
}

/* this method is invisible: called by Wall::whenIntersect */
void iconStick(Wall *pwall, void *_picon, time_t sec, time_t usec)
{
  Icon *picon = (Icon *) _picon;

  //trace(DBG_FORCE, "Stick: icon=%.2f,%.2f,%.2f,%.2f wall=%.2f,%.2f,%.2f,%.2f", picon->pos.x, picon->pos.y, picon->pos.z, picon->pos.az, pwall->pos.x, pwall->pos.y, pwall->pos.z, pwall->pos.az);
#if 0 //pd
  if (pwall->pos.az == 0 || ABSF(pwall->pos.az - M_PI) < 0.01) {
   if (picon->pos.y < pwall->pos.y)
     picon->pos.az = 0;
   else
     picon->pos.az = M_PI;
  }
  else if (ABSF(pwall->pos.az - M_PI_2) < 0.01) {
   if (picon->pos.x < pwall->pos.x)
     picon->pos.az = -M_PI_2;
   else
     picon->pos.az = M_PI_2;
  }
  else {
    picon->pos.az = pwall->pos.az;
  }
#endif

  GLfloat iconcolor[3] = {1, 0, 0};		// red
  if (picon->soh)
    picon->soh->setSolidFlashy(iconcolor);	// flash the icon
  GLfloat wallcolor[3] = {1, 1, 0};		// yellow
  pwall->soh->setSolidFlashy(wallcolor);	// flash the wall

  picon->update3D();
  picon->getBB();
  //pd2 picon->enableBehavior(COLLIDE_NEVER);
  //pd1 picon->enableBehavior(COLLIDE_ONCE);
  picon->enableBehavior(COLLIDE_ONCE);
  picon->taken = false;
}

void iconPin(Icon *po, void *d, time_t s, time_t u)
{
  po->move.lspeed.v[0] = po->lspeed * 4 * Cos(worlds->plocaluser->pos.az);
  po->move.lspeed.v[1] = po->lspeed * 4 * Sin(worlds->plocaluser->pos.az);
  po->move.lspeed.v[2] = 0;
  po->move.aspeed.v[0] = 0;
  po->move.aspeed.v[2] = 0;
  po->ttl = ICON_TTL * 50;
  po->initImposedMovement(po->ttl);
  po->disablePermanentMovement();
  po->enableBehavior(COLLIDE_ONCE);
  //pd po->disableBehavior(COLLIDE_NEVER);
  po->taken = false;
}

void iconPush(Icon *po, void *d, time_t s, time_t u)
{
  po->move.lspeed.v[0] = po->lspeed * Cos(worlds->plocaluser->pos.az);
  po->move.lspeed.v[1] = po->lspeed * Sin(worlds->plocaluser->pos.az);
  po->move.lspeed.v[2] = 0;
  po->move.aspeed.v[0] = 0;
  po->move.aspeed.v[2] = 0;
  po->ttl = ICON_TTL * 4;
  po->initImposedMovement(po->ttl);
  po->disablePermanentMovement();
  po->enableBehavior(COLLIDE_ONCE);
  po->taken = false;
}

void iconPull(Icon *po, void *d, time_t s, time_t u)
{
  if (! po->taken) {
    po->soh->resetSolidFlashy();
    po->move.lspeed.v[0] = -po->lspeed * Cos(worlds->plocaluser->pos.az);
    po->move.lspeed.v[1] = -po->lspeed * Sin(worlds->plocaluser->pos.az);
    po->move.lspeed.v[2] = 0;
    po->move.aspeed.v[0] = 0;
    po->move.aspeed.v[2] = 0;
    po->ttl = ICON_TTL;
    po->initImposedMovement(po->ttl);
    po->disablePermanentMovement();
    po->enableBehavior(COLLIDE_ONCE);
    po->taken = false;
  }
}

void iconCarry(Icon *po, void *d, time_t s, time_t u)
{
  if (! po->taken) {
    po->soh->resetSolidFlashy();
    po->ttl = MAXFLOAT;
    po->initImposedMovement(po->ttl);
    po->disablePermanentMovement();
    //pd po->enableBehavior(COLLIDE_ONCE);
    po->disableBehavior(COLLIDE_NEVER);
    po->taken = true;
  }
}

void iconLeave(Icon *po, void *d, time_t s, time_t u)
{
  if (po->taken) {
    po->ttl = 0;
    po->initImposedMovement(po->ttl);
    po->taken = false;
  }
}

void iconDrop(Icon *po, void *d, time_t s, time_t u)
{
  if (! po->taken) {
    po->enablePermanentMovement();
    po->ttl = 0.;
    po->initImposedMovement(po->ttl);
    po->enableBehavior(COLLIDE_ONCE);
  }
}

void iconOpen(Icon *po, void *d, time_t s, time_t u)
{
  char *p, ext[8] = "???";
  getExt(po->name.url, ext);

  switch (getPlayerByExt(ext)) {
  case PLAYER_PS:
    App::viewps(po->name.url); break;
  case PLAYER_PDF:
    App::startpdf(po->name.url); break;
  case PLAYER_WGET:
    if ((p = strrchr(po->name.url, '/')))
      App::startwget(po->name.url, ++p, "-O");
    break;
  case PLAYER_OFFICE:
    App::startoffice(po->name.url); break;
  case PLAYER_MP3:
    App::startmp3(po->name.url); break;
  case PLAYER_MIDI:
    App::startmidi(po->name.url); break;
  default:
    App::startbrowser(po->name.url, false);	// without rewrite
  }
#if 0 //obsolete
  else
    GuiLaunchIconDisplay(po);
#endif
}

void iconSave(Icon *po, void *d, time_t s, time_t u)
{
  chdir(vrengcwd);
  char *pfile;
  if (po->name.url[strlen(po->name.url) - 1] == '/') {
    App::startwget(po->name.url, NULL, "");
    trace(DBG_FORCE, "web %s saved", po->name.url);
  }
  else if ((pfile = strrchr(po->name.url, '/'))) {
    App::startwget(po->name.url, ++pfile, "-O");
    trace(DBG_FORCE, "file %s saved", pfile);
  }
  else
    notice("nothing to save");
}

void iconUp(Icon *po, void *d, time_t s, time_t u)
{
  if (! po->taken) {
    po->pos.z += ICON_HEIGHT;
    po->update3D();
    po->getBB();
    po->disablePermanentMovement();
  }
}

void iconDown(Icon *po, void *d, time_t s, time_t u)
{
  if (! po->taken) {
    po->pos.z -= ICON_HEIGHT;
    po->update3D();
    po->getBB();
    po->disablePermanentMovement();
  }
}

void iconLeft(Icon *po, void *d, time_t s, time_t u)
{
  if (! po->taken) {
    po->pos.y += ICON_WIDTH;
    po->update3D();
    po->getBB();
  }
}

void iconRight(Icon *po, void *d, time_t s, time_t u)
{
  if (! po->taken) {
    po->pos.y -= ICON_WIDTH;
    po->update3D();
    po->getBB();
  }
}

void iconTurn(Icon *po, void *d, time_t s, time_t u)
{
  po->move.lspeed.v[0] = 0;
  po->move.lspeed.v[1] = 0;
  po->move.lspeed.v[2] = 0;
  po->move.aspeed.v[0] = po->aspeed * 2;
  po->move.aspeed.v[2] = 0;
  po->ttl = ICON_TTL;
  po->initImposedMovement(po->ttl);
  po->disablePermanentMovement();
}

void iconRoll(Icon *po, void *d, time_t s, time_t u)
{
  po->move.lspeed.v[0] = 0;
  po->move.lspeed.v[1] = 0;
  po->move.lspeed.v[2] = 0;
  po->move.aspeed.v[0] = 0;
  po->move.aspeed.v[2] = po->aspeed * 2;
  po->ttl = ICON_TTL;
  po->initImposedMovement(po->ttl);
  po->disablePermanentMovement();
}

void iconRemove(Icon *po, void *d, time_t s, time_t u)
{
  if (! strcmp(worlds->plocaluser->name.instance_name, po->name.owner)) {
    po->toDelete();	// delete Wobject
    po->soh->clearObjectBar();
    po->remove = true;

    // remove file
    char ficon[URL_LEN];

    url2filename(po->name.url, ficon);
    chdir(vrengicons);
    chdir(po->name.world_name);
    unlink(ficon);
    chdir(vrengcwd);
  }
  else
    warning("Permission denied, not owner!");
}

void get_url(Icon *po, Payload *pp)
{
  if (po) {
    pp->getPayload("s", po->name.url);
    if (po->name.url) trace(DBG_WO, "get_url: %s", po->name.url);
  }
}

void put_url(Icon *po, Payload *pp)
{
  if (po && po->name.url)
    pp->putPayload("s", po->name.url);
}

void get_tex(Icon *po, Payload *pp)
{
  if (po) {
    po->tex = new char[128];
    pp->getPayload("s", po->tex);
    if (po->tex) trace(DBG_WO, "get_tex: %s", po->tex);
  }
}

void put_tex(Icon *po, Payload *pp)
{
  if (po && po->tex)
    pp->putPayload("s", po->tex);
}

void get_gname(Icon *po, Payload *pp)
{
  if (po) {
    pp->getPayload("s", po->name.given_name);
    if (po->name.given_name) trace(DBG_WO,"get_gname: %s", po->name.given_name);
  }
}

void put_gname(Icon *po, Payload *pp)
{
  if (po && po->name.instance_name)
    pp->putPayload("s", po->name.instance_name);
}

void Icon::quit()
{
  // Save icons for persistency
  if ((! taken) && (! remove) && *name.url) {
    char ficon[URL_LEN];

    url2filename(name.url, ficon);

    if (chdir(vrengicons) != -1 ) {
      struct stat bufstat;
      if (stat(name.world_name, &bufstat) < 0)
        mkdir(name.world_name, 0700);
      chdir(name.world_name);

      FILE *fp;
      if ((fp = fopen(ficon, "r")) != NULL) {
        fclose(fp);
        unlink(ficon);
      }
      if ((fp = fopen(ficon, "w")) != NULL) {
        char buf[256];
        memset(buf, 0, sizeof(buf));
        sprintf(buf, " name=\"%s\" pos=\"%.2f,%.2f,%.2f,%.2f,%.2f\" solid=\"box,dim=%.2f,%.2f,%.2f,dif=%s,xn=%s\"",
                name.instance_name,
                pos.x, pos.y, pos.z, pos.az, pos.ax,
                ICON_WIDTH/2, ICON_DEPTH/2, ICON_HEIGHT/2, ICON_COLOR, tex);
        if (name.owner[0]) {
          strcat(buf, " owner=\"");
          strcat(buf, name.owner);
          strcat(buf, "\"/>");
        }
        strcat(buf, "\n");
        fputs(buf, fp);
        fclose(fp);
      }
      chdir(vrengcwd);
    }
    else
      error("can't chdir to %s", vrengicons);
  }

  delete[] tex;
  if (ifile)
    free(ifile);
  if (ofile)
    delete[] ofile;
  if (vref)
    free(vref);
}

void iconInitFuncList(void)
{
  getPropertyFunc[ICON_PROPXY][ICON_TYPE].pf = WO_PAYLOAD get_xy;
  getPropertyFunc[ICON_PROPZ][ICON_TYPE].pf = WO_PAYLOAD get_z;
  getPropertyFunc[ICON_PROPAZ][ICON_TYPE].pf = WO_PAYLOAD get_az;
  getPropertyFunc[ICON_PROPHNAME][ICON_TYPE].pf = WO_PAYLOAD get_hname;
  getPropertyFunc[ICON_PROPAX][ICON_TYPE].pf = WO_PAYLOAD get_ax;
  getPropertyFunc[ICON_PROPURL][ICON_TYPE].pf = WO_PAYLOAD get_url;
  getPropertyFunc[ICON_PROPTEX][ICON_TYPE].pf = WO_PAYLOAD get_tex;
  getPropertyFunc[ICON_PROPGNAME][ICON_TYPE].pf = WO_PAYLOAD get_gname;

  putPropertyFunc[ICON_PROPXY][ICON_TYPE].pf = WO_PAYLOAD put_xy;
  putPropertyFunc[ICON_PROPZ][ICON_TYPE].pf = WO_PAYLOAD put_z;
  putPropertyFunc[ICON_PROPAZ][ICON_TYPE].pf = WO_PAYLOAD put_az;
  putPropertyFunc[ICON_PROPHNAME][ICON_TYPE].pf = WO_PAYLOAD put_hname;
  putPropertyFunc[ICON_PROPAX][ICON_TYPE].pf = WO_PAYLOAD put_ax;
  putPropertyFunc[ICON_PROPURL][ICON_TYPE].pf = WO_PAYLOAD put_url;
  putPropertyFunc[ICON_PROPTEX][ICON_TYPE].pf = WO_PAYLOAD put_tex;
  putPropertyFunc[ICON_PROPGNAME][ICON_TYPE].pf = WO_PAYLOAD put_gname;

  setActionFunc(ICON_TYPE, ICON_OPEN, WO_ACTION iconOpen, "Open");
  setActionFunc(ICON_TYPE, ICON_SAVE, WO_ACTION iconSave, "Save");
  setActionFunc(ICON_TYPE, ICON_PIN, WO_ACTION iconPin, "Pin");
  setActionFunc(ICON_TYPE, ICON_LEAVE, WO_ACTION iconLeave, "Leave");
  setActionFunc(ICON_TYPE, ICON_CARRY, WO_ACTION iconCarry, "Carry");
  setActionFunc(ICON_TYPE, ICON_KILL, WO_ACTION iconRemove, "Remove");
  setActionFunc(ICON_TYPE, ICON_PUSH, WO_ACTION iconPush, "Push");
  setActionFunc(ICON_TYPE, ICON_PULL, WO_ACTION iconPull, "Pull");
  setActionFunc(ICON_TYPE, ICON_UP, WO_ACTION iconUp, "Up");
  setActionFunc(ICON_TYPE, ICON_DOWN, WO_ACTION iconDown, "Down");
  setActionFunc(ICON_TYPE, ICON_DROP, WO_ACTION iconDrop, "Drop");
  setActionFunc(ICON_TYPE, ICON_TURN, WO_ACTION iconTurn, "Turn");
  setActionFunc(ICON_TYPE, ICON_ROLL, WO_ACTION iconRoll, "Roll");
  setActionFunc(ICON_TYPE, ICON_CREAT, WO_ACTION iconCreate, "");
  setActionFunc(ICON_TYPE, ICON_STICK, WO_ACTION iconStick, "");
#if 0
  setActionFunc(ICON_TYPE, ICON_LEFT, WO_ACTION iconLeft, "Left");
  setActionFunc(ICON_TYPE, ICON_RIGHT, WO_ACTION iconRight, "Right");
#endif
}
