#include "global.h"
#include "wo.h"
#include "world.h"
#include "user.h"
#include "carrier.h"	// redirectkeyEvent
#include "move.h"

#include "net.h"	// isResponsible

#define MIN_MOVES	20	// orig: 20

static float maxlasting[OBJECTSNUMBER+1];


/**
 * notify a key has changed
 * key_id = 0..MAXKEYS-1
 * pressed is TRUE for a Key Press and FALSE for a Key Release
 * sec: seconds, usec: micro-seconds
 */
void changeKey(int k_id, bool pressed, time_t sec, time_t usec)
{
  if (k_id >= MAXKEYS || k_id < 0) {
    warning("changeKey: invalid k_id = %d", k_id);
    return;
  }

  User *pu = worlds->plocaluser;  // possibility to act with an other user

  if (pu->kpressed[k_id] != pressed) {
    pu->kpressed[k_id] = pressed;
    if (pressed) {
      pu->kpstart_s[k_id] = sec;
      pu->kpstart_u[k_id] = usec;
    }
    else {
      pu->kpdur_s[k_id] = sec - pu->kpstart_s[k_id];
      pu->kpdur_u[k_id] = usec - pu->kpstart_u[k_id];
    }
  }
}

/** clear keys times array */
void User::clearKeyTab()
{
  for (int k=0; k < MAXKEYS; k++) {
    kpressed[k] = false;
    kpstart_s[k] = kpstart_u[k] = 0;
    kpdur_s[k] = kpdur_u[k] = 0;
 }
}

/** update the keydifftime arrays */
void User::updateKeysTimes(time_t sec, time_t usec)
{
  for (int k=0; k < MAXKEYS; k++) {
    if (kpressed[k]) {
      kpdur_s[k] = sec - kpstart_s[k];
      kpdur_u[k] = usec - kpstart_u[k];
      kpstart_s[k] = sec;
      kpstart_u[k] = usec;
    }
  }
}

/** modify user position in a direction */
void User::changePositionOneDir(const int move_type, const float lasting)
{
  pos.old_x = pos.x;
  pos.old_y = pos.y;
  pos.old_z = pos.z;

  switch (move_type) {
  
  case KEY_AV:	// move forward left
    if (car->iscarring)
      car->redirectkeyEvent(KEY_AV, lasting);
    else {
      pos.x += lasting * lspeed * Cos(pos.az);
      pos.y += lasting * lspeed * Sin(pos.az);
    }
    break;
  case KEY_AR:	// move backward right
    if (car->iscarring)
      car->redirectkeyEvent(KEY_AR, lasting);
    else {
      pos.x -= lasting * lspeed * Cos(pos.az);
      pos.y -= lasting * lspeed * Sin(pos.az);
    }
    break;
  case KEY_SD:	// move forward right
    pos.x += lasting * lspeed * sin(pos.az);
    pos.y -= lasting * lspeed * Cos(pos.az);
    break;
  case KEY_SG:	// move backward left
    pos.x -= lasting * lspeed * Sin(pos.az);
    pos.y += lasting * lspeed * Cos(pos.az);
    break;
  case KEY_DR:	// turn right
    if (car->iscarring)
      car->redirectkeyEvent(KEY_DR, lasting);
    else {
      pos.az -= lasting * aspeed;
      pos.az -= M_2PI * (float) floor(pos.az / M_2PI);
    }
    break;
  case KEY_GA:	// turn left
    if (car->iscarring)
      car->redirectkeyEvent(KEY_GA, lasting);
    else {
      pos.az += lasting * aspeed;
      pos.az -= M_2PI * (float) floor(pos.az / M_2PI);
    }
    break;
  case KEY_MT:	// roll backward
    pos.ay = MIN(pos.ay + lasting * aspeed, M_2PI_5);
    break;
  case KEY_DE:	// roll forward
    pos.ay = MAX(pos.ay - lasting * aspeed, -M_2PI_5);
    break;
  case KEY_HZ:	// stand up
    pos.ay = pos.ax = 0;
    updateObjectIntoGrid(pos);
    break;
  case KEY_JU:	// move up
    pos.z += lasting * lspeed;
    break;
  case KEY_JD:	// move down
    pos.z -= lasting * lspeed;
    break;
  case KEY_TL:	// tilt left
    pos.ax = MIN(pos.ax + lasting * aspeed, M_2PI_5);
    break;
  case KEY_TR:	// tilt right
    pos.ax = MAX(pos.ax - lasting * aspeed, -M_2PI_5);
    break;
  }
}

/** fill times's array for each user motion direction */
void User::updateTime(float lastings[])
{
  for (int k=0; k < MAXKEYS; k++) {
    lastings[k] = (float) kpdur_s[k] + (float) kpdur_u[k] / 1e6;
    kpdur_s[k] = kpdur_u[k] = 0;
  }
}

/** do the motion in each direction */
void User::changePosition(const float lastings[])
{
  for (int k=0; (k < MAXKEYS); k++)
    if (lastings[k] > MIN_KEYLASTING)
      changePositionOneDir(k, lastings[k] * (kpressed[KEY_VI]+1));
}

void setMaxLasting(const int type_id, const float maxlast)
{
  maxlasting[type_id] = maxlast;
}

void WObject::setMaxLasting(const float maxlast)
{
  maxlasting[type] = maxlast;
}

float getMaxLasting(const int type_id)
{
  return maxlasting[type_id];
}

void WObject::enableImposedMovement()
{
  struct timeval t;
  gettimeofday(&t, NULL);
  move.sec = t.tv_sec;
  move.usec = t.tv_usec;
}

void WObject::initImposedMovement(const float _ttl)
{
  enableImposedMovement();
  move.ttl = (_ttl < 0) ? -_ttl : _ttl;
}

float WObject::diffTime(time_t sec, time_t usec)
{
  return (float) (sec - move.sec) + ((float) (usec - move.usec) / 1e6);
}

void WObject::stopImposedMovement()
{
  move.ttl = 0;
  move.sec = 0;
  move.usec = 0;
}

void WObject::enablePermanentMovement()
{
  struct timeval t;
  gettimeofday(&t, NULL);
  move.perm_sec = t.tv_sec;
  move.perm_usec = t.tv_usec;
}

void WObject::disablePermanentMovement()
{
  move.perm_sec = 0;
  move.perm_usec = 0;
}

void User::elementaryUserMovement(const float deltalastings[])
{
  WObject *pold = new WObject();
   copyPositionAndBB(pold);	// keep oldpos for intersection

   changePosition(deltalastings);

   update3D();
   getBB();
   ObjectList *vicinity = getVicinityObjectList(pold->pos);
   if (vicinity) {
     generalIntersect(pold, vicinity);
     vicinity->freeObjectList();
   }
   //else
   //  error("elementaryUserMovement: FIXME vicinity NULL");
  delete pold;
}

/** user general motion */
void User::userMovement(time_t sec, time_t usec)
{
  Pos oldpos = pos;
  copyPosAndBB(oldpos);		// keep oldpos for network

  float keylastings[MAXKEYS];

  updateKeysTimes(sec, usec);
  updateTime(keylastings);

  float lasting = -1.;
  for (int k=0; k < MAXKEYS; k++) {
    if (keylastings[k] > lasting)
      lasting = keylastings[k];
  }

  if (lasting > MIN_LASTING) {	// user is moving
    float maxlast = maxlasting[type];
    int moves = (int) (lasting / maxlast);
    float deltalastings[MAXKEYS];

    for (int m=0; m <= moves; m++) {
      for (int k=0; k < MAXKEYS; k++) {
	if (keylastings[k] > maxlast) {
	  deltalastings[k] = maxlast;
	  keylastings[k] -= maxlast;
	}
	else {
	  deltalastings[k] = keylastings[k];	// last movement
	  keylastings[k] = 0.;
	}
      }
      elementaryUserMovement(deltalastings);
    }

    updateToNetwork(oldpos);

    updateObjectIntoGrid(oldpos);
    update3D();
    updateCamera();
  }
}

void WObject::elementaryImposedMovement(const float deltalasting)
{
  if (isBehavior(NO_BBABLE) || isBehavior(COLLIDE_NEVER)) {
    changePosition(deltalasting);	// handled by each object
    update3D();
    return;
  }

  WObject *pold = new WObject();
   copyPositionAndBB(pold);		// keep oldpos for intersection

   changePosition(deltalasting);	// handled by each object

   update3D();
   getBB();
   ObjectList *vicinity = getVicinityObjectList(pold->pos);
   if (vicinity) {
     generalIntersect(pold, vicinity);
     vicinity->freeObjectList();
   }
   //else
   //  error("elementaryImposedMovement: FIXME vicinity NULL");
  delete pold;
}

/** object imposed motion */
void WObject::imposedMovement(time_t sec, time_t usec)
{
  if (! isValid()) {
    error("imposedMovement: id_type=%d invalid", type);
    return;
  }
  if (World::isDead())
    return;

  if (! isMoving())
    return;

#if 1 //gpl PQ
  if (up_p && soh)
    changeComplexePosition();
#endif //gpl PQ

  Pos oldpos = pos;
  copyPosAndBB(oldpos);			// keep oldpos for network

  float lasting = -1;

  updateTime(sec, usec, &lasting);	// handled by each object

  float maxlast = maxlasting[type];
  if (lasting > MIN_LASTING) {
    // spliting movement in m elementary movements
    float deltalasting = 0.;
    int moves = (int) (lasting / maxlast);

    for (int m=0; m <= moves; m++) {
      if (lasting > maxlast) {
        deltalasting = maxlast;
        lasting -= maxlast;
      }
      else {
        deltalasting = lasting;
        lasting = 0.;
      }
      elementaryImposedMovement(deltalasting);
      if (isBehavior(NO_ELEMENTARY_MOVE))
        return;
    }
  }

  if (noh && noh->isResponsible())
    updateToNetwork(oldpos);		// handled by each object

  updateObjectIntoGrid(oldpos);
  update3D();
}

void WObject::elementaryPermanentMovement(const float deltalasting)
{
  if (isBehavior(NO_BBABLE) || isBehavior(COLLIDE_NEVER)) {
    changePermanent(deltalasting);	// handled by each object
    update3D();
    return;
  }

  WObject *pold = new WObject();
   copyPositionAndBB(pold);		// keep oldpos for intersection

   changePermanent(deltalasting);	// handled by each object

   update3D();
   getBB();
   ObjectList *vicinity = getVicinityObjectList(pold->pos);
   if (vicinity) {
     generalIntersect(pold, vicinity);
     vicinity->freeObjectList();
   }
   //else
   //  error("elementaryPermanentMovement: FIXME vicinity NULL");
  delete pold;
}

/** object permanent motion */
void WObject::permanentMovement(time_t sec, time_t usec)
{
  if (move.perm_sec > 0) {	// is permanent movement activated ?
    Pos oldpos = pos;
    copyPosAndBB(oldpos);

    float lasting = (float)(sec - move.perm_sec) + (float)(usec - move.perm_usec) / 1e6;
    move.perm_sec = sec;
    move.perm_usec = usec;
 
    float maxlast = maxlasting[type];
    if (lasting > MIN_LASTING) {
      float deltalasting = 0.;
      int moves = (int) (lasting / maxlast);
      moves = MIN(moves, MIN_MOVES);

      for (int m=0; m <= moves; m++) {
	if (lasting > maxlast) {
	  deltalasting = maxlast;
	  lasting -= maxlast;
	}
	else {
	  deltalasting = lasting;
	  lasting = 0.;
	}
	elementaryPermanentMovement(deltalasting);
        if (isBehavior(NO_ELEMENTARY_MOVE))
          return;
      }
    }

    if (noh && noh->isResponsible())
      updateToNetwork(oldpos);		// handled by each object

    updateObjectIntoGrid(oldpos);
    update3D();
    if (this == worlds->plocaluser)
      updateCamera();
  }
}
