#include "global.h"
#include "wo.h"
#include "vjc.h"

#include "net.h"	// setNoBlocking


const WClass Vjc::wclass(VJC_TYPE, "Vjc", Vjc::creator);
void vjcInitFuncList(void) {}
 
// Static class member initialization
Vjc *Vjc::server = NULL;


WObject * Vjc::creator(char *l)
{
  return new Vjc(l);
}

Vjc::Vjc(char *l)
{
  if (getServer()) {
    warning("You can't have more than one Vjc per world");
    return;
  }

  // defaults
  strcpy(host, DEF_VJS_SERVER);
  serverPort = VJS_PORT;
  localPort = VJC_LOCAL_PORT;

  l = parseObject(l);

  while (l) {
    if (!strncmp(l, "server", 6)) {
      l = parseString(l, host, "server");
    }
    // wanted ports
    else if (!strncmp(l, "port", 4)) {
      l = parseUInt16(l, &serverPort, "port");
    }
#if 0 //pd
    else if (!strncmp(l, "lport", 5)) {
      l = parseUInt16(l, &localPort, "lport");
    }
#endif
  }

  initializeObject(LIST_INVISIBLE);

  setServer(this);
  sock = NULL;
  lastMessage = NULL;
  lastping = 0;
  ssrc = getMySsrcId();
  start();
} 

Vjc::~Vjc()
{
  stop();
}

/* Returns the singleton instance */
Vjc * Vjc::getServer()
{
  return server;
}

/* Sets the singleton instance */
void Vjc::setServer(Vjc *_server)
{
  server = _server;
}

/* Used to send pings to the server */
void Vjc::render()
{
  getServer()->ping();
}

/* Sends a simple (no data) control command to the child */
int Vjc::sendCommand(WObject *po, int id)
{
  VjcMessage *msg = new VjcMessage(po, VJC_MSGT_CTRL, id);
  int ret = msg->sendData();
  delete msg;
  return ret;
}

/* Sends a ping to the server every VJC_PING_WAIT times */
void Vjc::ping()
{
  lastping++;
  if (lastping >= VJC_PING_WAIT) {
    sendCommand(this, VJC_MSGV_PING);
    lastping = 0;
  }
}

void Vjc::quit()
{
  //PD done by World::quit// delete this;
}

/* Open the socket */
void Vjc::start()
{
  sock = new VjcSocket(localPort, host, serverPort);

  if (sock->openSocket() == -1) {
    warning("Unable to open local comm. socket");
    delete sock;
    sock = NULL;
  }
  else {
    localPort = sock->listenPort;
    VjcMessage *msg = new VjcMessage(this, VJC_MSGT_CTRL, VJC_MSGV_INITIATE);
    msg->sendData();
    delete msg;
  }
}

/* Sends a terminate notification to the server */
void Vjc::stop()
{
  Vjc *srv = getServer();
  if (! srv) return;

  trace(DBG_IFC, "Shutting down vijs server");
  sendCommand(NULL, VJC_MSGV_TERMINATE);
  delete srv->sock;
  srv->sock = NULL;
  srv->serverPort = 0;
  srv->localPort = 0;
  setServer(NULL);
}

/* Unregister a Vrelet object with the server */
void Vjc::stopApp(Vrelet *po)
{
  Vjc *srv = getServer();
  if (! srv) return;

  sendCommand(po, VJC_MSGV_UNREGISTER);
  if (srv->lastMessage) {
    delete srv->lastMessage;
    srv->lastMessage = NULL;
  }
}

/* Register a Vrelet object with the server */
void Vjc::startApp(Vrelet *po)
{
  Vjc *srv = getServer();
  if (! srv)
    return;

  char hostname[MAXHOSTNAMELEN];
  if (gethostname(hostname, sizeof(hostname)-1) < 0)
    return;

  VjcMessage *msg = new VjcMessage(po, VJC_MSGT_CTRL, VJC_MSGV_REGISTER);
  msg->putStr(hostname);
  msg->put16(srv->localPort);
  msg->putStr(po->name.given_name);
  msg->putStr(po->app);
  msg->putStr(po->url);
  msg->put32(po->incrx);
  msg->put32(po->incry);
  msg->sendData();
  delete msg;
}


//*
//* Network management
//*

/* Initialize a pair of sockets for communicating with the child app */
VjcSocket::VjcSocket(int _listenPort, char *_destHost, int _destPort)
{
  listenPort = _listenPort;
  destPort = _destPort;
  sdr = -1;
  sdw = -1;

  struct sockaddr_in *sa;
  if ((sa = new struct sockaddr_in[1]) == NULL) {
    error("VjcSocket: can't new sa");
    return;
  }
  memset(sa, 0, sizeof(struct sockaddr_in));
  sa->sin_family = AF_INET; 
  sa->sin_port = htons(destPort);

  struct hostent *hp;
  if ((hp = my_gethostbyname(_destHost, AF_INET)) == NULL) {
    warning("VjcSocket: server unknown");
    delete[] sa;
    return;
  }
  memcpy(&sa->sin_addr, hp->h_addr_list[0], hp->h_length);
  destAddr = sa;

  state = VJC_SOCK_CLOSED;
}

/* Opens the receiver socket for this socket pair */
int VjcSocket::openRecv()
{
  if ((sdr = socketDatagram()) < 0)
    return 0;

  struct sockaddr_in sa;
  memset(&sa, 0, sizeof(sa));
  sa.sin_family = AF_INET; 
  sa.sin_addr.s_addr = INADDR_ANY;

  // Bind it
  setReuseAddr(sdr);
  for (int tries = 0; tries<10; tries++) {
    sa.sin_port = htons(listenPort + tries);
    if (bind(sdr, (struct sockaddr *) &sa, sizeof(sa)) != -1) {
      listenPort += tries;
      notice("Vjc receiver bound to port %d", listenPort);
      state = VJC_SOCK_RCVOPEN;
      return 1;
    }
  }
  error("openRecv bind: %s", strerror(errno));
  close(sdr);
  return 0;
}

/* Opens the sender socket. IO is set to non-blocking on this one */
int VjcSocket::openSend()
{
  if ((sdw = socketDatagram()) < 0)
    return 0;

  setNoBlocking(sdw);
  state = VJC_SOCK_CONNECT;
  return 1;
}

/* Tries a connect to the client app */
int VjcSocket::connectSend()
{
  if (connect(sdw, (const struct sockaddr *) destAddr, sizeof(struct sockaddr_in)) < 0) {
    if ((errno == EINPROGRESS) || (errno == EALREADY)) {
      // Socket set to non-blocking to prevent an app freeze
      return 1;
    }
    else {
      error("connectSend: %s", strerror(errno));
      return 0;
    }
  }
  else
    state = VJC_SOCK_OPEN;
  return 1;
}

/* Check if the non-blocking connect call on the send socket finished or not */
bool VjcSocket::isConnected()
{
  fd_set set;
  struct timeval timeout;

  FD_ZERO(&set);
  FD_SET(sdw, &set);
  timeout.tv_sec = 0;
  timeout.tv_usec = 0;
  if (select(sdw+1, NULL, &set, NULL, &timeout) > 0) {
    state = VJC_SOCK_OPEN;	// The socket finished connecting
    return true;
  }
  return false;
}

/* Opens a VjcSocket. This call is 'non-blocking' */
int VjcSocket::openSocket()
{
  // check if it's not already open
  if (state == VJC_SOCK_OPEN)
    return 1;
  
  // Open the receiver
  if (state == VJC_SOCK_CLOSED) {
    if (openRecv() == 0)
      return -1;
  }
  // Open the sender
  if (state == VJC_SOCK_RCVOPEN) {
    if (openSend() == 0)
      return -1;
  }
  // Connect the sender
  if (state == VJC_SOCK_CONNECT) {
    if (connectSend() == 0)
      return -1;
  }

  if (state == VJC_SOCK_OPEN)
    return 1;	// Everything is connected

  // One of the steps didn't finish yet.
  return 0;
}

/* Closes and frees a VjcSocket */
VjcSocket::~VjcSocket()
{
  if (state >= VJC_SOCK_RCVOPEN) {
    close(sdr);
    sdr = -1;
  }
  if (state >= VJC_SOCK_CONNECT) {
    close(sdw);
    sdw = -1;
  }
  if (destAddr)
    delete[] destAddr;
  destAddr = NULL;
}

/*
 * Utility functions
 */
bool VjcSocket::isOpen()
{
  return (this && (state == VJC_SOCK_OPEN));
}

bool VjcSocket::isClosed()
{
  return (!this || (state == VJC_SOCK_CLOSED));
}


/*
 * VjcMessage
 * Data structure used to send messages to the controler
 */

/* Constructor for outgoing messages */
VjcMessage::VjcMessage(WObject *po, int ssrc, int type, int id)
{
  setup(po, ssrc, type, id);
}

/* Constructor for outgoing messages */
VjcMessage::VjcMessage(WObject *po, int type, int id)
{
  setup(po, (Vjc::getServer() == NULL ? getMySsrcId() : Vjc::getServer()->ssrc), type, id);
}

/* Creates a new vjcHeader */
static
vjcHeader vjcNewHeader(int ssrc, int type, int id, int len)
{
  vjcHeader hdr;

  hdr.proto = VJC_PROTO;
  hdr.version = VJC_VERSION;
  hdr.app_ssrc = ssrc;
  hdr.msg_type = type;
  hdr.msg_id  = id;
  hdr.data_len = len;
  return hdr;
}

/* Setups the header */
void VjcMessage::setup(WObject *po, int ssrc, int type, int id)
{
  cursor = 0;	// set the cursor to the start of the buffer
  header = vjcNewHeader(ssrc, type, id, 0);	// create the header
  sender = po;	// set the sender object
  if ((data = new int8_t[VJC_MAX_PACKET]) == NULL) {	// alloc our data array
    warning("cannot new data buffer for outgoing VjcMessage");
    return;
  }
  putHeader(); // Write out the header
}

/* Constructor for incoming messages */
VjcMessage::VjcMessage(int8_t *_data, int _datalen)
{
  cursor = 0;		// set the cursor to the start of the buffer
  sender = NULL;	// sender will always be null on incoming messages
  if ((data = new int8_t[_datalen]) == NULL) {	// alloc the buffer
    warning("VjcMessage::VjcMessage: can't new for incoming VjcMessage");
    return;
  }
  memcpy(data, _data, _datalen);	// copy the data
  maxlen = _datalen;			// mark the maximum read length
  header = readHeader();		// read the header in
}

/* Destructor */
VjcMessage::~VjcMessage()
{
  if (data)
    delete[] data;
  data = NULL;
}

/* Checks whether the packet was for this WObject */
bool VjcMessage::isForObject(WObject *po)
{
  return (header.src_id == po->getSrcId())
      && (header.port_id == po->getPortId())
      && (header.obj_id == po->getObjId());
}

/* Returns this message's header */
vjcHeader VjcMessage::getHeader()
{
  return header;
}

#define VJC_CHECK_OVERFLOW_1(a) \
  if (cursor >= VJC_MAX_PACKET-a) { \
    warning("Write past end of buffer (%d %d)", getHeader().msg_type, getHeader().msg_id); \
    return; \
  }

/* Add a 8bit int to the message */
void VjcMessage::put8(const int val)
{
  //pd VJC_CHECK_OVERFLOW_1(sizeof(int8_t))
  data[cursor] = (int8_t) (0x000000FF) & val;
  cursor += sizeof(int8_t);
}

/* Add a 16bit int to the message */
void VjcMessage::put16(const int val)
{
  //pd VJC_CHECK_OVERFLOW_1(sizeof(int16_t))
  data[cursor]   = (val & 0xFF00) >> 8;
  data[cursor+1] = (val & 0x00FF);
  cursor += sizeof(int16_t);
}
 
/* Add a 32bit int to the message */
void VjcMessage::put32(const int val)
{
  //pd VJC_CHECK_OVERFLOW_1(sizeof(int32_t))
  data[cursor]   = (val & 0xFF000000) >> 24;
  data[cursor+1] = (val & 0x00FF0000) >> 16;
  data[cursor+2] = (val & 0x0000FF00) >> 8;
  data[cursor+3] = (val & 0x000000FF);
  cursor += sizeof(int32_t);
}

/* Add a string to the message */
void VjcMessage::putStr(const char *str)
{
  int i, len = strlen(str);
  put16(len);
  VJC_CHECK_OVERFLOW_1(len)
  for (i=0; i<len ; i++)
    data[cursor+i] = (int8_t) str[i];
  cursor += len;
}
 
/* Puts an object's net id */
void VjcMessage::putOID(WObject *po)
{
  if (po) {
    put8((po == sender ? 0 : po->type));
    put32(po->getSrcId());
    put16(po->getPortId());
    put16(po->getObjId());
  }
  else {
    put8( 0);
    put32(0);
    put16(0);
    put16(0);
  }
}
 
/* Puts an object position */
void VjcMessage::putPos(WObject *po)
{
  put32((int32_t) (po->pos.x * 1000));
  put32((int32_t) (po->pos.y * 1000));
  put32((int32_t) (po->pos.z * 1000));
  put32((int32_t) (po->pos.az * 1000));
  put32((int32_t) (po->pos.ax * 1000));
  put32((int32_t) 0);
}

/* Writes the message header */
void VjcMessage::putHeader()
{
  // Write the header (proto part)
  put16(header.proto);
  put8( header.version);
  put32(header.app_ssrc);

  // Write the header (message part)
  put8( header.msg_type);
  put8( header.msg_id);
  put16(header.data_len);

  // Write the caller's id
  putOID(sender);
}

/* Packs this message into an int8_t array */
int8_t *VjcMessage::toBytes(int *len)
{
  int datalen = cursor - VJC_HDR_LEN;	// total length - header
  *len = cursor;	// total length is where our cursor currently is
  cursor = VJC_HDR_DATALEN_IDX;	// set the data length in the packet
  put16(datalen);
  cursor = VJC_HDR_SSRC_IDX;	// set the ssrc (could have been updated)
  put32(Vjc::getServer()->ssrc);
  // Restore the cursor. This should actually never be needed, except
  // if someone sends the same message twice, after updating it
  cursor = *len;
  return data;		// return a pointer to the data.
}
 
/* Send data over to the server */
int VjcMessage::sendData()
{
  int sent = 0, total = 0;
  int8_t *pkt;

  /* Check that the socket is ready */
  Vjc *srv = Vjc::getServer();
  if (srv && (srv->sock->state == VJC_SOCK_CONNECT)) {
    // The socket is still trying to connect. Check if it's done or not.
    if (! srv->sock->isConnected()) {
      return 0;	// The connection is still in progress
    }
  }
  if (srv && (srv->sock->state == VJC_SOCK_OPEN)) {
    /* Check if an SSRC change occured since we sent the REGISTER commands */
    if (getMySsrcId() != srv->ssrc) {
      if (! ((getHeader().msg_type == VJC_MSGT_CTRL)
          && (getHeader().msg_id  == VJC_MSGV_REGISTER))) {

        VjcMessage *msg = new VjcMessage(srv, srv->ssrc, VJC_MSGT_CTRL, VJC_MSGV_UPDATE);
        trace(DBG_IFC, "updating SSRC for Vjc (old:%d, new:%d)", srv->ssrc, getMySsrcId());
        msg->put32(getMySsrcId());
        pkt = msg->toBytes(&total);
        sent = send(srv->sock->sdw, pkt, total, 0);
        delete msg;
      }
      srv->ssrc = getMySsrcId();
    }

    /* send the message */
    pkt = toBytes(&total);
    sent = send(srv->sock->sdw, pkt, total, 0);
    trace(DBG_IFC, "sending %d bytes with %02x %02x %d %08x (%d %d)",
	  sent, pkt[0], pkt[1], pkt[2], getMySsrcId(),
	  getHeader().msg_type,
	  getHeader().msg_id);
    return 1;
  }
  return 0;
}

#define VJC_CHECK_OVERFLOW_2(a) \
  if (cursor > maxlen-a) { \
    warning("Read past end of buffer (%d %d)", getHeader().msg_type, getHeader().msg_id); \
    return NULL; \
  }

/* Reads in data from the child process, if any is availabe */
VjcMessage * Vjc::getData(WObject *po)
{
  /* check that the socket is really open */
  Vjc *srv = getServer();
  if ((! srv) || (srv->sock->state != VJC_SOCK_OPEN)) {
    error("vjc socket");
    return NULL;
  }

  if (srv->lastMessage)
    goto haspack;
 
  int8_t *pkt;

  /* set a 0 sec timeout:
   * the select will return instantly if no data is available */
  fd_set fds;
  struct timeval to;

  FD_ZERO(&fds);
  FD_SET(srv->sock->sdr, &fds);
  to.tv_sec = 0;
  to.tv_usec = 0;

  if (select(srv->sock->sdr+1, &fds, NULL, NULL, &to) > 0) {
    int r;

    if ((pkt = new int8_t[VJC_MAX_PACKET]) == NULL) {
      warning("Vjc::getData: can't new reception buffer");
      return NULL;
    }

    if ((r = recv(srv->sock->sdr, (void *) pkt, VJC_MAX_PACKET, 0)) > 0) {
      if (r < VJC_HDR_LEN) {
	/* We didn't get a whole header */
	warning("Dropped incomplete packet (%d)", r);
      }
      else {
	/* Message size is big enough to contain a header */
	srv->lastMessage = new VjcMessage(pkt, r);

	/* Check the header */
	// TODO: check protocol id and version number
	if (srv->lastMessage->getHeader().data_len > (VJC_MAX_PACKET-VJC_HDR_LEN)) {
	  /* Header and real length don't agree */
	  warning("Illegal data lenght");
	  delete srv->lastMessage;
	  srv->lastMessage = NULL;
	}
        else
	  goto haspack;
      }
    }
    // (r < 0) || (packet too short) || (invalid header)
    delete[] pkt;
    return NULL;
  }
  return NULL; // select <= 0

haspack:
  // A message was read from the socket, or had been read previously
  if (srv->lastMessage->isForObject(po)) {
    VjcMessage *ret = srv->lastMessage;
    srv->lastMessage = NULL;
    return ret;
  }
  return NULL;
}

/* Reads an 8 bit signed int */
int8_t VjcMessage::read8()
{
  //pd VJC_CHECK_OVERFLOW_2(sizeof(int8_t))
  int8_t val = data[cursor];
  cursor += sizeof(int8_t);
  return val;
}
 
/* Reads a 16 bit signed int */
int16_t VjcMessage::read16()
{
  //pd VJC_CHECK_OVERFLOW_2(sizeof(int16_t))
  int16_t val = (data[cursor] << 8) + (0x00ff & data[cursor+1]);
  cursor += sizeof(int16_t);
  return val;
}
 
/* Reads a 32 bit signed int */
int32_t VjcMessage::read32()
{
  //pd VJC_CHECK_OVERFLOW_2(sizeof(int32_t))
  int32_t val =
          (0xff000000 & (data[cursor]   << 24))
        + (0x00ff0000 & (data[cursor+1] << 16))
        + (0x0000ff00 & (data[cursor+2] << 8))
        + (0x000000ff &  data[cursor+3]);
  cursor += sizeof(int32_t);
  return val;
}

/* Read a header struct for the raw packet data */
vjcHeader VjcMessage::readHeader()
{
  vjcHeader hdr;

  hdr.proto    = read16();
  hdr.version  = read8();
  hdr.app_ssrc = read32();
  hdr.msg_type = read8();
  hdr.msg_id   = read8();
  hdr.data_len = read16();
  hdr.obj_type = read8(); if (hdr.obj_type == 0) hdr.obj_type = VJC_TYPE;
  hdr.src_id   = read32();
  hdr.port_id  = read16();
  hdr.obj_id   = read16();
  return hdr;
}
 
/* Read two 32bit ints, and return them as the x and y coords. of a V3 */
V3 VjcMessage::readPoint2D()
{
  V3 point;

  //pd VJC_CHECK_OVERFLOW_2(2*sizeof(int32_t))
  point.v[0] = ((float) read32());
  point.v[1] = ((float) read32());
  point.v[2] = 0;
  return point;
}

V3 VjcMessage::readPoint3D()
{
  V3 point = readPoint2D();

  //pd VJC_CHECK_OVERFLOW_2(sizeof(int32_t))
  point.v[2] = ((float) read32());
  return point;
}
 
/* Read three 32bit ints, converted to floats by division by 1000 */
V3 VjcMessage::readDelta()
{
  V3 point = readPoint3D();

  point.v[0] /= 1000.;
  point.v[1] /= 1000.;
  point.v[2] /= 1000.;
  return point;
}

/* Returns true if there is at least size bytes left to be read */
bool VjcMessage::hasData(const int size)
{
  return ((cursor+size) <= maxlen);
}
