//<copyright>
//
// Copyright (c) 1996
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        faceattr.C
//
// Purpose:     Interface to class FaceSet
//
// Created:      7 Feb 1996  Georg Meszaros
//
// Modified:  
//
//
// Description:
//
// Implementation of the class FaceAttr.
//
//</file>


//<class>
//
// Name:    FaceAttr
//
// Purpose: Indexed faceset with light, material settings.
//
//
// Public Interface:
//
//
//</class>


#include "faceattr.h"


#include "vrml/QvMFVec3f.h"
#include <hyperg/utils/types.h>
#include "polygon.h"
#include "attributes.h"

//just to test
#include "clipping.h"
#include <iostream.h>
 

unsigned long FaceAttr::attr_number_ = 0;
unsigned long FaceAttr::counter_ = 0;
int FaceAttr::alloc_ahead_ = 100;


FaceAttr::FaceAttr()
{
  id_ = counter_++;
  cout << "*:Constructor FaceAttr - id:" << id_ <<"\n";

  attr_number_++; 
  vertex_num_ = 0;
  normal_num_ = 0;

  light_num_ = 0;
  material_ = nil; 
  light_ = nil;           
  backface_culling_ = 1;  
}


FaceAttr::FaceAttr(                           
      int vertex_num,        // number of vertices
      QvMFVec3f* vertex_list,     // vertices
      int normal_num,        // number of normals
      QvMFVec3f* normal_list)      // normals
{
  id_ = counter_++;
 
//cout << "*FaceAttr::FaceAttr id_ : " << id_ << "\n";
  attr_number_++; 

  face_ref_count_ = 0; 
  light_num_ = 0;
  material_ = nil; 
  light_ = nil;           
  backface_culling_ = 1;  

  // is allocated in QvIndexedFaceSet::buildBSP
  vertex_list_ = vertex_list;
  vertex_num_ = vertex_num;
  normal_list_ = normal_list;
  normal_num_ = normal_num;

  //cout << "*:vertex_list_->values, end of constructor faceattr\n";
  //printVertices();

 
   
} // FaceAttr


FaceAttr::FaceAttr(const FaceAttr& faceattr)
{
  copy(faceattr);
}


FaceAttr::~FaceAttr() 
{
  free(); 
  attr_number_--;
  //cerr << "FaceAttr: " << attr_number_ << " deleted\n";
}



void FaceAttr::free()
{
  //cout << "* FaceAttr::free id: " << id_ << "\n";
  if (vertex_list_) 
  {
    delete vertex_list_; 
  }
  if (normal_list_) 
  {
    delete normal_list_; 
  }

  delete [] light_;
}


void FaceAttr::copy(const FaceAttr& faceattr)
{
cout << "*:FaceAttr::copy\n";
      
  material_ = faceattr.material_; 
  backface_culling_ = faceattr.backface_culling_;

  //cout << "min bound box\n";

  min_ = faceattr.min_;
  max_ = faceattr.max_;

  cout << "light list:\n";

  if (faceattr.light_)
  {
    light_ = new AnyLight*[faceattr.light_num_];
    for (int i=0; i < faceattr.light_num_; i++)
    {
      light_[i] = faceattr.light_[i];
      //light_[i]->print();
    }
    light_num_ = faceattr.light_num_;    
  }


  cout << "vertex list:\n";
  if (faceattr.vertex_list_->num > 0)
  {  
    vertex_list_ = new QvMFVec3f;
    vertex_list_->allocValues(faceattr.vertex_list_->num);
    vertex_num_ = faceattr.vertex_num_;

    for (int i=0; i < 3 * vertex_num_; i++)
      vertex_list_->values[i] = faceattr.vertex_list_->values[i];
   
  }


  //cout << "normal list:\n";
  if (faceattr.normal_list_->num > 0)
  {  
    normal_list_ = new QvMFVec3f;
    normal_list_->allocValues(faceattr.normal_list_->num);
    normal_num_ = faceattr.normal_num_;

    for (int i=0; i < 3 * normal_num_; i++)
      normal_list_->values[i] = faceattr.normal_list_->values[i];
   }

}


FaceAttr& FaceAttr::operator=(const FaceAttr& faceattr)
{
cout << "*faceAttr::operator=\n"; 
  if (this != &faceattr)
  {
    free();
    copy(faceattr);
  }
 
  return *this;
}



void FaceAttr::print()
{
  cerr << "Faceattr ID:" << id_ << "\n";
  cerr << "Addr of faceattr: " << this << "\n";
  cerr << "Addr of light list: " << light_ << "\n"; 
  cerr << "Active lights:" << light_num_ << "\n";
  for (int i=0; i < light_num_; i++)
  {
    if (light_[i]) light_[i]->print();

    else cerr << "no light in light list at position: " << i << "\n";
  }
  if (material_) cerr << "Material set\n";
  cerr << "Culling:" << backface_culling_ << "\n";
  //cerr << "Vertices:" << vertex_list_->num << "\n";
  printVertices();
  printNormals();
  cerr << "\n";
}


void FaceAttr::printVertices()
{
  cerr << "Vertices:" << vertex_num_ << "\n";
  for (int i=0; i < vertex_num_; i++)
    Clipping::print3D(*( (point3D*) (vertex_list_->values+(3*i))));
  cerr << "\n";
}

void FaceAttr::printNormals()
{
  cerr << "Normals:" << normal_num_ << "\n";
  for (int i=0; i < normal_num_; i++)
    Clipping::print3D(*( (point3D*) (normal_list_->values+(3*i))));
  cerr << "\n";
}



void FaceAttr::setAttributes(
      materialsGE3D* material,    // one material structure
      unsigned light_num,
      AnyLight** light,  // list of some light structures
      int backface_culling)
{
  material_ = material;
  light_num_ = light_num;

  // copy the light-pointers into a new array

  light_ =  new AnyLight*[light_num];
  for (unsigned i = 0; i < light_num; i++) 
  {
    light_[i]=light[i];
    //light_[i]->print();
  }

  backface_culling_ = backface_culling;
}





int FaceAttr::extendVertices(
              point3D& new_vertex,
              vector3D& new_normal)
{
//cout << "*:FaceAttr::extendVertices\n";
  
  vertex_num_ += 1; 
  if (vertex_list_->num < vertex_num_)  
     vertex_list_->allocValues(vertex_num_ + alloc_ahead_);

//  vertex_list_->set1Value(vertex_num_ - 1, new_vertex.x, new_vertex.y, new_vertex.z);
  ((point3D*) vertex_list_->values) [vertex_num_ - 1] = new_vertex;  // mpi

  if ((new_normal.x != 0) || (new_normal.y !=0) || (new_normal.z !=0))
  {
    normal_num_ += 1; 
    if (normal_list_->num < normal_num_)
       normal_list_->allocValues(normal_num_ + alloc_ahead_);      
//    normal_list_->set1Value(normal_num_ - 1, new_normal.x, new_normal.y, new_normal.z);
    ((vector3D*) normal_list_->values) [normal_num_ - 1] = new_normal;  // mpi
  }

//cout << "*:FaceAttr::extendVertices finished with index:" <<
//        vertex_num_ - 1  << " for faceattr id: " << id_ << "\n";
//printVertices();

  return (vertex_num_-1);

}


void FaceAttr::setVertexListSize(int num) 
{ 
  vertex_list_->allocValues(num);
  if (vertex_num_ > num) vertex_num_ = num;
}


void FaceAttr::setNormalListSize(int num) 
{ 
  normal_list_->allocValues(num); 
  if (normal_num_ > num) normal_num_ = num;
}



point3D* FaceAttr::vertexList() const
{
  //cout << "*:FaceAttr::vertexList()\n"; 
  if ((vertex_list_) && (vertex_list_->num > 0))
    return (point3D*) vertex_list_->values;
  else return 0;
}



vector3D* FaceAttr::normalList() const 
{
  if ((normal_list_) && (normal_list_->num > 0))
    return (vector3D*) normal_list_->values; 
  else return 0;
} 

 

void FaceAttr::drawBoundBox()
{
  ge3dWireCube (&min_, &max_);
}   
