/*-----------------------------------------------------------------------------
 *  FILE: PreferencesDialog.cc
 *
 *      Copyright(c) 2006 Gareth Foster.
 *
 *      Dialog derivation that implements Wallpaper Tray preferences.
 *
 *-----------------------------------------------------------------------------
 */

#include "PreferencesDialog.hh"
#include "config.h"
#include <glib/gi18n.h>
#include <stdexcept>
#include <sstream>
#include <algorithm>

/** @defgroup WPTray WPTray
 * WPTray contains classes used in the Wallpaper Tray program.
 */

//------------------------------------------------------------------------
/**
<PreferencesDialog::PreferencesDialog>
@brief Construct a new preferences dialog.

@param _pWindow Required to allow this class to be instanciated from glade xml.
@param _pGlade Required to allow this class to be instanciated from glade xml.
@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
PreferencesDialog::PreferencesDialog
(
	BaseObjectType *							_pWindow,
	const Glib::RefPtr<Gnome::Glade::Xml> &		_pGlade
) :
	Gtk::Dialog(_pWindow),
	m_pGlade(_pGlade),
	m_pButtonClose(NULL),
	m_pButtonApply(NULL),
	m_pButtonAddDir(NULL),
	m_pButtonRemDir(NULL),
	m_pEntryTime(NULL),
	m_pEntryMode(NULL),
	m_pEntrySelectionMode(NULL),
	m_pCheckButtonTimeout(NULL),
	m_pCheckButtonLogonNew(NULL),
	m_pCheckButtonImageCheck(NULL),
	m_pCheckButtonFollowLinks(NULL),
	m_pCheckButtonNotifications(NULL)
{
	// set up the gconf client
	m_pGconfClient = Gnome::Conf::Client::get_default_client();

	// get the glade instantiated widgets and connect signals
	m_pGlade->get_widget("button_close", m_pButtonClose);
	if(m_pButtonClose)
	{
		m_pButtonClose->signal_clicked().connect(sigc::mem_fun(*this, &PreferencesDialog::hide));
	}// end if

	m_pGlade->get_widget("button_apply", m_pButtonApply);
	if(m_pButtonApply)
	{
		m_pButtonApply->signal_clicked().connect(sigc::mem_fun(*this, &PreferencesDialog::OnButtonApplyClick));
	}// end if

	m_pGlade->get_widget("button_add_dir", m_pButtonAddDir);
	if(m_pButtonAddDir)
	{
		m_pButtonAddDir->signal_clicked().connect(sigc::mem_fun(*this, &PreferencesDialog::OnButtonAddDirClick));
	}// end if

	m_pGlade->get_widget("button_rem_dir", m_pButtonRemDir);
	if(m_pButtonRemDir)
	{
		m_pButtonRemDir->signal_clicked().connect(sigc::mem_fun(*this, &PreferencesDialog::OnButtonRemDirClick));
	}// end if

	m_pGlade->get_widget("entry_time", m_pEntryTime);
	m_pGlade->get_widget("combo-entry_mode", m_pEntryMode);
	m_pGlade->get_widget("combo-entry_selection_mode", m_pEntrySelectionMode);
	m_pGlade->get_widget("checkbutton_timeout", m_pCheckButtonTimeout);
	m_pGlade->get_widget("checkbutton_logon_new", m_pCheckButtonLogonNew);
	m_pGlade->get_widget("checkbutton_image_check", m_pCheckButtonImageCheck);
	m_pGlade->get_widget("checkbutton_follow_links", m_pCheckButtonFollowLinks);
	m_pGlade->get_widget("checkbutton_show_notification", m_pCheckButtonNotifications);
	
	// grab the tree view
	m_pGlade->get_widget("treeview_directories", m_pTreeView);

	if(m_pTreeView == NULL)
	{
		throw std::runtime_error("PreferencesDialog::PreferencesDialog() : failed to get widget from glade");
	}// end if

	// set up the list store and model etc
	m_pListstore = Gtk::ListStore::create(m_Model);

	m_pTreeView->set_model(m_pListstore);

	m_pTreeView->append_column(_("Directories"), m_Model.m_DirectoryName);

	// update the members/cache
	PreferencesDialog::GetSettingsInGconf();

	// populate the list
	for(std::list<Glib::ustring>::iterator it = m_lsWpDir.begin(); it != m_lsWpDir.end(); ++ it)
	{
		Gtk::TreeModel::iterator it_new = m_pListstore->append();

		(*it_new)[m_Model.m_DirectoryName] = (*it);
	}// end for

	// set the other widgets up from the gconf values
	std::ostringstream ss;
	ss.imbue(std::locale("")); // use the user's locale for this stream
	ss << m_nTimeout;
	m_pEntryTime->set_text(Glib::locale_to_utf8(ss.str()));

	m_pEntryMode->set_text(m_szPictureOptions);
	//TODO: m_pEntrySelectionMode->set_text();
	m_pCheckButtonTimeout->set_active(m_bTimeout);
	m_pCheckButtonLogonNew->set_active(m_bWpLogon);
	m_pCheckButtonImageCheck->set_active(m_bImgCheck);
	m_pCheckButtonFollowLinks->set_active(m_bFollowLinks);
	m_pCheckButtonNotifications->set_active(m_bNotifications);

	// give the window an icon
	PreferencesDialog::set_icon_from_file(GNOMEICONDIR"/wp_tray-applet.png");

	// make the PreferencesDialog window start hidden
	PreferencesDialog::hide();
}// end PreferencesDialog::PreferencesDialog


//------------------------------------------------------------------------
/**
<PreferencesDialog::~PreferencesDialog>
@brief Destroy a preferences dialog.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
PreferencesDialog::~PreferencesDialog
(
)
{
}// end PreferencesDialog::~PreferencesDialog

//------------------------------------------------------------------------
/**
<PreferencesDialog::OnButtonCloseClick>
@brief Handle the users request to close the dialog.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void PreferencesDialog::OnButtonCloseClick
(
)
{
	PreferencesDialog::hide();
}// end PreferencesDialog::OnButtonCloseClick

//------------------------------------------------------------------------
/**
<PreferencesDialog::OnButtonApplyClick>
@brief Handle the users request to close the dialog and apply their new settings.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void PreferencesDialog::OnButtonApplyClick
(
)
{
	PreferencesDialog::SetSettingsInGconf();

	PreferencesDialog::hide();
}// end PreferencesDialog::OnButtonCloseClick

//------------------------------------------------------------------------
/**
<PreferencesDialog::OnButtonAddDirClick>
@brief Handle the users request to add a new wallpaper directory.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void PreferencesDialog::OnButtonAddDirClick
(
)
{
	Gtk::FileChooserDialog gtkFileChooserDialog(_("Wallpaper Tray - Please choose a folder"), Gtk::FILE_CHOOSER_ACTION_SELECT_FOLDER);

	gtkFileChooserDialog.set_transient_for(*this);
		
	// add response buttons the the dialog
	gtkFileChooserDialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	gtkFileChooserDialog.add_button(_("Select"), Gtk::RESPONSE_OK);

	gtkFileChooserDialog.set_current_folder(Glib::get_home_dir());
	
	int n_Result = gtkFileChooserDialog.run();

	// handle the response
	switch(n_Result)
	{
		case(Gtk::RESPONSE_OK):
		{
			m_lsWpDir.push_back(gtkFileChooserDialog.get_filename());

			Gtk::TreeModel::iterator it_new = m_pListstore->append();

			(*it_new)[m_Model.m_DirectoryName] = gtkFileChooserDialog.get_filename();
		}// end case
		break;

		case(Gtk::RESPONSE_CANCEL):
		{
			// who cares?
		}// end case
		break;

		default:
		{
			// gah, i dunno, throw something?
		}// end default
		break;
	}// end switch
}// end PreferencesDialog::OnButtonAddDirClick

//------------------------------------------------------------------------
/**
<PreferencesDialog::OnButtonRemDirClick>
@brief Handle the users request to remove a wallpaper directory.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void PreferencesDialog::OnButtonRemDirClick
(
)
{
	Glib::RefPtr<Gtk::TreeSelection> pTreeSelection = m_pTreeView->get_selection();

	Gtk::TreeModel::iterator it = pTreeSelection->get_selected();
	
	// if anything is selected
	if(it)
	{
		Glib::ustring szValue;

		it->get_value(0, szValue);

		m_lsWpDir.erase(std::find(m_lsWpDir.begin(), m_lsWpDir.end(), szValue));

		m_pListstore->erase(it);
	}// end if
}// end PreferencesDialog::OnButtonRemDirClick

//------------------------------------------------------------------------
/**
<PreferencesDialog::GetSettingsInGconf>
@brief Retrieve settings from gconf into local variables.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void PreferencesDialog::GetSettingsInGconf
(
)
{
	m_lsWpDir				= m_pGconfClient->get_string_list("/apps/wp_tray/dir_list");
	m_nTimeout				= m_pGconfClient->get_int("/apps/wp_tray/n_timeout");
	m_bTimeout				= m_pGconfClient->get_bool("/apps/wp_tray/b_timeout");
	m_bWpLogon				= m_pGconfClient->get_bool("/apps/wp_tray/b_wp_logon");
	m_bImgCheck				= m_pGconfClient->get_bool("/apps/wp_tray/b_img_check");
	m_bFollowLinks			= m_pGconfClient->get_bool("/apps/wp_tray/b_follow_links");
	m_bNotifications			= m_pGconfClient->get_bool("/apps/wp_tray/b_notifications");
	m_szPictureOptions 		= m_pGconfClient->get_string("/desktop/gnome/background/picture_options");
}// end PreferencesDialog::GetSettingsInGconf


//------------------------------------------------------------------------
/**
<PreferencesDialog::SetSettingsInGconf>
@brief Store settings into gconf from local variables.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void PreferencesDialog::SetSettingsInGconf
(
)
{
	// mwahaha scum bag C coders, we have overloading!
	m_pGconfClient->set_string_list("/apps/wp_tray/dir_list", m_lsWpDir);

	// set the other widgets up from the gconf values
	std::stringstream ss;
	ss.imbue(std::locale("")); // use the user's locale for this stream
	ss << m_pEntryTime->get_text().c_str();
	ss >> m_nTimeout;

	m_pGconfClient->set("/apps/wp_tray/n_timeout", m_nTimeout);
	m_pGconfClient->set("/apps/wp_tray/b_timeout", m_pCheckButtonTimeout->get_active());
	m_pGconfClient->set("/apps/wp_tray/b_wp_logon", m_pCheckButtonLogonNew->get_active());
	m_pGconfClient->set("/apps/wp_tray/b_img_check", m_pCheckButtonImageCheck->get_active());
	m_pGconfClient->set("/apps/wp_tray/b_follow_links", m_pCheckButtonFollowLinks->get_active());
	m_pGconfClient->set("/apps/wp_tray/b_notifications", m_pCheckButtonNotifications->get_active());
	m_pGconfClient->set("/desktop/gnome/background/picture_options", m_pEntryMode->get_text());
}// end PreferencesDialog::SetSettingsToGconf
