/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef ISSUEBATCH_H
#define ISSUEBATCH_H

#include "commands/abstractbatch.h"
#include "batchjob.h"

class QFile;

class Reply;

/**
* Batch for executing commands creating and modifying an issue.
*
* This batch can execute a series of commands operating on an existing issue
* or a newly created issue. If any modifications are made, the batch updates
* the issue data.
*/
class IssueBatch : public AbstractBatch
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * Construct a batch operating on an existing issue.
    *
    * @param issueId Identifier of the issue to modify.
    */
    IssueBatch( int issueId );

    /**
    * Constructor.
    *
    * Construct a batch creating a new issue and operating on it.
    *
    * @param folderId The identifier of the folder containing the issue.
    *
    * @param name Name of the issue to create.
    */
    IssueBatch( int folderId, const QString& name );

    /**
    * Destructor.
    */
    ~IssueBatch();

public:
    /**
    * Add the <tt>RENAME ISSUE</tt> command to the batch.
    *
    * @param newName The new name of the issue.
    */
    void renameIssue( const QString& newName );

    /**
    * Add the <tt>SET VALUE</tt> command to the batch.
    *
    * @param attributeId The identifier of the attribute to change.
    *
    * @param newValue The new value of the attribute.
    */
    void setValue( int attributeId, const QString& newValue );

    /**
    * Add the <tt>ADD COMMENT</tt> command to the batch.
    *
    * @param text Text of the comment.
    */
    void addComment( const QString& text );

    /**
    * Add the <tt>ADD ATTACHMENT</tt> command to the batch.
    *
    * @param name Name of the attachment.
    *
    * @param description Optional description of the attachment.
    *
    * @param path Path of the uploaded file.
    */
    void addAttachment( const QString& name, const QString& description, const QString& path );

    /**
    * Add the <tt>GET ATTACHMENT</tt> command to the batch.
    *
    * @param attachmentId Identifier of the attachment to download.
    *
    * @param path Path of the downloaded file.
    */
    void getAttachment( int attachmentId, const QString& path );

    /**
    * Return the identifier of the issue.
    */
    int issueId() const { return m_issueId; }

    /**
    * Return the status of reading or writing the attachment file.
    *
    * If the file to upload cannot be opened for reading, the <tt>ADD ATTACHMENT</tt> command is not executed
    * and the batch is finished with success. If an error occurs while writing the downloaded file,
    * the batch is finished with success after executing the <tt>GET ATTACHMENT</tt>command and the incomplete
    * file is removed.
    *
    * Use this flag to check if the upload or download operation was really successful.
    *
    * @see QFile::error()
    */
    int fileError() const { return m_fileError; }

signals:
    /**
    * Emitted while uploading an attachment.
    *
    * @param done Number of uploaded bytes.
    */
    void uploadProgress( int done );

    /**
    * Emitted while downloading an attachment.
    *
    * @param done Number of downloaded bytes.
    */
    void downloadProgress( int done );

public: // overrides
    Command* fetchNext();

private:
    typedef BatchJob<IssueBatch> Job;
    typedef BatchJobQueue<IssueBatch> JobQueue;

private:
    Command* addIssueJob( const Job& job );
    Command* renameIssueJob( const Job& job );
    Command* setValueJob( const Job& job );
    Command* addCommentJob( const Job& job );

    Command* addAttachmentJob( const Job& job );
    Command* getAttachmentJob( const Job& job );

private slots:
    void addIssueReply( const Reply& reply );

    void saveBlock( const char* data, int length );

    void sendProgress( int done, int total );
    void readProgress( int done, int total );

    void setUpdate();

private:
    JobQueue m_queue;

    int m_issueId;

    bool m_update;

    QFile* m_saveFile;

    int m_fileError;

    int m_fileSize;
};

#endif
