/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "searcheditbox.h"

#include <QToolButton>
#include <QStyle>
#include <QStyleOption>
#include <QPainter>

#include "iconloader.h"

SearchEditBox::SearchEditBox( QWidget* parent ) : QLineEdit( parent ),
    m_showPrompt( false )
{
    m_optionsButton = new QToolButton( this );
    m_optionsButton->setIcon( IconLoader::icon( "find-options" ) );
    m_optionsButton->setToolTip( tr( "Search Options" ) );
    m_optionsButton->setStyleSheet( "QToolButton { border: none; padding: 0px; } "
        "QToolButton::menu-indicator { right: -2px; bottom: -2px; }" );
    m_optionsButton->setCursor( Qt::ArrowCursor );
    m_optionsButton->setFocusPolicy( Qt::NoFocus );
    m_optionsButton->setPopupMode( QToolButton::InstantPopup );

    m_clearButton = new QToolButton( this );
    m_clearButton->setIcon( IconLoader::icon( "find-clear" ) );
    m_clearButton->setToolTip( tr( "Clear" ) );
    m_clearButton->setStyleSheet( "QToolButton { border: none; padding: 0px; }" );
    m_clearButton->setCursor( Qt::ArrowCursor );
    m_clearButton->setFocusPolicy( Qt::NoFocus );
    m_clearButton->hide();
  
    connect( m_clearButton, SIGNAL( clicked() ), this, SLOT( clear() ) );
    connect( this, SIGNAL( textChanged( const QString& ) ), this, SLOT( updateCloseButton( const QString& ) ) );

    QSize buttonSize = m_optionsButton->sizeHint();
    int frameWidth = style()->pixelMetric( QStyle::PM_DefaultFrameWidth );

    int padding = buttonSize.width() + frameWidth + 1;
    setStyleSheet( QString( "QLineEdit { padding-left: %1px; padding-right: %1px; } " ).arg( padding ) );

    setMinimumHeight( qMax( minimumSizeHint().height(), buttonSize.height() + 2 * frameWidth ) );

    showPrompt( true );
}

SearchEditBox::~SearchEditBox()
{
}

void SearchEditBox::resizeEvent( QResizeEvent* e )
{
    QLineEdit::resizeEvent( e );

    QSize buttonSize = m_optionsButton->sizeHint();
    int frameWidth = style()->pixelMetric(QStyle::PM_DefaultFrameWidth);

    int y = ( rect().bottom() + 1 - buttonSize.height() ) / 2;

    m_optionsButton->move( frameWidth, y );
    m_clearButton->move( rect().right() - frameWidth - buttonSize.width(), y );
}

void SearchEditBox::updateCloseButton( const QString& text )
{
    m_clearButton->setVisible( !text.isEmpty() );
}

void SearchEditBox::focusInEvent( QFocusEvent* e )
{
    QLineEdit::focusInEvent( e );

    if ( m_showPrompt )
        showPrompt( false );
}

void SearchEditBox::focusOutEvent( QFocusEvent* e )
{
    QLineEdit::focusOutEvent( e );

    if ( text().isEmpty() )
        showPrompt( true );
}

void SearchEditBox::setPromptText( const QString& text )
{
    m_promptText = text;

    if ( m_showPrompt )
        update();
}

void SearchEditBox::showPrompt( bool on )
{
    m_showPrompt = on;

    update();
}

void SearchEditBox::paintEvent( QPaintEvent* e )
{
    QLineEdit::paintEvent( e );

    if ( m_showPrompt && !m_promptText.isEmpty() ) {
        QPainter painter( this );

        painter.setPen( palette().color( QPalette::Disabled, QPalette::Text ) );

        QStyleOption option;
        option.initFrom( this );
        QRect rect = style()->subElementRect( QStyle::SE_LineEditContents, &option, this );

        painter.setClipRect( rect );

        int x = rect.x() + 2;
        int y = rect.y() + ( rect.height() - fontMetrics().height() + 1 ) / 2;

        painter.drawText( x, y + fontMetrics().ascent(), m_promptText );
    }
}

void SearchEditBox::setOptionsMenu( QMenu* menu )
{
    m_optionsButton->setMenu( menu );
}
