/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "popeditbox.h"

#include <QTextBrowser>
#include <QPushButton>
#include <QLayout>
#include <QMenu>
#include <QKeyEvent>

#include "iconloader.h"

PopEditBox::PopEditBox( QWidget* parent ) : QFrame( parent ),
    m_menu( false )
{
    initialize();
}

PopEditBox::PopEditBox( const QString& text, QWidget* parent ) : QFrame( parent ),
    m_menu( false )
{
    initialize();
    m_browser->setText( text );
}

PopEditBox::~PopEditBox()
{
}

void PopEditBox::initialize()
{
    setFrameStyle( StyledPanel | Sunken );

    QHBoxLayout* layout = new QHBoxLayout( this );
    layout->setMargin( 0 );
    layout->setSpacing( 0 );

    m_browser = new QTextBrowser( this );
    m_browser->document()->setDocumentMargin( 2 );
    m_browser->setSizePolicy( QSizePolicy::Ignored, QSizePolicy::Fixed );
    m_browser->setFrameStyle( QFrame::NoFrame );
    m_browser->viewport()->setBackgroundRole( QPalette::Window );
    m_browser->setHorizontalScrollBarPolicy( Qt::ScrollBarAlwaysOff );
    m_browser->setVerticalScrollBarPolicy( Qt::ScrollBarAlwaysOff );
    m_browser->setWordWrapMode( QTextOption::NoWrap );
    m_browser->setContextMenuPolicy( Qt::CustomContextMenu );
    layout->addWidget( m_browser, 1 );

    layout->addSpacing( 2 );

    QPalette palette = m_browser->palette();
    palette.setBrush( QPalette::Inactive, QPalette::Highlight, palette.brush( QPalette::Active, QPalette::Highlight ) );
    palette.setBrush( QPalette::Inactive, QPalette::HighlightedText, palette.brush( QPalette::Active, QPalette::HighlightedText ) );
    m_browser->setPalette( palette );

    connect( m_browser, SIGNAL( anchorClicked( const QUrl& ) ),
        this, SLOT( browserAnchorClicked( const QUrl& ) ) );

    connect( m_browser, SIGNAL( customContextMenuRequested( const QPoint& ) ),
        this, SLOT( browserContextMenu( const QPoint& ) ) );

    m_browser->installEventFilter( this );

    m_button = new QPushButton( tr( "..." ), this );
    layout->addWidget( m_button, 0 );

    connect( m_button, SIGNAL( clicked() ), this, SIGNAL( buttonClicked() ) );

    int height = m_browser->fontMetrics().lineSpacing() + 4;
#if !defined( Q_WS_WIN )
    height += 1;
#endif
    m_browser->setFixedHeight( height );
    m_button->setFixedSize( height * 4 / 3, height );
}

void PopEditBox::setText( const QString& text )
{
    m_browser->setText( text );
}

QString PopEditBox::text() const
{
    return m_browser->toPlainText();
}

void PopEditBox::clear()
{
    m_browser->clear();
}

void PopEditBox::browserAnchorClicked( const QUrl& url )
{
    m_browser->setSource( QUrl() );

    emit linkClicked( url.toString() );
}

void PopEditBox::browserContextMenu( const QPoint& pos )
{
    m_menu = true;

    QString link = m_browser->anchorAt( pos );
    if ( !link.isEmpty() ) {
        emit linkContextMenu( link, m_browser->mapToGlobal( pos ) );
    } else {
        QMenu menu;
        QAction* action;

        action = new QAction( IconLoader::icon( "edit-copy" ), tr( "&Copy" ), &menu );
        action->setShortcut( QKeySequence::Copy );
        action->setEnabled( m_browser->textCursor().hasSelection() );
        connect( action, SIGNAL( triggered() ), m_browser, SLOT( copy() ) );
        menu.addAction( action );

        menu.addSeparator();

        action = new QAction( tr( "&Select &All" ), &menu );
        action->setShortcut( QKeySequence::SelectAll );
        action->setEnabled( !m_browser->document()->isEmpty() );
        connect( action, SIGNAL( triggered() ), m_browser, SLOT( selectAll() ) );
        menu.addAction( action );

        menu.exec( m_browser->mapToGlobal( pos ) );
    }

    m_menu = false;

    if ( !m_browser->hasFocus() ) {
        QTextCursor cursor = m_browser->textCursor();
        cursor.clearSelection();
        m_browser->setTextCursor( cursor );
    }
}

bool PopEditBox::eventFilter( QObject* watched, QEvent* e )
{
    if ( watched == m_browser ) {
        if ( e->type() == QEvent::FocusOut && !m_menu ) {
            QTextCursor cursor = m_browser->textCursor();
            cursor.clearSelection();
            m_browser->setTextCursor( cursor );
        } else if ( e->type() == QEvent::ShortcutOverride ) {
            QKeyEvent* ke = (QKeyEvent*)e;
            if ( ke == QKeySequence::Copy ) {
                m_browser->copy();
                ke->accept();
                return true;
            } else if ( ke == QKeySequence::SelectAll ) {
                m_browser->selectAll();
                ke->accept();
                return true;
            }
        }
    }
    return QFrame::eventFilter( watched, e );
}
