/*
 * Copyright (c) 2003-2007 by FlashCode <flashcode@flashtux.org>
 * See README for License detail, AUTHORS for developers list.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* log.c: WeeChat log file */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdarg.h>

#ifdef HAVE_FLOCK
#include <sys/file.h>
#endif

#include <sys/types.h>
#include <time.h>

#include "weechat.h"
#include "log.h"
#include "util.h"


char *weechat_log_filename = NULL; /* log name (~/.weechat/weechat.log)     */
FILE *weechat_log_file = NULL;     /* WeeChat log file                      */


/*
 * weechat_log_open: initialize log file
 */

int
weechat_log_open (char *filename, char *mode)
{
    int filename_length;

    /* exit if log already opened */
    if (weechat_log_file)
        return 0;

    if (filename)
        weechat_log_filename = strdup (filename);
    else
    {
        filename_length = strlen (weechat_home) + 64;
        weechat_log_filename =
            (char *) malloc (filename_length * sizeof (char));
        snprintf (weechat_log_filename, filename_length,
                  "%s/%s", weechat_home, WEECHAT_LOG_NAME);
    }
    
    weechat_log_file = fopen (weechat_log_filename, mode);
    if (!weechat_log_file)
    {
        free (weechat_log_filename);
        weechat_log_filename = NULL;
        return 0;
    }

#ifdef HAVE_FLOCK
    if ((flock (fileno (weechat_log_file), LOCK_EX | LOCK_NB) != 0))
    {
        fclose (weechat_log_file);
        weechat_log_file = NULL;
        free (weechat_log_filename);
        weechat_log_filename = NULL;
        return 0;
    }
#endif

    return 1;
}

/*
 * weechat_log_init: initialize log file
 */

void
weechat_log_init ()
{
    if (!weechat_log_open (NULL, "w"))
    {
        weechat_iconv_fprintf (stderr,
                               _("%s unable to create/append to log file\n"
                                 "If another WeeChat process is using this file, try to run WeeChat\n"
                                 "with another home using \"--dir\" command line option.\n"),
                               WEECHAT_ERROR);
        exit (1);
    }
}

/*
 * weechat_log_printf: write a message in WeeChat log (<weechat_home>/weechat.log)
 */

void
weechat_log_printf (char *message, ...)
{
    static char buffer[4096];
    char *ptr_buffer;
    va_list argptr;
    static time_t seconds;
    struct tm *date_tmp;
    
    if (!weechat_log_file)
        return;
    
    va_start (argptr, message);
    vsnprintf (buffer, sizeof (buffer) - 1, message, argptr);
    va_end (argptr);
    
    /* keep only valid chars */
    ptr_buffer = buffer;
    while (ptr_buffer[0])
    {
        if ((ptr_buffer[0] != '\n')
            && (ptr_buffer[0] != '\r')
            && ((unsigned char)(ptr_buffer[0]) < 32))
            ptr_buffer[0] = '.';
        ptr_buffer++;
    }
    
    seconds = time (NULL);
    date_tmp = localtime (&seconds);
    if (date_tmp)
        weechat_iconv_fprintf (weechat_log_file, "[%04d-%02d-%02d %02d:%02d:%02d] %s",
                               date_tmp->tm_year + 1900, date_tmp->tm_mon + 1, date_tmp->tm_mday,
                               date_tmp->tm_hour, date_tmp->tm_min, date_tmp->tm_sec,
                               buffer);
    else
        weechat_iconv_fprintf (weechat_log_file, "%s", buffer);
    
    fflush (weechat_log_file);
}

/*
 * weechat_log_close: close log file
 */

void
weechat_log_close ()
{
    /* close log file */
    if (weechat_log_file)
    {
#ifdef HAVE_FLOCK
        flock (fileno (weechat_log_file), LOCK_UN);
#endif
        fclose (weechat_log_file);
        weechat_log_file = NULL;
    }

    /* free filename */
    if (weechat_log_filename)
    {
        free (weechat_log_filename);
        weechat_log_filename = NULL;
    }
}

/*
 * weechat_log_crash_rename: rename log file when crashing
 */

int
weechat_log_crash_rename ()
{
    char *old_name, *new_name;
    int length;
    time_t time_now;
    struct tm *local_time;

    if (!weechat_log_filename)
        return 0;
    
    old_name = strdup (weechat_log_filename);
    if (!old_name)
        return 0;

    weechat_log_close ();
    
    length = strlen (weechat_home) + 128;
    new_name = (char *) malloc (length);
    if (new_name)
    {
        time_now = time (NULL);
        local_time = localtime (&time_now);
        snprintf (new_name, length,
                  "%s/weechat_crash_%04d%02d%02d_%d.log",
                  weechat_home,
                  local_time->tm_year + 1900,
                  local_time->tm_mon + 1,
                  local_time->tm_mday,
                  getpid());
        if (rename (old_name, new_name) == 0)
        {
            weechat_iconv_fprintf (stderr, "*** Full crash dump was saved to %s file.\n",
                                   new_name);
            weechat_log_open (new_name, "a");
            free (old_name);
            free (new_name);
            return 1;
        }
        free (new_name);
    }

    free (old_name);
    weechat_log_open (NULL, "a");
    return 0;
}
