/* -*-c-*- */
/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

%{

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>

#include "wfipaddr.h"
#include "wfnetwork.h"

#include "filter.h"
#include "defs.h"

/*
#ifdef DEBUG
#define YYDEBUG 1
#endif
*/

#if 1
#define ASSERTION(cond)  do { \
  if (!(cond)) { \
    yyerror(__FILE__ ":%i: assertion `" #cond "' failed", __LINE__); \
    YYABORT; \
  } \
 } while(0)
#else
#define ASSERTION(cond) do {} while(0)
#endif

extern int yylex();
extern char* yytext; /* really needed? */

int yyerror(const char* str, ...);
void ydebug(const char* str, ...);
void yline(const char* str, ...);

static exprnode* tree; /* result of the parsing */

%}

%union {
  int intval;
  char* str;
  wf_ipaddr* ipaddr;
  wf_network* network;
  wf_macaddr* macaddr;
  exprnode* node;
  enum filter_op_unary unary;
}

%token ALL NULLNETWORK NOT
%token PROTO

%token <ipaddr> IPADDR
%token <network> IPNET
%token <macaddr> MACADDR
%token <intval> INTNUM
%token <str> NAME STRING

%token <intval> IP UDP TCP ICMP

/* ICMP types */
%token <intval> ECHO_REPLY
%token <intval> DESTINATION_UNREACHABLE
%token <intval> NETWORK_UNREACHABLE
%token <intval> HOST_UNREACHABLE
%token <intval> PROTOCOL_UNREACHABLE
%token <intval> PORT_UNREACHABLE
%token <intval> FRAGMENTATION_NEEDED
%token <intval> SOURCE_ROUTE_FAILED
%token <intval> NETWORK_UNKNOWN
%token <intval> HOST_UNKNOWN
%token <intval> NETWORK_PROHIBITED
%token <intval> HOST_PROHIBITED
%token <intval> TOS_NETWORK_UNREACHABLE
%token <intval> TOS_HOST_UNREACHABLE
%token <intval> COMMUNICATION_PROHIBITED
%token <intval> HOST_PRECEDENCE_VIOLATION
%token <intval> PRECEDENCE_CUTOFF
%token <intval> SOURCE_QUENCH
%token <intval> REDIRECT
%token <intval> NETWORK_REDIRECT
%token <intval> HOST_REDIRECT
%token <intval> TOS_NETWORK_REDIRECT
%token <intval> TOS_HOST_REDIRECT
%token <intval> ECHO_REQUEST
%token <intval> ROUTER_ADVERTISEMENT
%token <intval> ROUTER_SOLICITATION
%token <intval> TIME_EXCEEDED
%token <intval> TTL_ZERO_DURING_TRANSIT
%token <intval> TTL_ZERO_DURING_REASSEMBLY
%token <intval> PARAMETER_PROBLEM
%token <intval> IP_HEADER_BAD
%token <intval> REQUIRED_OPTION_MISSING
%token <intval> TIMESTAMP_REQUEST
%token <intval> TIMESTAMP_REPLY
%token <intval> INFO_REQUEST
%token <intval> INFO_REPLY
%token <intval> ADDRESS_MASK_REQUEST
%token <intval> ADDRESS_MASK_REPLY

%token <intval> TCP_FLAGS
%token LE_OP GE_OP EQ_OP NE_OP REG_OP LAND_OP LOR_OP '&' '|' '^'
%token <str> IDENTIFIER STRING_LITERAL STRING_REGEXP DATE_STRING
%token <intval> '-' '!' '+'

%type <node> conditional_expr and_expr exclusive_or_expr inclusive_or_expr logical_or_expr logical_and_expr equality_expr relational_expr additive_expr unary_expr primary_expr
%type <unary> unary_operator

%%

input:
	  conditional_expr
		{
		  tree = $1;
		}
	;

conditional_expr
	: logical_or_expr
	;

logical_or_expr
	: logical_and_expr
	| logical_or_expr LOR_OP logical_and_expr
		{
		  $$ = new exprnode_binary(FOP_LOR, $1, $3);
		}
	;

logical_and_expr
	: inclusive_or_expr
	| logical_and_expr LAND_OP inclusive_or_expr
		{
		  $$ = new exprnode_binary(FOP_LAND, $1, $3);
		}
	;

equality_expr
	: relational_expr
	| equality_expr EQ_OP relational_expr
		{
		  $$ = new exprnode_binary(FOP_EQ, $1, $3);
		}
	| equality_expr NE_OP relational_expr
		{
		  $$ = new exprnode_binary(FOP_NE, $1, $3);
		}
	| equality_expr REG_OP relational_expr /* regexp */
		{
		  $$ = new exprnode_binary(FOP_REG, $1, $3);
		}
	;

and_expr
	: equality_expr
	| and_expr '&' equality_expr
		{
		  $$ = new exprnode_binary(FOP_AND, $1, $3);
		}
	;

exclusive_or_expr
	: and_expr
	| exclusive_or_expr '^' and_expr
		{
		  $$ = new exprnode_binary(FOP_XOR, $1, $3);
		}
	;

inclusive_or_expr
	: exclusive_or_expr
	| inclusive_or_expr '|' exclusive_or_expr
		{
		  $$ = new exprnode_binary(FOP_OR, $1, $3);
		}
	;

relational_expr
	: additive_expr
	| relational_expr '<' unary_expr
		{
		  $$ = new exprnode_binary(FOP_LT, $1, $3);
		}
	| relational_expr '>' unary_expr
		{
		  $$ = new exprnode_binary(FOP_GT, $1, $3);
		}
	| relational_expr LE_OP unary_expr
		{
		  $$ = new exprnode_binary(FOP_LE, $1, $3);
		}
	| relational_expr GE_OP unary_expr
		{
		  $$ = new exprnode_binary(FOP_GE, $1, $3);
		}
	;

additive_expr
	: unary_expr
	| additive_expr '+' unary_expr
		{
		  $$ = new exprnode_binary(FOP_PLUS, $1, $3);
		}
	| additive_expr '-' unary_expr
		{
		  $$ = new exprnode_binary(FOP_MINUS, $1, $3);
		}
	;

unary_expr
	: primary_expr
	| unary_operator primary_expr
		{
		  $$ = new exprnode_unary($1, $2);
		}
	;

unary_operator
	: '-'
		{ $$ = FOP_NEG; }
	| '!'
		{ $$ = FOP_NOT; }
	| '+'
		{ $$ = FOP_POS; }
	;

primary_expr
	: IDENTIFIER
		{
		  $$ = new exprnode_var_id_wflogs($1);
		  if (((exprnode_var_id*)$$)->num() == VARS_UNKNOWN) {
		    yyerror(_("unknown var `%s'"), $1);
		    delete $$;
		    YYABORT;
		  }
		}
	| INTNUM
		{
		  $$ = new exprnode_var_int($1);
		}
	| TCP_FLAGS
		{
		  $$ = new exprnode_var_int($1);
		}
	| STRING_LITERAL
		{
		  $$ = new exprnode_var_str($1);
		}
	| STRING_REGEXP
		{
		  int flags = 0;
		  $1++; /* get rid of first slash */
		  char* slashptr = strrchr($1, '/'); /* cannot be NULL */
		  for (*slashptr++ = '\0'; *slashptr != '\0'; ++slashptr) {
		    switch (*slashptr) {
		    case 'i':
		      flags |= REG_ICASE; break;
		    }
		  }
		  $$ = new exprnode_var_regexp();
		  // test return RV@@8
		  if (((exprnode_var_regexp*)$$)->set($1, flags) == false) {
		    delete $$;
		    YYABORT;
		  }
		}
	| IPADDR
		{
		  /* $1 = wf_ipaddr* */
		  $$ = new exprnode_var_ipnet(*$1);
		}
	| IPNET
		{
		  /* $1 = wf_network* */
		  $$ = new exprnode_var_ipnet(*$1);
		}
	| MACADDR
		{
		  /* $1 = wf_macaddr* */
		  $$ = new exprnode_var_macaddr(*$1);
		}
	| DATE_STRING
		{
		  $$ = new exprnode_var_date();
		  // test return RV@@8
		  if (((exprnode_var_date*)$$)->set($1) == false) {
		    delete $$;
		    YYABORT;
		  }
		}
	| '(' conditional_expr ')'
		{
		  $$ = $2; // RV@@9 check
		}
	;

%%

void
yline(const char* str, ...) {
  va_list args;

  va_start(args, str);
  vfprintf(stderr, str, args);
  va_end(args);
  cerr << endl;
}

void
ydebug(const char* str, ...) {
  va_list args;

  return;
  va_start(args, str);
  vfprintf(stderr, str, args);
  va_end(args);
}

int
yyerror(const char* str, ...) {
  va_list args;

  fprintf(stderr, _("error before `%s': "), yytext);
  va_start(args, str);
  vfprintf(stderr, str, args);
  va_end(args);
  cerr << endl;
  return 0;
}

extern int yy_scan_string(const char*);

exprnode*
exprnode_parse(const string& expr) {
#if YYDEBUG != 0
  yydebug = 1;
#endif
  
  yy_scan_string(expr.c_str());
  if (yyparse())
    return NULL;

  if (tree->check() == false) {
    cerr << _("check failed") << endl;
    delete tree;
    return NULL;
  }
  return tree;
}
