/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sstream>
#include <stdio.h>

#include "wflogs_obfuscator.h"
#include "defs.h"


wflogs_obfuscator::wflogs_obfuscator() :
  _hostnamecache(),
  _ipcache(),
  _maccache()
{
  reset();
  srandom(time(NULL));
  _time_offset = random() % (3600*24*365); /* one year ahead in maximum */
}

wflogs_obfuscator::~wflogs_obfuscator() {}


void
wflogs_obfuscator::reset() {
  hostname = false;
  date = false;
  ipaddr = false;
  macaddr = false;
}

void
wflogs_obfuscator::setall() {
  hostname = true;
  date = true;
  ipaddr = true;
  macaddr = true;
}

bool
wflogs_obfuscator::set(const string& criterias) {
  if (criterias.empty() || criterias == "all") { /* default values, or all */
    setall();
    return true;
  }

  reset();
  unsigned int pos, oldpos = 0;
  do {
    pos = criterias.find(',', oldpos);
    string crit = criterias.substr(oldpos, pos - oldpos);
    if (crit == "hostname")
      hostname = true;
    else if (crit == "date")
      date = true;
    else if (crit == "ipaddr")
      ipaddr = true;
    else if (crit == "macaddr")
      macaddr = true;
    else {
      fprintf(stderr, _("Error: unknown obfuscator criteria `%s'\n"),
	      crit.c_str());
      return false;
    }
    
    oldpos = pos + 1;
  } while (pos != string::npos);
  return true;
}

string
wflogs_obfuscator::changehostname(const string& hostname) {
  map<string, u_int16_t>::const_iterator iter, last = _hostnamecache.end();
  iter = _hostnamecache.find(hostname);
  if (iter != last) { /* found hostname in cache */
    ostringstream os;
    os << "host" << (*iter).second;
    return os.str();
  }

  u_int16_t n = _hostnamecache.size() + 1;
  _hostnamecache[hostname] = n;
  ostringstream os;
  os << "host" << n;
  return os.str();
}

unsigned int
wflogs_obfuscator::randomip(const wf_ipaddr& ipaddr) {
  map<u_int32_t, u_int32_t>::const_iterator iter, last = _ipcache.end();
  iter = _ipcache.find(ipaddr.get());
  if (iter != last) /* found IP address in cache */
    return (*iter).second;

  unsigned int addr;
  do { /* find a non already existing address */
    addr = random();
    ((char*)&addr)[0] = 0;
    for (iter = _ipcache.begin(); iter != last; ++iter) {
      if ((*iter).second == addr)
	break;
    }
    /* Here, if iter == last, random addr does not already exist. That's ok. */
  } while (iter != last);

  _ipcache[ipaddr.get()] = addr;
  return addr;
}

wf_macaddr
wflogs_obfuscator::changemacaddr(const wf_macaddr& macaddr) {
  if (macaddr.isdefined() == false)
    return macaddr;

  map<wf_macaddr, u_int32_t>::const_iterator iter, last = _maccache.end();
  wf_macaddr retmacaddr;
  u_int32_t num, n;
  iter = _maccache.find(macaddr);
  if (iter == last) { /* MAC address not found in cache */
    /* insert new MAC address in cache: find an associated number */
    num = _maccache.size() + 1; /* start with 0:0:0:0:0:1 */
    _maccache[macaddr] = num;
  }
  else
    num = (*iter).second;

  /* construct a MAC address based on integer num */
  u_int8_t addr[ETH_ALEN];
  n = htonl(num);
  memset(&addr, 0, 2); /* first 2 bytes to 0 (out of 6) */
  memcpy(&addr[2], &n, 4); /* next 4 bytes set to int */
  retmacaddr.set(addr);
  
  return retmacaddr;
}

time_t
wflogs_obfuscator::changedate(time_t time) {
  return time + _time_offset;
}

void
wflogs_obfuscator::operator() (wf_logentry* entry) {
  if (date) {
    entry->start_time = changedate(entry->start_time);
    if (entry->end_time != 0)
      entry->end_time = changedate(entry->end_time);
  }
  if (hostname)
    entry->hostname = changehostname(entry->hostname);
  if (ipaddr) {
    if (entry->sipaddr.isroutable())
      entry->sipaddr.set(randomip(entry->sipaddr));
    if (entry->dipaddr.isroutable())
      entry->dipaddr.set(randomip(entry->dipaddr));
  }
  if (macaddr) {
    entry->smacaddr = changemacaddr(entry->smacaddr);
    entry->dmacaddr = changemacaddr(entry->dmacaddr);
  }
}

ostream&
wflogs_obfuscator::print(ostream& os) const {
  if (hostname)
    os << "hostname" << endl;
  if (date)
    os << "date" << endl;
  if (ipaddr)
    os << "ipaddr" << endl;
  if (macaddr)
    os << "macaddr" << endl;
  return os;
}
