/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <netdb.h>
#include <sys/socket.h>  /* for AF_INET */
#include <netinet/in.h>  /* for htons */

#include "dns.h"
#include "defs.h"


wf_dns::wf_dns() :
  cache(),
  verbose(0)
{}

wf_dns::wf_dns(unsigned char verb) :
  cache(),
  verbose(verb)
{}


/* If no name is found, we create the entry, but name will be "". */
wf_dns_entry*
wf_dns::resolv(const wf_ipaddr& ipaddr) {
  wf_dns_entry* dns_entry;

  dns_entry = cache_get(ipaddr);
  if (dns_entry != NULL) { /* found in cache */
    if (verbose > 1)
      fprintf(stderr, _("Found DNS info for %s in cache\n"),
	      ipaddr.tostr().c_str());
    return dns_entry;
  }

  /* Not found in cache. */
  if (verbose > 1)
    fprintf(stderr, _("Looking up DNS info for %s\n"),
	    ipaddr.tostr().c_str());

  struct hostent* reverse;
  struct in_addr ip;
  ip.s_addr = ipaddr.get();
  reverse = gethostbyaddr((char*)&ip.s_addr, sizeof(struct in_addr), AF_INET);

  dns_entry = new wf_dns_entry();
  if (dns_entry == NULL)
    return NULL;

  dns_entry->ipaddr = ipaddr;
  if (reverse != NULL && reverse->h_name != NULL)
    dns_entry->name = reverse->h_name;
  /* If no name is found, we create the entry, but name will be empty (""). */

  cache_add(dns_entry);  /* Add entry to cache. */
  return dns_entry;
}

void
wf_dns::cache_add(const wf_dns_entry* dns_entry) {
  if (dns_entry != NULL)
    cache.push_back((wf_dns_entry*)dns_entry);
}

wf_dns_entry*
wf_dns::cache_get(const wf_ipaddr& ipaddr) const {
  if (ipaddr.isdefined() == false)
    return NULL;

  list<wf_dns_entry*>::const_iterator first = cache.begin(),
    last = cache.end();
  for (; first != last; ++first) {
    if ((*first)->ipaddr == ipaddr)
      return *first;
  }
  return NULL;
}

wf_dns_entry*
wf_dns::cache_get(const string& dnsname) const {
  if (dnsname.empty())
    return NULL;

  list<wf_dns_entry*>::const_iterator first = cache.begin(),
    last = cache.end();
  for (; first != last; ++first) {
    if ((*first)->name == dnsname)
      return *first;
  }
  return NULL;
}

ostream&
wf_dns::cache_debugprint(ostream& os) const {
  list<wf_dns_entry*>::const_iterator first = cache.begin(),
    last = cache.end();
  for (; first != last; ++first)
    (*first)->print(os) << endl;
  return os;
}

ostream&
wf_dns_entry::print(ostream& os) const {
  return os << ipaddr << ' ' << name;
}

ostream&
operator<<(ostream& os, const wf_dns_entry& dns_entry) {
  return dns_entry.print(os);
}
