/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <time.h>
#include <pwd.h>    /* for getpwuid() */
#include <unistd.h> /* for getuid() */

#include "xml.h"
#include "list1.h"
#include "defs.h"

#include "conf.h"
#include "resolv.h" // for service RV@@8
#include "preresolv.h"

#include "wfipaddr.h"
#include "wfprotocol.h"

#define TIMESIZE 40


wf_outmodule_xml::wf_outmodule_xml() :
  wf_outmodule(),
  dns(NULL),
  whois(NULL)
{
  conf = new wf_outmodule_xml_conf();
}

wf_outmodule_xml::~wf_outmodule_xml() {
  if (dns != NULL)
    delete dns;
  if (whois != NULL)
    delete whois;
  delete conf;
}


static string
indent(unsigned int level) {
  string spaces;
  for (; level > 0; level--)
    spaces += "  ";
  return spaces;
}

static void
output_header(ostream& os) {
  char nowstr[TIMESIZE];
  time_t now;
  now = time(NULL);
  strftime(nowstr, TIMESIZE, "%a %b %d %H:%M:%S %Z %Y", localtime(&now));

  os << "<?xml version=\"1.0\" encoding=\"ISO-8859-1\" ?>\n" <<
    "<wflogs xmlns=\"http://wallfire.org/wflogs.dtd\">\n";

  os << "<!-- Generated by wflogs on " << nowstr << " -->\n";
  struct passwd* gen_user;
  gen_user = getpwuid(getuid());
  if (gen_user != NULL) {
    os << indent(1) << "<gen_by>";
    if (gen_user->pw_gecos[0] != '\0')
      os << gen_user->pw_gecos;
    else
      os << gen_user->pw_name;
    os << "</gen_by>\n";
  }
  os << indent(1) << "<gen_date>" << nowstr << "</gen_date>\n" << endl;
}


static void
output_footer(ostream& os) {
  os << "</wflogs>" << endl;
}

bool
wf_outmodule_xml::print(const wf_logentry* entry, ostream& os) {
  os << indent(1) << "<line>\n";
  
  if (conf->config["format"].tobool())
    os << indent(2) << "<format>" << entry->format << "</format>\n";

  if (entry->count > 1)
    os << indent(2) << "<count>" << entry->count << "</count>\n";

  os << indent(2) << "<time_start>" << entry->start_time << "</time_start>\n";

  if (entry->end_time != 0)
    os << indent(2) << "<time_end>" << entry->end_time << "</time_end>\n";

  os << indent(2) << "<host>" << entry->hostname << "</host>\n";

  if (conf->config["chain"].tobool())
    os << indent(2) << "<chainlabel>" << entry->chainlabel <<
      "</chainlabel>\n";

  if (conf->config["branch"].tobool())
    os << indent(2) << "<branchname>" << entry->branchname <<
      "</branchname>\n";

  if (entry->input_iface.empty() == false)
    os << indent(2) << "<in_iface>" << entry->input_iface << "</in_iface>\n";

  if (entry->smacaddr.isdefined()) {
    os << indent(2) << "<src_mac>" << entry->smacaddr << "</src_mac>\n";
    if (conf->config["mac_vendor"].tobool())
      os << indent(2) << "<src_mac_vendor>" << entry->smacaddr.vendor() <<
	"</src_mac_vendor>\n";
  }
  
  if (entry->output_iface.empty() == false)
    os << indent(2) << "<out_iface>" << entry->output_iface <<
      "</out_iface>\n";

  if (entry->dmacaddr.isdefined()) {
    os << indent(2) << "<dst_mac>" << entry->dmacaddr << "</dst_mac>\n";
    if (conf->config["mac_vendor"].tobool())
      os << indent(2) << "<dst_mac_vendor>" << entry->dmacaddr.vendor() <<
	"</dst_mac_vendor>\n";
  }

  string proto;
  {
    wf_protocol protocol = wf_protocol(entry->protocol);
    proto = protocol.tostr();
  }

  os << indent(2) << "<data_len>" << entry->datalen << "</data_len>\n";

  os << indent(2) << "<from>" << entry->sipaddr << "</from>\n";
  if (conf->config["resolve"].tobool()) {
    const wf_dns_entry* dnsentry = dns->resolv(entry->sipaddr);
    if (dnsentry != NULL)
      os << indent(2) << "<from_resolv>" << dnsentry->name <<
	"</from_resolv>\n";
  }

  if (conf->config["whois_lookup"].tobool()) {
    os << indent(2) << "<from_whois>";
    wf_whois_entry* we = whois->whois(entry->sipaddr);
    if (we != NULL)
      os << *we;
    os << "</from_whois>\n";
  }

  os << indent(2) << "<to>" << entry->dipaddr << "</to>\n";
  if (conf->config["resolve"].tobool()) {
    wf_dns_entry* dnsentry = dns->resolv(entry->dipaddr);
    if (dnsentry != NULL)
      os << indent(2) << "<to_resolv>" << dnsentry->name << "</to_resolv>\n";
  }
  
  if (conf->config["whois_lookup"].tobool()) {
    os << indent(2) << "<to_whois>";
    wf_whois_entry* we = whois->whois(entry->dipaddr);
    if (we != NULL)
      os << *we;
    os << "</to_whois>\n";
  }

  os << indent(2) << "<proto name=\"" << proto << "\">\n";
  if (entry->protocol == IPPROTO_UDP || entry->protocol == IPPROTO_TCP) {
    os << indent(3) << "<sport>" << entry->sport << "</sport>\n";
    if (conf->config["sresolve"].tobool())
      os << indent(3) << "<sport_resolv>" <<
	resolv_service(entry->sport, proto) << "</sport_resolv>\n";

    os << indent(3) << "<dport>" << entry->dport << "</dport>\n";
    if (conf->config["sresolve"].tobool())
      os << indent(3) << "<dport_resolv>" <<
	resolv_service(entry->dport, proto) << "</dport_resolv>\n";
    
    if (entry->protocol == IPPROTO_TCP && entry->tcpflags != 0)
      os << indent(3) << "<tcpflags>" << entry->tcpflags_tostr_mini() <<
	"</tcpflags>\n";
  }
  else if (entry->protocol == IPPROTO_ICMP) {
    os << indent(3) << "<type>" << entry->sport << "</type>\n";
    if (wf_protocol_icmp_type_hascode(entry->sport))
      os << indent(3) << "<code>" << entry->dport << "</code>\n";
    if (conf->config["icmp_typename"].tobool())
      os << indent(3) << "<typename>" <<
	wf_protocol_icmp_type_tostr(entry->sport, entry->dport) <<
	"</typename>\n";
  }
  os << indent(2) << "</proto>\n";

  os << indent(1) << "</line>\n";

  return true;
}

bool
wf_outmodule_xml::print(const wf_logentries& logentries, ostream& os) {
  unsigned char verbose = conf->config["verbose"].toint();
  int lines = conf->config["lines"].toint();

  wf_logentries* newlogentries;
  if (lines) { /* copy n first logentries */
    newlogentries = new wf_logentries();
    if (newlogentries == NULL)
      return false;
    list1_firstncopy(newlogentries->elems, logentries.elems, lines);
  }
  else /* do not copy logentries */
    newlogentries = (wf_logentries*)&logentries;

  if (conf->config["whois_lookup"].tobool()) {
    if (whois == NULL) {
      whois = new wf_whois(verbose);
      if (whois == NULL) {
	if (lines)
	  delete newlogentries;
	return false;
      }
    }
    if (whois->isconnected() == false && whois->connect() == false) {
      if (lines)
	delete newlogentries;
      return false;
    }
  }

  if (conf->config["resolve"].tobool()) {
    if (dns == NULL) {
      dns = new wf_dns(verbose);
      if (dns == NULL) {
	if (lines)
	  delete newlogentries;
	return false;
      }
    }
    preresolv(*newlogentries, dns, true, true, verbose);
  }

  output_header(os);
  if (wf_outmodule::print(*newlogentries, os) == false) {
    if (lines)
      delete newlogentries;
    return false;
  }
  output_footer(os);

  if (conf->config["whois_lookup"].tobool())
    whois->close();
  if (lines)
    delete newlogentries;

  return true;
}

extern "C" wf_outmodule*
wf_outmodule_xml_init() {
  return new wf_outmodule_xml();
}
